<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from VIAVI
// Copyright JDSU 2018. All rights reserved.
// *********************************************************
namespace app\serviceshelper\otdr;

use app\util\SMTLogger;
use app\util\SMTIOException;
use app\http\SMTSessionDirectoryUtil;
use app\serviceshelper\activity\SMTActivity;

/**
 * Utility class to handle OTDR trace files on OTU file system
 *
 * @author Sylvain Desplat
 */
class SMTBaseOtdrTraceFileUtil
{
	/**
	 * OTDR file bellcore extension
	 * @var string
	 */
	const OTDR_FILE_SOR_EXTENSION= ".sor";
	
	/**
	 * OTDR file bellcore extension
	 * @var string
	 */
	const OTDR_FILE_CSOR_EXTENSION= ".csor";
	
	/**
	 * OTDR file bellcore extension
	 * @var string
	 */
	const OTDR_FILE_MSOR_EXTENSION= ".msor";
	const OTU_MEASURE_ON_DEMAND_DIR = "/otu/result/measure_on_demand/";
	const OTU_MEASURE_ON_DEMAND_FILE = "measure.sor";
	
	//monitoring test reference traces
	const OTU_MEASURE_REF_DET_DIR = "/otu/reference/detection/";
	const OTU_MEASURE_REF_LOC_DIR = "/otu/reference/localisation/";
	const OTU_MEASURE_REF_SLIDING_DIR = "/otu/reference/highsens/";
	
	//monitoring traces (on alarm trace)
	const OTU_RESULT_DET_DIR = "/otu/result/detection/";
	const OTU_RESULT_LOC_DIR = "/otu/result/localisation/";
	
	//last monitoring trace
	const OTU_TEST_RESULT_DET_DIR = "/otu/test/detection/";
	const OTU_TEST_RESULT_LOC_DIR = "/otu/test/localisation/";
	const OTDR_TRACE_FILE_DET = "det";
	const OTDR_TRACE_FILE_LOC = "loc";
	const OTDR_TRACE_FILE_SHORTACQ = "loc_opti";
	const OTU_TEST_RESULT_DET_FILE_NAME = "det_%s_port%s.sor";
	const OTU_TEST_RESULT_LOC_FILE_NAME = "loc_%s_port%s.sor";
	
	/**
	 * OTDR reference trace name for localisation and detection
	 * smtref + {portNumber}  //one test per link in SmartOTU mode
	 * @var string
	 */
	const REF_TRACE_NAME = "smtref_port%s.sor";
	
	/**
	 * OTDR file suffix to discriminate duplicated OTDR trace files (case of update of OTDR trace for monitoring )
	 * @var string
	 */
	//const OTDR_FILE_DUPLICATE_SUFFIX = "XX";
	const OTDR_FILE_MEMORIZED_SUFFIX = "_copy";
		
	/**
	 * Return file extension .sor, .csor, .msor 
	 * 
	 * @param string $acquisitionTrace file path with extension
	 * @param string $Error
	 * @return string
	 */
	public static function getSmartAcqFileExtension($acquisitionTrace, &$Error=FALSE)
	{
		$ext = self::OTDR_FILE_SOR_EXTENSION;
		$Error = FALSE;

		
		//search if csor, msor or sor exists
		if ( strrpos($acquisitionTrace,".") !== FALSE)
		{
			$acquisitionTraceWithoutExt = substr($acquisitionTrace, 0,strrpos($acquisitionTrace,"."));
		}
		else 
		{
			$acquisitionTraceWithoutExt = $acquisitionTrace;
		}
		
		if ( file_exists($acquisitionTraceWithoutExt.self::OTDR_FILE_CSOR_EXTENSION) )
		{
			$ext = self::OTDR_FILE_CSOR_EXTENSION;
		}
		else if ( file_exists($acquisitionTraceWithoutExt.self::OTDR_FILE_MSOR_EXTENSION) )
		{
			$ext = self::OTDR_FILE_MSOR_EXTENSION;
		}
		else if ( file_exists($acquisitionTraceWithoutExt.self::OTDR_FILE_SOR_EXTENSION) )
		{
			$ext = self::OTDR_FILE_SOR_EXTENSION;
		}
		else
		{
			SMTLogger::getInstance()->trace("file not found: ".$acquisitionTrace, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
			$Error = TRUE;
		}
		return $ext;
	}
	
	public static function isSmartAcqFileExtension($acquisitionTraceExtension)
	{
		return (strcasecmp($acquisitionTraceExtension, self::OTDR_FILE_CSOR_EXTENSION) == 0) || (strcasecmp($acquisitionTraceExtension, self::OTDR_FILE_MSOR_EXTENSION) == 0);
	}
	
	public static function removeExtension($acquisitionTrace)
	{
		return ( strrpos($acquisitionTrace,".") !== FALSE ) ? substr($acquisitionTrace, 0,strrpos($acquisitionTrace,".")) : $acquisitionTrace;
	}
	
	public static function replaceExtension($acquisitionTrace, $newExtension)
	{
		$acquisitionTraceWithoutExt = $acquisitionTrace;
		if ( strrpos($acquisitionTrace,".") !== FALSE)
		{
			$acquisitionTraceWithoutExt = substr($acquisitionTrace, 0,strrpos($acquisitionTrace,"."));
		}
		return $acquisitionTraceWithoutExt.$newExtension;
	}
	
	/**
	 * Returns last monitoring OTDR trace full file name from OTU directory
	 *
	 * @param int $testId
	 * @return string
	 */
	public static function buildLastMonitoringDetectionTraceNameOnOtu( $testId, $portNumber)
	{
		return self::OTU_TEST_RESULT_DET_DIR.sprintf( self::OTU_TEST_RESULT_DET_FILE_NAME, $testId, $portNumber);
	}
	
	/**
	 * Returns last monitoring localisation OTDR trace full file name from OTU directory
	 *
	 * @param int $testId
	 * @return string
	 */
	public static function buildLastMonitoringLocalisationTraceNameOnOtu( $testId, $portNumber)
	{
		return self::OTU_TEST_RESULT_LOC_DIR.sprintf( self::OTU_TEST_RESULT_LOC_FILE_NAME, $testId, $portNumber);
	}
	
	/**
	 * Returns last monitoring OTDR trace full file name from SmartOTU directory
	 *
	 * @param int $testId
	 * @return string
	 */
	public static function buildLastMonitoringDetectionTraceNameOnSmartOtu( $testId, $portNumber)
	{
		return SMTSessionDirectoryUtil::createSessionDirectoryIfNeeded().sprintf( self::OTU_TEST_RESULT_DET_FILE_NAME, $testId, $portNumber);
	}
	
	/**
	 * Returns the last monitoring OTDR trace full file name from SmartOtu
	 * or throw exception if not found
	 * @param string $testId
	 * @return string
	 * @throws SMTIOException if file was not found
	 */
	public static function getLastMonitoringTraceNameFromSmartOtu( $testId, $portNumber)
	{
		$traceFilePath = self::buildLastMonitoringDetectionTraceNameOnSmartOtu($testId, $portNumber);
		
		$acquisitionTraceWithoutExt = $traceFilePath;
		if ( strrpos($traceFilePath,".") !== FALSE)
		{
			$acquisitionTraceWithoutExt = substr($traceFilePath, 0,strrpos($traceFilePath,"."));
		}
		
		foreach ( glob( $acquisitionTraceWithoutExt."*" ) as $file )
		{
			$traceFilePath= $file;
			break;
		}
		
		if ( !file_exists( $traceFilePath ) )
		{
			throw new SMTIOException( SMTIOException::FILE_NOT_FOUND, $traceFilePath );
		}
		return $traceFilePath;
	}
	
	/**
	 * Copy last monitoring OTDR acquisition trace (SOR,MSOR,CSOR) from OTU directories to SmartOtu directories.
	 *
	 * @param int $testId The test identifier.
	 *
	 * @return string the last monitoring OTDR acquisition trace file name stored in SmartOTU "lastacquisition" directory.
	 */
	private static function copyOtuLastMonitoringDectectionAcquisitionTraceInSmartOtu( $testId, $portNumber, &$isLoc, $smartAcq = TRUE )
	{
		$fullTraceNameOnOTU = self::buildLastMonitoringDetectionTraceNameOnOtu($testId, $portNumber);
		
		SMTLogger::getInstance()->trace( sprintf( "Start retrieving last monitoring OTDR acquisition trace %s ", $fullTraceNameOnOTU ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
		
		//backup OTDR smartacq file to the smartOTU directory to be used for monitoring test addition
		$newOtdrAcquisitionTraceName = SMTSessionDirectoryUtil::backupLastMonitoringTraceToSessionDirectory( $testId, $portNumber, $isLoc, $smartAcq);
		
		SMTLogger::getInstance()->trace( sprintf( "End copying last monitoring OTDR acquisition trace %s to %s ", $fullTraceNameOnOTU, $newOtdrAcquisitionTraceName ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
		
		return $newOtdrAcquisitionTraceName;
	}
	
	/**
	 * Copy the last acquisition trace (CSOR,MSOR,SOR) of the given monitoring test to SmartOtu directory.
	 * If the activity of the given test is in ANALYSIS status, sleep a while
	 *
	 * @param $testId
	 * @param boolean smartAcq FALSE to limit the search to sor file
	 *
	 * @return string the last monitoring OTDR acquisition trace file name stored in SmartOTU "lastacquisition" directory.
	 */
	public static function retrieveLastMonitoringAcquisitionTrace( $testId, $portNumber, &$isLoc, $smartAcq = TRUE )
	{
		$newOtdrAcquisitionTraceName = "";
		$fullTraceNameOnOTU = self::buildLastMonitoringDetectionTraceNameOnOtu($testId, $portNumber);
		
		try
		{
			//check that the given test is not in analysing phase
			//(the copy of the last acquisition otdr trace is done before the COMPLETE activity event is sent,
			// don't retrieve that trace during the ANALIZING activity).
			$count = 0;
			$maxLoop = 5;
			while ( SMTActivity::isActivityStatusAnalysis( $testId ) && $count++ < $maxLoop )
			{
				sleep( 1 );
			}
			$newOtdrAcquisitionTraceName = self::copyOtuLastMonitoringDectectionAcquisitionTraceInSmartOtu($testId, $portNumber, $isLoc, $smartAcq);
		}
		catch( \Exception $e )
		{
			SMTLogger::getInstance()->traceException($e);
			throw new SMTIOException( SMTIOException::FILE_NOT_FOUND, $fullTraceNameOnOTU );
		}
		return $newOtdrAcquisitionTraceName;
	}    
}
?>