<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\otau;

use app\parser\SMTOtuSocket;
use app\services\setup\SMTSwitchDto;

/**
 * 
 * @author Sylvain Desplat
 *
 */
abstract class SMTMacroSwitch
{
    const GLOB = 'GLOB';
    const INTERNAL = 'INT';
    const INTERNAL_POSITION = 0;    
    
    /**
     * ROTAU
     * @var string
     */
    const ROTAU = 'ROTAU';
    /**
     * OSU/OSX
     * @var string
     */
    const OSU = 'OSU';    

    /**
     * OSU/OSX distributor
     * @var string
     */
    const OSU_DISTRIBUTOR = 'OSU31';

    /**
     * OSU/OSX distributor
     * @var interger
     */
    const OSU_DISTRIBUTOR_INDEX = 31;
    
    
    /**
     *
     * @var app\parser\SMTOtuSocket
     */
    protected $socket = NULL;
    
    /**
     *
     * @var array app\services\setup\SMTSwitchDto
     */
    protected $switchesDto = array();
    
    /**
     *
     * @var string
     */
    protected $globalSwitchEncodedString = NULL;
    
    /**
     * Array of SMTSwitchDecoder storing and decoding switch string.
     *
     * @var array
     */
    protected $switchDecode = array();
    
    /**
     * Internal Switch serial number
     *
     * @var string
    */
    protected $serialNumber;
    
    /**
     * 
     * @param app\parser\SMTOtuSocket $socket
     */
    function __construct( SMTOtuSocket $socket )
    {
    	$this->socket = $socket;
    }
        
    /**
     * Retrieve GLOB switch and set switch decoder
     */
    abstract protected function retrieveGlobal();

    /**
     * Retrieve INT switch and set switch decoder
     */    
    abstract protected function retrieveInternal();
    
    /**
     * Return the number of output ports
     * 
     * @return number
     */
    public function getOutputPorts()
    {
        $ports = 1;
        $decoder = $this->getSwitchDecoder( self::GLOB );
        if ( $decoder != NULL )
        {
        	$ports = intval( $decoder->getOut() );
        }
        return $ports;
    }
    
    /**
     * Return the global Switch Encoded String
     *
     * @return String
     */
    public function getGlobalSwitchEncodedString()
    {
        return $this->globalSwitchEncodedString;
    }
    
    /**
     *
     * @var array app\services\setup\SMTSwitchDto
     */
    public function getSwitchesDto()
    {
        return $this->switchesDto;
    }
    
    /**
     *
     * @param $detected boolean Whether the switch is detected or retrieved from configuration 
     * 
     */
    protected function buildSwitchDto( $detected )
    {       
        $osuNumber = 0;
        
    	//if global switch decoder exists (case of detection), set it in the dto.
    	$decoder = $this->getSwitchDecoder( self::GLOB );
    	if ( $decoder != NULL )
    	{
    		$this->globalSwitchEncodedString = $decoder->getSwitchEncodingString();
    		$osuNumber = $decoder->getOSUCount();
    		
    		// distributor OSX musn't be added twice
    		if ( $osuNumber>= self::OSU_DISTRIBUTOR_INDEX)
    		{
    			$osuNumber= self::OSU_DISTRIBUTOR_INDEX -1 ;
    		}
    	}
    	
    	$decoder = $this->getSwitchDecoder( self::INTERNAL );
    	if ( $decoder != NULL )
    	{
    	    $switchDto = new SMTSwitchDto();
    	    $switchDto->setSerialNumber( $this->serialNumber );
    	    $switchDto->setDetected( $detected );
    	    $switchDto->setIndex( 0 );
    	    
    		$switchDto->setInput( $decoder->getIn() );
    		$switchDto->setOutput( $decoder->getOut() );
    		$switchDto->setGlobalSwitchEncodedString($this->globalSwitchEncodedString);
    		
    		array_push($this->switchesDto, $switchDto);
    	}    	
    	
    	for ( $osuCount = 1; $osuCount <= $osuNumber; $osuCount++)
    	{
    	    $decoder = $this->getSwitchDecoder( self::OSU.$osuCount );
    	    
    	    if ( $decoder != NULL )
    	    {
    	        $switchDto = new SMTSwitchDto();
    	        $switchDto->setDetected( $detected );
    	        $switchDto->setIndex( $osuCount );
    	        
    	    	$switchDto->setInput( $decoder->getIn() );
    	    	$switchDto->setOutput( $decoder->getOut() );
    	    	
    	    	$switchDto->setGlobalSwitchEncodedString($this->globalSwitchEncodedString);
    	    	
    	    	array_push($this->switchesDto, $switchDto);
    	    }
    	}
    	
    	$decoder = $this->getSwitchDecoder( self::OSU_DISTRIBUTOR );
        if ( $decoder != NULL )
	    {
	        $switchDto = new SMTSwitchDto();
	        $switchDto->setDetected( $detected );
	        $switchDto->setIndex( self::OSU_DISTRIBUTOR_INDEX );
	        
	    	$switchDto->setInput( $decoder->getIn() );
	    	$switchDto->setOutput( $decoder->getOut() );
	    	
	    	$switchDto->setGlobalSwitchEncodedString($this->globalSwitchEncodedString);
	    	
	    	array_push($this->switchesDto, $switchDto);
	    }
    }     
    
    /**
     *
     * @param string $switchKey self::GLOB, self::INTERNAL
     * @param string $switchStringCode
     */
    protected function setSwitchDecoder( $switchKey, $switchStringCode )
    {
    	$this->switchDecode[ $switchKey ] = new SMTSwitchDecoder( trim( $switchStringCode ) );
    }
    
    /**
     *
     * @param string $switchKey self::GLOB, self::INTERNAL
     *
     * @return SMTSwitchDecoder
     */
    protected function getSwitchDecoder( $switchKey )
    {
    	return array_key_exists( $switchKey, $this->switchDecode )? $this->switchDecode[ $switchKey ] : NULL;
    }
    
    /**
     * Whether OTU has OSX
     * 
     * @return boolean
     */
    function hasOsx()
    {
    	return ($this->getSwitchDecoder( self::GLOB) != NULL)? ($this->getSwitchDecoder( self::GLOB)->getOSUCount() > 0) || ($this->getSwitchDecoder( self::GLOB)->getOSUDistributor() != '') : FALSE; 
    }
}

?>