<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\setup;

use app\settings\SMTSmartOtuSettings;

use app\serviceshelper\media\SMTMediaException;
use app\serviceshelper\media\SMTSms;

use app\view\SMTTestRestApiPage;
use app\services\SMTIService;
use app\services\SMTService;
use app\parser\SMTOtuApi;


/**
 * Service to setup the OTU Sms features and display its configuration.
 * 
 * @author Sylvain Desplat
 */
class SMTSmsService extends SMTService implements SMTIService
{      
    /**
     * Create empty Dto Sms to be sent to the client
     *
     * @url GET /new
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function newSms( $request_data )
    {
    	try
    	{
    		$smsDto = new SMTSmsDto();
    		$smsDto->setValid( FALSE);
    			
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $smsDto->getJsonData();
    }    
    
    /**
     * Full json request data are by default stored in an array in parameter Default::$fullRequestDataName='request_Data'
     * Update full sms config
     *
     * @url POST
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function updateSms( $request_data )
    {
        $sms = new SMTSms();
        $sms->setContext( $this->getContext() );
        
    	try
    	{
    		$configSmsDto = SMTConfSmsDto::getInstance( $request_data );

    		if ( $configSmsDto->isActive() )
    		{
    		    $sms->checkSmsAvailability();
    		}
    		
    		$sms->setSmsMedia( $configSmsDto->isSmsActive() );
    		$sms->updateSms($configSmsDto);
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	//fetch applied config
    	return $this->fetchSms();
    }        
    
    /**
     *
     * Fetch Sms configurations. If no modem is set, returns a SMTConfSmsDto with empty SMTSMSDto array and modem availability property set to FALSE.
     *
     * @url GET
     * 
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchSms()    
    {        
        $smsConfDto = new SMTConfSmsDto();
        $sms = new SMTSms();
        $sms->setContext( $this->getContext() );
    	try
    	{
    	    //modem type GSM...
    	    $smsConfDto->setModemType( $sms->retrieveModemType() );
    		$modemOption = $sms->isModemOption();
    		$validModemType = $sms->isValidModemType();
    		//save sms option availability in cache
    		SMTSmartOtuSettings::setSmsOptionValid( $modemOption && $validModemType );
    	}
    	catch ( \Exception $e )
    	{
    		//catch exception to retrieve settings and status even if an error occurs
    		$this->getContext()->getLogger()->traceException( $e );
    		//set by default sms option as not available in case of error (OTU application not available...)
    		$modemOption = FALSE;
    		$validModemType = FALSE;
    	}
        $smsConfDto->setSmsOption( $modemOption && $validModemType );
        
        try
        {            
            //by default modem available SMTConfSmsDto property is set to FALSE if modem option is not set
            //don't retrieve sms properties if modem option is not set
            if ( $modemOption && $validModemType )
            {            
                $smsConfDto->setModemAvailable( $sms->isModemReady() );
                
                for( $index= SMTSms::MIN_SMS_INDEX; $index <= SMTSms::MAX_SMS_INDEX; $index++)
                {
                    $smsDto = $sms->retrieveSms( $index );
                    if ( $smsDto != NULL )
                    { 
                        $smsConfDto->addSms( $smsDto );
                    }
                }
                $smsConfDto->setActive( $sms->isSmsMediaOn() );
            }                             	        
    	}
    	catch ( \Exception $e )
    	{
    	    $this->handleException($e);
    	}
    
    	return $smsConfDto->getJsonData();
    }
    
    /**
     * Perform a Test of Sms media: send an Sms with the given phone number
     *
     * @url POST /{smsPhoneNumber}/test
     * 
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function testSms( $smsPhoneNumber )
    {
        try 
        {            
            $sms = new SMTSms();
            $sms->setContext( $this->getContext() );
            $sms->checkSmsAvailability();

		    $testMessage = $sms->buildTestSmsMessage();
			$cmd_test_media = sprintf( SMTOtuApi::CMD_test_modem_sms, SMTSms::INTERNAl_1 , $smsPhoneNumber, $testMessage );
			$this->getContext()->getOtuSocket()->send( $cmd_test_media, 0, 480 );//8mn of timeout for sms test
    	}
    	catch ( \Exception $e )
    	{
    	    $this->getContext()->traceException($e);
    	    if ( !( $e instanceof SMTMediaException ) )
    	    {
    	        $e = new SMTMediaException( SMTMediaException::ERROR_START_SMS_TEST );
    	    }    		   
    		$this->handleException($e);
    	}    			
    	return self::RESULT_OK;
    }
    
    /**
     * @url GET /test/updatesms
     */
    function testUpdateSms()
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$smsConfDto = new SMTConfSmsDto();
        	for ( $index = SMTSms::MIN_SMS_INDEX; $index <= SMTSms::MAX_SMS_INDEX; $index++)
        	{
        		$smsDto = new SMTSmsDto();
        		$smsDto->setValid( FALSE);
        		$smsDto->setIndex( $index );
        		$smsDto->setPhoneNumber( "" );
        		$smsConfDto->addSms( $smsDto );
        	}
        	$smsConfDto->setModemType("");
        	
        	$url = '/sms';
        	$json_string = json_encode( $smsConfDto->getJsonData() );
        	$page = new SMTTestRestApiPage($url, $json_string, 'POST');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }
    
    /**
     * @url GET /test/smstest
     */
    function testSmsTest()
    {    	
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$url = '/sms/0612134757/test';
        	$page = new SMTTestRestApiPage($url, "", 'POST');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }    
    
}

?>