<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\monitoring;

use app\services\alarm\SMTAttenuationOpticalAlarmDto;

use app\services\alarm\SMTOpticalAlarmDto;

use app\parser\SMTOtuApi;

use app\util\SMTUtil;

use app\util\SMTInvalidValueException;

use app\util\SMTRwDto;

use app\sharedmemory\SMTMemoryManager;

use app\sharedmemory\SMTIMemorySupport;
use app\services\alarm\SMTPeakOpticalAlarmDto;
use app\serviceshelper\monitoring\SMTTestMode;

/**
 * Link-test dto: port number, module running number, sequnecer, measure allowed. 
 * The identifier of the object is the link Id. 
 * The monitoring test can be NULL.
 * 
 * @author Sylvain Desplat
 */
class SMTLinkTestDto extends SMTRwDto implements SMTIMemorySupport
{       
    /**
     * Default link-test name prefix used when no name is specified by user
     * @var string
     */
    const DEFAULT_LINK_TEST_NAME_PREFIX = "Link ";
    
    /**
     * Link identifier
     * @var integer
     */
    protected $id = NULL;
    /**
     * Test identifier
     * @var integer
     */
    protected $testId = NULL;
    /**
     * @var string
     */    
    protected $name;        
    /**
     * @var string
     */
    protected $gpsFileName;
    /**
     * @var string
     */
    protected $kmlUrl;
    /**
     * @var int
     */
    protected $moduleRunningNumber;    
    /**
     * Port Number
     * @var int
     */    
    protected $portNumber;
    /**
     * @var bool
     */
    protected $measurementAllowed = FALSE;    
    /**
     * @var bool
     */
    protected $monitoringAllowed = FALSE;    

    /**
     * Last monitoring UTC timestamp in seconds
     * 
     * @var number
     */
    protected $lastMonitoringTimeStamp;    
    /**
     * Whether sequensor is not stopped 
     */
    protected $sequensorOn = FALSE;
    
    /**
     * @SMTAttributeInfosAnnotation(classname='app\services\monitoring\SMTTestDetailDto')
     * @var SMTTestDetailDto 
     */
    protected $testDetail = NULL;    
    
    /**
     * @SMTAttributeInfosAnnotation(classname='app\services\alarm\SMTAttenuationOpticalAlarmDto')
     * @var SMTAttenuationOpticalAlarmDto
     */
    protected $attenuationOpticalAlarm = NULL;
    
    /**
     * @SMTAttributeInfosAnnotation(classname='app\services\alarm\SMTPeakOpticalAlarmDto',islist='true')
     * @var array app\services\alarm\SMTPeakOpticalAlarmDto
     */
    protected $peakOpticalAlarm = array();
    
    /**
     * Test mode: OTDR,HIGHSENS,TAPPING...
     * @var string
     */
    protected $testMode = SMTTestMode::HIGHSENS;
        
    /**
     * Link Id 
     * @param string $id
     */
    public function setId( $id )
    {           
        if ( !isset($id) || $id === NULL || !SMTUtil::isInt( $id ) )
        {
        	throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "linkid", $id );
        }
    	$this->id = intval($id);
    }
    
    /**
     * Link Id
     * @return string
     */
    public function getId()
    {
    	return $this->id;
    }
    
    /**
     * Set Link Id = NULL
     * @param string $id
     */
    public function resetId( )
    {
    	$this->id = NULL;
    }
    
    /**
     * Set Test Id = NULL
     * 
     */
    public function resetTestId( )
    {
    	$this->testId = NULL;
    	$this->testDetail = NULL;
    }
    
    public function setTestId( $testId )
    {
    	if ($testId != NULL)
    	{
    	    $this->testId = SMTUtil::convertToInt( $testId );
        	
    	    //Update test detail info DTO with test Id (case of test creation).
    	    if ( $this->testDetail != NULL )
    	    {
    	       $this->testDetail->setTestId($testId);
    	    }
    	}
    	else
    	{
    		$this->testId = NULL;
    		$this->testDetail = NULL;
    	}
    }       
    
    public function getTestId()
    {
    	return $this->testId;
    }

    function setTestMode( $mode)
    {
    	$this->testMode= $mode;
    	//Update test detail info DTO with test mode
    	if ( $this->testDetail != NULL )
    	{
    		$this->testDetail->setTestMode($mode);
    	}
    }
    
    function getTestMode()
    {
    	return $this->testMode;
    }
    
    /**
     * Remove monitoring test from link-test dto
     */
    public function removeTest()
    {
        $this->testId = NULL;
        $this->testDetail = NULL;
    }
    
    /**
     * Set last monitoring UTC date time in seconds
     * 
     * @param number $lastMonitoringTimeStamp UTC date time in seconds
     */
    public function setLastMonitoringTimeStamp( $lastMonitoringTimeStamp )
    {
        $this->lastMonitoringTimeStamp = $lastMonitoringTimeStamp;
    }
        
    /**
     * Return last monitoring UTC date time in seconds
     * 
     * @return number last monitoring timestamp of the test in UTC
     */
    public function getLastMonitoringTimeStamp()
    {
    	return $this->lastMonitoringTimeStamp;
    }
        
    public function setName( $name )
    {
    	$this->name = $name;
    }    
    
    public function getName( )
    {
    	return $this->name;
    }
    
    public function setGpsFileName( $name )
    {
    	$this->gpsFileName = $name;
    }
    
    public function getGpsFileName( )
    {
    	return $this->gpsFileName;
    }
        
    public function setKmlUrl( $kmlUrl )
    {
    	$this->kmlUrl = $kmlUrl;
    }
        
    public function getKmlUrl( )
    {
    	return $this->kmlUrl;
    }
    
    /**
     * @param int $portNumber (if $portNumber is an int encoded in a string, the function try to convert it in integer)
     * @throws SMTInvalidValueException
     */
    public function setPortNumber( $portNumber )
    {
        if ( !isset($portNumber) || $portNumber === NULL || !SMTUtil::isInt( $portNumber ) )
        {
        	throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "portNumber", $portNumber );
        }        
    	$this->portNumber = intval($portNumber);
    }

    public function getPortNumber()
    {
    	return $this->portNumber != NULL ? intval($this->portNumber) : NULL;
    }    
    /**
     * @param int $moduleRunningNumber (if $moduleRunningNumber is an int encoded in a string, the function try to convert it in integer)
     * @throws SMTInvalidValueException
     */
    public function setModuleRunningNumber( $moduleRunningNumber )
    {
        if ( !isset($moduleRunningNumber) || $moduleRunningNumber === NULL || !SMTUtil::isInt( $moduleRunningNumber ) || ( $moduleRunningNumber <= 0 || $moduleRunningNumber > 2 ) )
        {
        	throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "moduleRunningNumber", $moduleRunningNumber );
        }        
    	$this->moduleRunningNumber = intval($moduleRunningNumber);
    }
        
    /**
     * Set module running number from the module position
     * @param string $modulePosition
     */
    public function setModulePosition( $modulePosition )
    {
        //WARNING mandatory for ltrim: ltrim can't evaluate the constant
        $mod_prefix = SMTOtuApi::RES_MOD;        
    	( $modulePosition!= NULL )? $this->setModuleRunningNumber( intval( ltrim( $modulePosition, $mod_prefix ) ) ): NULL;
    }
    
    public function getModuleRunningNumber()
    {
        return $this->moduleRunningNumber != NULL ? intval($this->moduleRunningNumber) : NULL;
    }    
    
    public function setMeasurementAllowed( $measurementAllowed )
    {
    	$this->measurementAllowed = filter_var($measurementAllowed, FILTER_VALIDATE_BOOLEAN);
    }
    
        
    public function isMeasurementAllowed()
    {
    	return $this->measurementAllowed;
    }    
    
    public function setMonitoringAllowed( $monitoringAllowed )
    {
    	$this->monitoringAllowed = filter_var($monitoringAllowed, FILTER_VALIDATE_BOOLEAN);
    }    
    
    public function isMonitoringAllowed()
    {
    	return $this->monitoringAllowed;
    }            
    
    public function setSequensorOn( $sequensorOn )
    {
    	$this->sequensorOn = filter_var($sequensorOn, FILTER_VALIDATE_BOOLEAN);
    }    
    
    public function isSequensorOn()
    {
    	return $this->sequensorOn;
    }        
        
    /**
     * 
     * @param SMTTestDetailDto $testDetail
     */
    public function setTestDetail( SMTTestDetailDto $testDetail = NULL )
    {
    	$this->testDetail = $testDetail;
    }
    /**
     * @return \app\services\monitoring\SMTTestDetailDto
     */
    public function getTestDetail()
    {
    	return $this->testDetail;
    }    
    
    /**
     * Attach the given optical alarm to its link-test
     * 
     * @param SMTOpticalAlarmDto $opticalAlarm
     */
    public function addOpticalAlarm( SMTOpticalAlarmDto $opticalAlarm )
    {
       if ( $opticalAlarm instanceof SMTAttenuationOpticalAlarmDto )
       {
           $this->setAttenuationOpticalAlarm( $opticalAlarm );
       }
       else if ( $opticalAlarm instanceof SMTPeakOpticalAlarmDto )
       {
       	$this->addPeakOpticalAlarm( $opticalAlarm );
       } 
    }
    
    public function setAttenuationOpticalAlarm( SMTAttenuationOpticalAlarmDto $attenuationOpticalAlarm )
    {
    	$this->attenuationOpticalAlarm = $attenuationOpticalAlarm;
    }
    
    /**
     * @return \app\services\alarm\SMTAttenuationOpticalAlarmDto
     */
    public function getAttenuationOpticalAlarm()
    {
    	return $this->attenuationOpticalAlarm;
    }
    
    /**
     * @param SMTPeakOpticalAlarmDto $peakAlarm
     */
    public function addPeakOpticalAlarm( SMTPeakOpticalAlarmDto $peakAlarm )
    {
    	array_push( $this->peakOpticalAlarm, $peakAlarm);
    } 
    
    /**
     * @param SMTPeakOpticalAlarmDto $peakAlarm
     */
    public function getPeakOpticalAlarms()
    {
    	return $this->peakOpticalAlarm;
    }  
    
    /**
     * Serialize as Json data (array) the object.<br> 
     * 
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
        $this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }

    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
        return get_class();
    }    
    
    /**
     * Create the Dto from a Json Dto in an array
     *
     * @param array $json_array json data in an array
     *
     * @return \app\services\monitoring\SMTLinkTestDto
     */
    static function getInstance( array &$json_array )
    {
    	return self::forge( get_class(), $json_array );
    }    
    
    /**
     * Save Dto in shared memory
     *
     * @throws SMTAPCMemoryException
     */
    public function save()
    {
    	SMTMemoryManager::saveDto( $this );
    }
    
    /**
     * Delete Dto from shared memory
     *
     * @throws SMTAPCMemoryException
     */
    public function delete()
    {
    	SMTMemoryManager::deleteDto( $this );
    }

    /**
     * Returns the dto class name.
     *
     * @return the dto class name.
     */
    public static function getClass()
    {
    	return __CLASS__;
    }
    
    /**
     * Link-Test dto cache is never obsolete: it must not be purged
     *
     * @return FALSE
     */
    public function isObsolete()
    {
    	return FALSE;
    }
    
    /**
     * No obsolescence for those data: returns current time.
     *
     * @see isObsolete()
     * @return number Timestamp of the last APC memory access (used to handle obsolescence of dtos in cache)
     */
    public function getLastAccess()
    {
    	return microtime(TRUE);
    }
    
    /**
     * No obsolescence for those data: do nothing
     *
     * @see isObsolete()
     *
     */
    public function touchAccess()
    {
    }
}

?>