<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\events;

use app\http\SMTHttpResponse;

use app\parser\SMTOtuSocketManager;

use app\http\SMTContext;

use app\services\activity\SMTActivityDto;

use app\services\alarm\SMTAlarmDto;

use app\util\SMTLogger;

use app\services\monitoring\SMTLinkTestDto;

use app\events\linktest\SMTLinkTestUpdateEventDto;

use app\util\SMTIOException;

use app\serviceshelper\monitoring\SMTTestOnDemandEventDto;

use app\serviceshelper\monitoring\SMTMeasureOnDemandEventDto;

use app\sharedmemory\SMTMemoryManager;

use app\events\operations\SMTOperationEventDto;

use app\services\alarm\otu\SMTOtuAlarmDto;
use app\services\alarm\otu\SMTOtuOpticalAlarmDto;
use app\services\alarm\otu\SMTOtuSystemAlarmDto;
use app\services\monitoring\SMTCurrentPortDto;

/**
 * Listens events sent by the application. If consumers are registered for that event, it is stored in a shared memory.
 * That shared memory is fetched by consumers which are long-delay-polling to wait for new event.
 * Events are retrieved by the consumers through messages with the id of the consumer and the list of events (only one in the case of an operation message). 
 * 
 * @author Sylvain Desplat
 */
class SMTEventMessageManager implements SMTIEventMessageProducer, SMTIEventMessageConsumer
{    
    /**
     * Conversation max duration: 30s
     * @var number
     */    
    const MAX_WAIT_TIME = 30;
    /**
     * Sleep 2 seconds between 2 alarm messages checks. 
     * 
     * @var number
     */    
    const SLEEP_TIME_SEC = 2;
    
    /**
     * @var SMTEventMessageManager
     * @access private
     * @static
     */
    private static $_instance = NULL;   
    
    /**
     * @return SMTEventMessageManager
     */
    public static function getInstance() 
    {
    	if( is_null(self::$_instance) ) 
    	{
    		self::$_instance = new SMTEventMessageManager();
    		
    	}
    	return self::$_instance;
    }           

    /**
     * Register a user operation, in order to be able to send to the consumer the matching result event.
     * If the operation message already exist just fetch it.
     * WARNING: ONLY ONE CONSUMER FOR ONE OPERATION: consumerId = operationId.
     *      
     * @param string $operationId The operation ID
     * 
     * @return SMTTestOnDemandEventConsumerDto
     */
    private function registerTestOnDemandOperation( $operationId )
    {
        $operationEventConsumer = NULL;
        
    	if ( SMTEventMessageManagerLockHandler::acquireTestOnDemandLock() )
    	{
    		try
    		{
    			if ( ( $operationEventConsumer = SMTMemoryManager::fetch( SMTTestOnDemandEventConsumerDto::getClass(), $operationId ) ) == NULL )
    			{
    				$operationEventConsumer = new SMTTestOnDemandEventConsumerDto();
    				$operationEventConsumer->setOperationId( $operationId );
    				$operationEventConsumer->save();
    			}
    			SMTEventMessageManagerLockHandler::releaseTestOnDemandLock();
    			
    			$this->purgeObsoleteTestOnDemandEventConsumers();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseTestOnDemandLock();
    			throw $e;
    		}
    	}
    	return $operationEventConsumer;
    }  
      
    /**
     * Whether test events have yet not been sent to consumers.
     * 
     * @return boolean
     */
    private function hasTestOnDemandStatusNotSent()
    {
    	$operationEventConsumers = $this->fetchTestOnDemandEventConsumers();
    	if ( $operationEventConsumers != NULL )
    	{
    		foreach ( $operationEventConsumers as $operationEventConsumer )
    		{
    			if ( $operationEventConsumer != NULL && $operationEventConsumer->hasEvents() )
    			{
    				return TRUE;
    			}
    		}
    		return FALSE;
    	}
    	return FALSE;
    }    
    
    /**
     * Add a new measure on demand event listener for the given consumer: 
     * create a measure consumer message for the given consumer.
     * WARNING: ONLY ONE CONSUMER FOR ONE OPERATION: consumerId = operationId.
     *
     * @param string $consumerId
     * @return the Event Consumer if it was created successfully and if it was not already registered
     */
    public function registerMeasureOnDemandConsumer( $consumerId )
    {
    	$measureOnDemandEventConsumer = NULL;
    
    	if ( SMTEventMessageManagerLockHandler::acquireMeasureOnDemandLock() )
    	{
    		try
    		{
    			if ( SMTMemoryManager::fetch( SMTMeasureOnDemandEventConsumerDto::getClass(), $consumerId) == NULL )
    			{
    				$measureOnDemandEventConsumer = new SMTMeasureOnDemandEventConsumerDto();
    				$measureOnDemandEventConsumer->setConsumerId( $consumerId );
    				$measureOnDemandEventConsumer->save();
    			}
    			SMTEventMessageManagerLockHandler::releaseMeasureOnDemandLock();
    
    			$this->purgeObsoleteMeasureOnDemandEventConsumers();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseMeasureOnDemandLock();
    			throw $e;
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to register measure on demand event consumer: ".$consumerId, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}
    
    	return $measureOnDemandEventConsumer;
    }    
    
    
    /**
     * Register a user operation, in order to be able to send to the consumer the matching result event.
     * If a consumer is already registered for the operation, just retrieve it: operationId = consumerId
     * WARNING: ONLY ONE CONSUMER FOR ONE OPERATION: consumerId = operationId.
     *
     * @param string $operationId The operation ID
     * 
     * @return SMTMeasureOnDemandEventConsumerDto
     */
    private function registerMeasureOnDemandOperation( $operationId )
    {
        $operationEventConsumer = NULL;
        
    	if ( SMTEventMessageManagerLockHandler::acquireMeasureOnDemandLock() )
    	{
    		try
    		{
    			if ( ( $operationEventConsumer = SMTMemoryManager::fetch( SMTMeasureOnDemandEventConsumerDto::getClass(), $operationId) ) == NULL )
    			{
    				$operationEventConsumer = new SMTMeasureOnDemandEventConsumerDto();
    				$operationEventConsumer->setOperationId($operationId);
    				$operationEventConsumer->save();
    			}
    			SMTEventMessageManagerLockHandler::releaseMeasureOnDemandLock();
    			
    			$this->purgeObsoleteMeasureOnDemandEventConsumers();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseMeasureOnDemandLock();
    			throw $e;
    		}
    	}
    	return $operationEventConsumer;
    }    
    
    /**
     * Whether measurement events have not yet been sent to consumers.
     * 
     * @return boolean
     */
    private function hasMeasureOndemandStatusNotSent()
    {
    	$operationEventConsumers = $this->fetchMeasureOnDemandEventConsumers();
    	if ( $operationEventConsumers != NULL )
    	{
    		foreach ( $operationEventConsumers as $operationEventConsumer )
    		{
    			if ( $operationEventConsumer != NULL && $operationEventConsumer->hasEvents() )
    			{
    				return TRUE;
    			}
    		}
    		return FALSE;
    	}
    	return FALSE;
    }    
    
    /**
     * Add the operation event (test on demand, measurement) to the corresponding operation message listener.
     * WARNING: ONLY ONE CONSUMER FOR ONE OPERATION: consumerId = operationId.
     *
     * @param SMTOperationEventDto event The operation event to add to the operation message and to be sent to the consumer.
     */
    public function sendOperationEvent( SMTOperationEventDto $event )
    {
        //try to add operation if it has not yet been added
        if ( $event instanceof SMTMeasureOnDemandEventDto )
        {
            $operationEventConsumer = $this->registerMeasureOnDemandOperation( $event->getOperationId() );
            if ( $operationEventConsumer != NULL && SMTEventMessageManagerLockHandler::acquireMeasureOnDemandLock() )
            {
            	try
            	{
            		$operationEventConsumer->addEvent( $event );
            		$operationEventConsumer->save();
            		SMTEventMessageManagerLockHandler::releaseMeasureOnDemandLock();
            	}
            	catch( \Exception $e )
            	{
            		SMTEventMessageManagerLockHandler::releaseMeasureOnDemandLock();
            		throw $e;
            	}
            }
        }
        elseif ( $event instanceof SMTTestOnDemandEventDto )
        {
            $operationEventConsumer = $this->registerTestOnDemandOperation( $event->getOperationId() );
            if ( $operationEventConsumer != NULL && SMTEventMessageManagerLockHandler::acquireTestOnDemandLock() )
            {
            	try
            	{
            		$operationEventConsumer->addEvent( $event );
            		$operationEventConsumer->save();
            		SMTEventMessageManagerLockHandler::releaseTestOnDemandLock();
            	}
            	catch( \Exception $e )
            	{
            		SMTEventMessageManagerLockHandler::releaseTestOnDemandLock();
            		throw $e;
            	}
            }
        }                
    }

    /**
     * Register all event listeners (except test on demand) for the given consumer
     *
     * @param string $consumerId
     */
    public function registerEventConsumer( SMTConversationMessageFilterDto $messageFilter )
    {
        SMTLogger::getInstance()->trace( "registerEventConsumer for consumerId = ".$messageFilter->getConsumerId()." and for message type: ".$messageFilter->getMessageType(), SMTLogger::DEBUG );        
        
        if ( SMTOpticalAlarmMessageDto::getShortClassName() == $messageFilter->getMessageType() )
        {
            $this->registerOpticalAlarmEventConsumer( $messageFilter->getConsumerId() );
        }
        else if ( SMTSystemAlarmMessageDto::getShortClassName() == $messageFilter->getMessageType() )
        {
        	$this->registerSystemAlarmEventConsumer( $messageFilter->getConsumerId() );
        }        
        else if ( SMTActivityMessageDto::getShortClassName() == $messageFilter->getMessageType() )
        {            
            $this->registerActivityEventConsumer( $messageFilter->getConsumerId() );            
        }
        else if ( SMTLinkTestUpdateMessageDto::getShortClassName() == $messageFilter->getMessageType() )
        {
        	$this->registerLinkTestUpdateEventConsumer( $messageFilter->getConsumerId() );        
        }        
        else if ( SMTMeasureOnDemandMessageDto::getShortClassName() == $messageFilter->getMessageType() )
        {
            $this->registerMeasureOnDemandConsumer( $messageFilter->getConsumerId() );
        }
        
        if ( SMTEventMessageManagerLockHandler::acquireConversationLock() )
        {
        	try
        	{
                //For each event consumer, always store message filter with consumerId, conversationId and message type in APC memory
                $messageFilter->save();        
                SMTEventMessageManagerLockHandler::releaseConversationLock();
                        
                $this->purgeObsoleteConversations();
        	}
        	catch( \Exception $e )
        	{
        		SMTEventMessageManagerLockHandler::releaseConversationLock();
        		throw $e;
        	}
        }
        else
        {
        	SMTLogger::getInstance()->trace("Couldn't acquire lock to register event consumer: ".$messageFilter->getConsumerId(), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
        }
    }
    
    /**
     * Unregister all event listeners for the given consumer
     *
     * @param string $consumerId
     */
    public function unregisterEventConsumer( $consumerId )
    {
        SMTLogger::getInstance()->trace( "unregisterEventConsumer for consumerId = ".$consumerId, SMTLogger::DEBUG );
        
        $this->unregisterActivityEventConsumer($consumerId);
        $this->unregisterLinkTestUpdateEventConsumer($consumerId);
        $this->unregisterOpticalAlarmEventConsumer($consumerId);
        $this->unregisterSystemAlarmEventConsumer($consumerId);
        $this->unregisterMeasureOndemandEventConsumer($consumerId);
        $this->unregisterTestOndemandEventConsumer($consumerId);
        
        //For each event consumer, always remove message filters with the given consumerId, conversationId and message type in APC memory
        if ( SMTEventMessageManagerLockHandler::acquireConversationLock() )
        {
        	try
        	{
        	    $conversationMessageFilters = SMTMemoryManager::fetchAll( SMTConversationMessageFilterDto::getClass() );
        	    if ( $conversationMessageFilters != NULL && count( $conversationMessageFilters ) > 0 )
        	    {
        	    	foreach ( $conversationMessageFilters as $conversationMessageFilter )
        	    	{
        	    		if ( $conversationMessageFilter != NULL && $conversationMessageFilter->getConsumerId() == $consumerId )
        	    		{
        	    		    $conversationMessageFilter->delete();        	    		    
        	    		}
        	    	}
        	    }
                SMTEventMessageManagerLockHandler::releaseConversationLock();
        	}
        	catch( \Exception $e )
        	{
        		SMTEventMessageManagerLockHandler::releaseConversationLock();
        		throw $e;
        	}
        }        
        else
        {
        	SMTLogger::getInstance()->trace("Couldn't acquire lock to unregister conversation event consumer: ".$consumerId, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );            
        }        
    }
    
    /**
     * Add a new Optical alarm event listener for the given consumer: create an alarm message for the given consumer
     *
     * @param string $consumerId
     *
     * @return the Event Consumer if it was created successfully and if it was not already registered
     */
    public function registerOpticalAlarmEventConsumer( $consumerId )
    {
    	$alarmEventConsumer = NULL;
    
    	if ( SMTEventMessageManagerLockHandler::acquireAlarmLock() )
    	{
    		try
    		{
    			if ( SMTMemoryManager::fetch( SMTOpticalAlarmEventConsumerDto::getClass(), $consumerId) == NULL )
    			{
    			    SMTLogger::getInstance()->trace( "Register Optical alarm event consumer: ".$consumerId, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );    			    	
    				$alarmEventConsumer = new SMTOpticalAlarmEventConsumerDto();
    				$alarmEventConsumer->setConsumerId( $consumerId );
    				$alarmEventConsumer->save();
    			}
    			SMTEventMessageManagerLockHandler::releaseAlarmLock();
    			 
    			$this->purgeObsoleteOpticalAlarmEventConsumers();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseAlarmLock();
    			throw $e;
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to register Optical alarm event consumer: ".$consumerId, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}
    	return $alarmEventConsumer;
    }
    
    /**
     * Remove the alarm event listener for the given consumer and clean up its events
     *
     * @param string $consumerId
     */
    public function unregisterOpticalAlarmEventConsumer( $consumerId )
    {
    	if ( SMTEventMessageManagerLockHandler::acquireAlarmLock() )
    	{
    		try
    		{
    		    SMTLogger::getInstance()->trace( "Unregister Optical alarm event consumer: ".$consumerId, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    			SMTMemoryManager::delete( SMTOpticalAlarmEventConsumerDto::getClass(), $consumerId);
    			SMTEventMessageManagerLockHandler::releaseAlarmLock();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseAlarmLock();
    			throw $e;
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to unregister Optical alarm event consumer: ".$consumerId, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}
    }    
    
    /**
     * Add a new System alarm event listener for the given consumer: create an alarm message for the given consumer
     *
     * @param string $consumerId
     * 
     * @return the Event Consumer if it was created successfully and if it was not already registered
     */    
    public function registerSystemAlarmEventConsumer( $consumerId )
    {
        $alarmEventConsumer = NULL;
    
    	if ( SMTEventMessageManagerLockHandler::acquireAlarmLock() )
    	{
    	    try
    	    {
        	    if ( SMTMemoryManager::fetch( SMTSystemAlarmEventConsumerDto::getClass(), $consumerId) == NULL )
        	    {
    	            SMTLogger::getInstance()->trace( "Register System alarm event consumer: ".$consumerId, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
            	    $alarmEventConsumer = new SMTSystemAlarmEventConsumerDto();
            	    $alarmEventConsumer->setConsumerId( $consumerId );   
            	    $alarmEventConsumer->save();                	                
            	}
            	SMTEventMessageManagerLockHandler::releaseAlarmLock();
            	
            	$this->purgeObsoleteSystemAlarmEventConsumers();
        	}
        	catch( \Exception $e )
        	{
        	    SMTEventMessageManagerLockHandler::releaseAlarmLock();
        		throw $e;
        	}        	 
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to register System alarm event consumer: ".$consumerId, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}    	
    	return $alarmEventConsumer;
    }
    
    /**
     * Remove the System alarm event listener for the given consumer and clean up its events
     *
     * @param string $consumerId
     */   
    public function unregisterSystemAlarmEventConsumer( $consumerId )
    {
        if ( SMTEventMessageManagerLockHandler::acquireAlarmLock() )
    	{
    	    try
    	    {
    	        SMTLogger::getInstance()->trace( "Unregister system alarm event consumer: ".$consumerId, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
        	    SMTMemoryManager::delete( SMTSystemAlarmEventConsumerDto::getClass(), $consumerId);
            	SMTEventMessageManagerLockHandler::releaseAlarmLock();
        	}
        	catch( \Exception $e )
        	{
        	    SMTEventMessageManagerLockHandler::releaseAlarmLock();
        		throw $e;
        	}        	 
    	}	
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to unregister alarm event consumer: ".$consumerId, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}    	
    }    
    
    /**
     * Add the alarm event to the alarm messages to be sent if consumers are registered.
     *
     * @param SMTAlarmDto $alarm The alarm to send
     */
    public function sendAlarmEvent( $alarms, SMTOtuAlarmDto $otuAlarm /*SMTAlarmDto $alarm*/ )
    {  
    	$event = SMTEventFactory::createAlarmEvent( $alarms, $otuAlarm);
        //SMTLogger::getInstance()->trace( "send Alarm Event message for alarm Id = ".$alarm->getId(), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
                
    	if ( $event != NULL )
    	{
	    	if ( SMTEventMessageManagerLockHandler::acquireAlarmLock())
	    	{
	    	    try
	    	    {
	    	    	if ( $otuAlarm instanceof SMTOtuOpticalAlarmDto /*$alarm instanceof SMTOpticalAlarmDto*/ )
	    	        {
	        	    	$alarmEventConsumer = SMTMemoryManager::fetchAll( SMTOpticalAlarmEventConsumerDto::getClass() );
	    
	        	    	if ( $alarmEventConsumer != NULL && count( $alarmEventConsumer ) > 0 )
	        	    	{
	            	    	foreach ( $alarmEventConsumer as &$eventConsumer)
	            	    	{
	            	    	    $eventConsumer->addEvent( $event );
	            	    	}
	            	    	SMTLogger::getInstance()->trace( "send Alarm Event message for alarm Id = ".$event->getId(), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
	            	    	SMTMemoryManager::saveDtos( $alarmEventConsumer );
	        	    	}
	    	        }    	    	
	    	        else if ( $otuAlarm instanceof SMTOtuSystemAlarmDto /*$alarm instanceof SMTSystemAlarmDto*/ )
	    	        {
	        	    	$alarmEventConsumer = SMTMemoryManager::fetchAll( SMTSystemAlarmEventConsumerDto::getClass() );        	    	
	        	    	        	    	
	        	    	if ( $alarmEventConsumer != NULL && count( $alarmEventConsumer ) > 0 )
	        	    	{        	    	    
	        	    		foreach ( $alarmEventConsumer as &$eventConsumer)
	        	    		{
	        	    			$eventConsumer->addEvent( $event );
	        	    		}
	        	    		SMTLogger::getInstance()->trace( "send Alarm Event message for alarm Id = ".$event->getId(), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
	        	    		SMTMemoryManager::saveDtos( $alarmEventConsumer );
	        	    	}
	    	        }    	    	
	    	    	
	    	    	SMTEventMessageManagerLockHandler::releaseAlarmLock();
	    	    }
	    	    catch( \Exception $e )
	    	    {
	    	    	SMTEventMessageManagerLockHandler::releaseAlarmLock();
	    	    	throw $e;
	    	    }
	    	}
	    	else
	    	{
	    		SMTLogger::getInstance()->trace("Couldn't acquire lock to send alarm event: ".$event->getId(), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
	    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
	    	}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't create alarm message event for alarm type: ".$otuAlarm->getEventTypeCode(), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    	}
    }    
    
    /**
     * Notify web clients that optical alarms must be reloaded.
     *
     */
    public function sendRefreshOpticalAlarms()
    {
        $this->sendRefreshAllAlarms( TRUE, FALSE );
    }
    
    
    /**
     * Notify web clients that all alarms (optical and system) must be reloaded.
     * 
     * @param boolean $refreshOpticalAlarm Optional parameters TRUE by default: optical alarms are refreshed.
     * @param boolean $refreshSystemAlarm Optional parameters TRUE by default: system alarms are refreshed.  
     */
    public function sendRefreshAllAlarms( $refreshOpticalAlarm = TRUE, $refreshSystemAlarm = TRUE )
    {
        SMTLogger::getInstance()->trace( "send Alarm Event message for refreshing all alarms", SMTLogger::INFO, __FILE__,__METHOD__,__LINE__ );
        
        if ( SMTEventMessageManagerLockHandler::acquireAlarmLock())
        {
        	try
        	{
        	    if ( $refreshOpticalAlarm )
        	    {
        	        $refreshAllOpticalAlarmsEvent = SMTEventFactory::createRefreshAllOpticalAlarmEvent();
        	        
        			$alarmEventConsumer = SMTMemoryManager::fetchAll( SMTOpticalAlarmEventConsumerDto::getClass() );
        
        			if ( $alarmEventConsumer != NULL && count( $alarmEventConsumer ) > 0 )
        			{
        				foreach ( $alarmEventConsumer as &$eventConsumer)
        				{
        					$eventConsumer->addEvent( $refreshAllOpticalAlarmsEvent );
        				}
        				SMTMemoryManager::saveDtos( $alarmEventConsumer );
        			}
        	    }

        	    if ( $refreshSystemAlarm )
        	    {
        	        $refreshAllSystemAlarmsEvent = SMTEventFactory::createRefreshAllSystemAlarmEvent();
        	        
        			$alarmEventConsumer = SMTMemoryManager::fetchAll( SMTSystemAlarmEventConsumerDto::getClass() );
        
        			if ( $alarmEventConsumer != NULL && count( $alarmEventConsumer ) > 0 )
        			{
        				foreach ( $alarmEventConsumer as &$eventConsumer)
        				{
        					$eventConsumer->addEvent( $refreshAllSystemAlarmsEvent );
        				}
        				SMTMemoryManager::saveDtos( $alarmEventConsumer );
    			}
        	    }
        
        		SMTEventMessageManagerLockHandler::releaseAlarmLock();
        	}
        	catch( \Exception $e )
        	{
        		SMTEventMessageManagerLockHandler::releaseAlarmLock();
        		throw $e;
        	}
        }
        else
        {
        	SMTLogger::getInstance()->trace("Couldn't acquire lock to send refresh all alarm event: ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
        }
        
    }
    
    
    /**
     * Add a new link-test update event listener for the given consumer: create a link-test message for the given consumer
     *
     * @param string $consumerId
     * @return the Event Consumer if it was created successfully and if it was not already registered
     */
    public function registerLinkTestUpdateEventConsumer( $consumerId )
    {
    	$linkTestUpdateEventConsumer = NULL;
    
    	//SMTLogger::getInstance()->trace( "registerLinkTestUpdateEventConsumer for consumerId = ".$consumerId );
    	if ( SMTEventMessageManagerLockHandler::acquireLinkTestUpdateLock() )
    	{
    		try
    		{
    			if ( SMTMemoryManager::fetch( SMTLinkTestUpdateEventConsumerDto::getClass(), $consumerId) == NULL )
    			{
    			    //SMTLogger::getInstance()->trace( "createLinkTestUpdateEventConsumer for consumerId = ".$consumerId );
    				$linkTestUpdateEventConsumer = new SMTLinkTestUpdateEventConsumerDto();
    				$linkTestUpdateEventConsumer->setConsumerId( $consumerId );
    				$linkTestUpdateEventConsumer->save();
    			}
    			SMTEventMessageManagerLockHandler::releaseLinkTestUpdateLock();
    
    			$this->purgeObsoleteLinkTestUpdateEventConsumers();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseLinkTestUpdateLock();
    			throw $e;
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to register link-test update event consumer: ".$consumerId, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}
    
    	return $linkTestUpdateEventConsumer;
    }
    
    /**
     * Remove the link-test update event listener for the given consumer and clean up its events
     *
     * @param string $consumerId
     */
    public function unregisterLinkTestUpdateEventConsumer( $consumerId )
    {
    	if ( SMTEventMessageManagerLockHandler::acquireLinkTestUpdateLock() )
    	{
    		try
    		{
    			SMTMemoryManager::delete( SMTLinkTestUpdateEventConsumerDto::getClass(), $consumerId );
    			SMTEventMessageManagerLockHandler::releaseLinkTestUpdateLock();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseLinkTestUpdateLock();
    			throw $e;
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to unregister link-test update event consumer: ".$consumerId, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}
    }
    
    /**
     * Add a new activity event listener for the given consumer: create an activity message for the given consumer
     *
     * @param string $consumerId
     * @return the Event Consumer if it was created successfully and if it was not already registered
     */
    public function registerActivityEventConsumer( $consumerId )
    {
        $activityEventConsumer = NULL;
        
        if ( SMTEventMessageManagerLockHandler::acquireActivityLock() )
        {
        	try
        	{
        		if ( SMTMemoryManager::fetch( SMTActivityEventConsumerDto::getClass(), $consumerId) == NULL )
        		{
        			$activityEventConsumer = new SMTActivityEventConsumerDto();
        			$activityEventConsumer->setConsumerId( $consumerId );
        			$activityEventConsumer->save();
        		}
        		SMTEventMessageManagerLockHandler::releaseActivityLock();
        		
        		$this->purgeObsoleteActivityEventConsumers();
        	}
        	catch( \Exception $e )
        	{
        		SMTEventMessageManagerLockHandler::releaseActivityLock();
        		throw $e;
        	}
        }    
        else
        {
        	SMTLogger::getInstance()->trace("Couldn't acquire lock to register activity event consumer: ".$consumerId, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
        }
        
        return $activityEventConsumer;
    }                   
    
    /**
     * Remove the activity event listener for the given consumer and clean up its events
     *
     * @param string $consumerId
     */   
    public function unregisterActivityEventConsumer( $consumerId )
    {
        if ( SMTEventMessageManagerLockHandler::acquireActivityLock() )
        {
        	try
        	{
        	    SMTLogger::getInstance()->trace( "Purge activity events consumer: ".$consumerId, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
        		SMTMemoryManager::delete( SMTActivityEventConsumerDto::getClass(), $consumerId );
        		SMTEventMessageManagerLockHandler::releaseActivityLock();
        	}
        	catch( \Exception $e )
        	{
        		SMTEventMessageManagerLockHandler::releaseActivityLock();
        		throw $e;
        	}
        }    
        else
        {
        	SMTLogger::getInstance()->trace("Couldn't acquire lock to unregister activity event consumer: ".$consumerId, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
        }
    }    
    
    /**
     * Remove the measure on demand event listener for the given consumer and clean up its events
     *
     * @param string $consumerId
     */
    public function unregisterMeasureOndemandEventConsumer( $consumerId )
    {
    	if ( SMTEventMessageManagerLockHandler::acquireMeasureOnDemandLock() )
    	{
    		try
    		{
    		    SMTLogger::getInstance()->trace( "Purge measure on demand events consumer: ".$consumerId, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    			SMTMemoryManager::delete( SMTMeasureOnDemandEventConsumerDto::getClass(), $consumerId );
    			SMTEventMessageManagerLockHandler::releaseMeasureOnDemandLock();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseMeasureOnDemandLock();
    			throw $e;
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to register MeasureOndemand event consumer: ".$consumerId, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}
    }
    
    /**
     * Remove the test on demand event listener for the given consumer and clean up its events
     *
     * @param string $consumerId
     */
    public function unregisterTestOndemandEventConsumer( $consumerId )
    {
    	if ( SMTEventMessageManagerLockHandler::acquireTestOnDemandLock() )
    	{
    		try
    		{
    		    SMTLogger::getInstance()->trace( "Purge test on demand events consumer: ".$consumerId, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    			SMTMemoryManager::delete( SMTTestOnDemandEventConsumerDto::getClass(), $consumerId );
    			SMTEventMessageManagerLockHandler::releaseTestOnDemandLock();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseTestOnDemandLock();
    			throw $e;
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to register TestOndemand event consumer: ".$consumerId, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}    	
    }       
    
    /**
     * Send the link-test update event to the link-test update messages to notify consumers of an update of the link-test DTO
     *
     * @param SMTLinkTestUpdateEventDto $event The link-test update event embedding the link-test DTO updated; If $linkTest is null, ask an update of all the link-test
     */
    public function sendLinkTestUpdateEvent( SMTLinkTestUpdateEventDto  $event )
    {
    	SMTLogger::getInstance()->trace( "sendLinkTestUpdateEvent for linkTest Id = ".$event->getId(), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    	 
    	if ( SMTEventMessageManagerLockHandler::acquireLinkTestUpdateLock() )
    	{
    		try
    		{
    			$linkTestUpdateEventConsumer = SMTMemoryManager::fetchAll( SMTLinkTestUpdateEventConsumerDto::getClass() );
    			if ( $linkTestUpdateEventConsumer != NULL && count( $linkTestUpdateEventConsumer ) > 0 )
    			{
    				foreach ( $linkTestUpdateEventConsumer as &$eventConsumer )
    				{
    					$eventConsumer->addEvent( $event );
    				}
    				SMTMemoryManager::saveDtos( $linkTestUpdateEventConsumer );
    			}
    
    			SMTEventMessageManagerLockHandler::releaseLinkTestUpdateLock();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseLinkTestUpdateLock();
    			throw $e;
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to send link-test update event for link-test: ".$event->getId(), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}
    }
    
    /**
     * Create and add the link-test update event to the link-test update messages to notify consumers of an update of the link-test DTO
     *
     * @param SMTLinkTestDto $linkTest The link-test DTO updated; If $linkTest is null, ask an update of all the link-test
     * @param boolean $isDeleted Whether the given link-test is being deleted ( the link + the test)
     * @param boolean $isRefreshAll Whether all the link-tests must be refreshed.
     */
    public function createAndSendLinkTestUpdateEvent( SMTLinkTestDto $linkTest = NULL, $isDeleted = FALSE, $isRefreshAll = FALSE, $currentPort = SMTCurrentPortDto::INVALID_CURRENT_PORT)
    {
        $event = SMTEventFactory::createLinkTestUpdateEvent( $linkTest, $isDeleted, $isRefreshAll, $currentPort );
        $this->sendLinkTestUpdateEvent($event);
    }    
    
    /**
     * Create and add the link-test update event to the link-test update messages to notify consumers of an update of all the link-test DTO
     *
     */
    public function createAndSendLinkTestUpdateAllEvent()
    {
    	$event = SMTEventFactory::createLinkTestUpdateEvent( NULL, FALSE, TRUE );
    	$this->sendLinkTestUpdateEvent($event);
    }    
    
    /**
     * Add the activity to the activity messages to be sent if consumers are registered.
     *
     * @param SMTActivityDto $activityDto The otu activity to send
     */
    public function sendActivityEvent( SMTActivityDto $activityDto )
    {
        $event = SMTEventFactory::createActivityEvent( $activityDto );
        SMTLogger::getInstance()->trace( "send Activity Event message for activity Id = ".$event->getId(), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
                
        if ( SMTEventMessageManagerLockHandler::acquireActivityLock())
        {
        	try
        	{
        		$activityEventConsumer = SMTMemoryManager::fetchAll( SMTActivityEventConsumerDto::getClass() );
        		if ( $activityEventConsumer != NULL && count( $activityEventConsumer ) > 0 )
        		{
            		foreach ( $activityEventConsumer as &$eventConsumer )
            		{
            			$eventConsumer->addEvent( $event );
            		}
            		SMTMemoryManager::saveDtos( $activityEventConsumer );
        		}
        
        		SMTEventMessageManagerLockHandler::releaseActivityLock();
        	}
        	catch( \Exception $e )
        	{
        		SMTEventMessageManagerLockHandler::releaseActivityLock();
        		throw $e;
        	}
        }    
        else
        {
        	SMTLogger::getInstance()->trace("Couldn't acquire lock to send activity event: ".$event->getId(), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
        }        
    }        
    
    /**
     * Whether there are activity message consumers registered
     *
     * @return Whether there are activity message consumers registered
     */
    public function hasActivityConsumersRegistered()
    {
        $consumersRegistered = FALSE;
    	if ( SMTEventMessageManagerLockHandler::acquireActivityLock())
    	{
    		try
    		{
    			$activityEventConsumer = SMTMemoryManager::fetchAll( SMTActivityEventConsumerDto::getClass() );
    			if ( $activityEventConsumer != NULL && count( $activityEventConsumer ) > 0 )
    			{
    			    $consumersRegistered = TRUE;
    			}
    
    			SMTEventMessageManagerLockHandler::releaseActivityLock();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseActivityLock();
    			throw $e;
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to retrive activity consumers", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}
    	return $consumersRegistered;
    }
    
    /**
     * Get all the pending messages for the given conversation id.
     * That method is called by long-delay-polling
     *
     * @param string $conversationId
     *
     * @return SMTMessageCarrierDto the messages with the events to send to the consumer
     */
    public function getConversationMessages( $conversationId, SMTContext $context )
    {
        //for long delay polling requests, unlock session data
        session_write_close();
        
        //conversation filters for the given conversationId
        $conversationMessageFilters = array();
        
        $messages = new SMTMessageCarrierDto();
        $time_start = microtime(true);
        
        //check if OTU application is still here
        //Add a keep alive to detect if OTU application is still running; that keep alive is synchronized with messages retrieval for GWT clients.
        try 
        {
            SMTOtuSocketManager::checkOtuConnection( $context );
        }
        catch( \Exception $e )
    	{
        	//handle exception and send back exception directly to client: no connection to OTU application
        	//WARNING: die on exception if a problem occurs:
        	SMTHttpResponse::handleException( $e, $context );
    	}
        
        while (  microtime(true) - $time_start < self::MAX_WAIT_TIME )
        {
            //sleep SLEEP_TIME_SEC / 2 before fetching messages
            sleep( self::SLEEP_TIME_SEC / 2 );
            try 
            {
                if ( SMTEventMessageManagerLockHandler::acquireConversationLock() )
                {
                	try
                	{
                	    //reset conversation message filters
                	    $conversationMessageFilters = array();
                	    
                	    //fetch current conversations:
                	    $allConversationMessageFilters = SMTMemoryManager::fetchAll( SMTConversationMessageFilterDto::getClass() );
                	    
                	    if ( $allConversationMessageFilters != NULL )
                	    {
                	    	foreach ( $allConversationMessageFilters as $conversationMessageFilter )
                	    	{
                	    		if ( $conversationMessageFilter->getConversationId() == $conversationId )
                	    		{                	    		    
                	    			//update last access to conversation
                	    			$conversationMessageFilter->touchAccess();
                	    			$conversationMessageFilter->save();
                	    			array_push( $conversationMessageFilters, $conversationMessageFilter );
                	    		}
                	    	}
                	    }
                	    
                	    SMTEventMessageManagerLockHandler::releaseConversationLock();
                	}            	
                	catch( \Exception $e )
                	{
                		SMTEventMessageManagerLockHandler::releaseConversationLock();
                		throw $e;
                	}

                	//retrieve messages outside critical section
            		foreach ( $conversationMessageFilters as $conversationMessageFilter )
            		{
            	       $this->retrieveMessages( $messages, $conversationMessageFilter->getConsumerId(), $conversationMessageFilter->getMessageType() );
            		}
                }
                else
                {
                	SMTLogger::getInstance()->trace("Couldn't acquire lock to retrieve Conversation ".$conversationId, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
                	throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
                }
                                                                                    	 
            	//messages were found, send them
            	if ( $messages->hasMessages() )
            	{
            		break;
            	}
            }
            //catch all exceptions: if message retrieval failed, retry.
            catch( \Exception $e)
            {
                SMTLogger::getInstance()->traceException($e);
            }
        	//sleep SLEEP_TIME_SEC / 2 before fetching messages 
        	sleep( self::SLEEP_TIME_SEC / 2 );
        }
        
        return $messages;
    }
    
    /**
     * Get all the pending messages for the given consumer id.
     * That method is called by long-delay-polling
     *
     * @param string $consumerId The destinator of the message
     *
     * @return SMTMessageCarrierDto the messages with the events to send to the consumer
     */
    public function getMessages( $consumerId )
    {
        //for long delay polling requests, unlock session data
        session_write_close();
        
    	$messages = new SMTMessageCarrierDto();
    	$time_start = microtime(true);
    	 
    	while (  microtime(true) - $time_start < self::MAX_WAIT_TIME )
    	{
    	    $this->retrieveMessages( $messages, $consumerId );
    		 
    		//messages found, send them
    		if ( $messages->hasMessages() )
    		{
    			break;
    		}
    			
    		sleep( self::SLEEP_TIME_SEC );
    	}
    	 
    	return $messages;
    }
    
    /**
     * Retrieve all the pending messages for the given consumer id.
     * 
     * @param SMTMessageCarrierDto $messages The message carrier to fill with retreived messages
     * @param string $consumerId The destinator of the message
     * @param string $messageType (optional: NUll by default): used to fetch only messages of the given type
     * 
     */
    private function retrieveMessages( SMTMessageCarrierDto $messages, $consumerId, $messageType = NULL )
    {       
        if ( $messageType === NULL || $messageType == SMTActivityMessageDto::getShortClassName() )
        {
    	    $activityMessages = $this->getActivityMessages( $consumerId );
    	    if ( $activityMessages != NULL )
    	    {
    	        for( $i=0; $i<count($activityMessages); $i++ )
    	        {
    	        	$messages->addActivityMessage( $activityMessages[$i] );
    	        }
    	    }
        }
            
        if ( $messageType == NULL || $messageType == SMTOpticalAlarmMessageDto::getShortClassName() )
        {
        	$alarmMessages = $this->getOpticalAlarmMessages( $consumerId );
        	if ( $alarmMessages != NULL )
        	{
        		for( $i=0; $i<count($alarmMessages); $i++ )
        		{
        			$messages->addOpticalAlarmMessage( $alarmMessages[$i] );
        		}
        	}
        }        

        if ( $messageType == NULL || $messageType == SMTSystemAlarmMessageDto::getShortClassName() )
        {
        	$systemAlarmMessages = $this->getSystemAlarmMessages( $consumerId );
        	if ( $systemAlarmMessages != NULL )
        	{
        		for( $i=0; $i<count($systemAlarmMessages); $i++ )
        		{
        			$messages->addSystemAlarmMessage( $systemAlarmMessages[$i] );
        		}
        	}
        }        
        
        if ( $messageType == NULL || $messageType == SMTMeasureOnDemandMessageDto::getShortClassName() )
        {
            $measureOnDemandMessages = $this->getMeasureOnDemandMessages( $consumerId );
            if ( $measureOnDemandMessages != NULL )
            {
            	for( $i=0; $i<count($measureOnDemandMessages); $i++ )
            	{
            		$messages->addMeasureOnDemandMessage( $measureOnDemandMessages[$i] );
            	}
            }            
        }    	        

        if ( $messageType == NULL || $messageType == SMTLinkTestUpdateMessageDto::getShortClassName() )
        {
        	$linkTestUpdateMessages = $this->getLinkTestUpdateMessages( $consumerId );
        	if ( $linkTestUpdateMessages != NULL )
        	{
        		for( $i=0; $i<count($linkTestUpdateMessages); $i++ )
        		{
        			$messages->addLinkTestUpdateMessage( $linkTestUpdateMessages[$i] );
        		}
        	}
        }
        
        if ( $messageType == NULL || $messageType == SMTTestOnDemandMessageDto::getShortClassName() )
        {
            $testOnDemandMessages = $this->getTestOnDemandMessages( $consumerId );
            if ( $testOnDemandMessages != NULL )
            {
            	for( $i=0; $i<count($testOnDemandMessages); $i++ )
            	{
            		$messages->addTestOnDemandMessage( $testOnDemandMessages[$i] );
            	}
            }            
        }	         
        
    	return $messages;
    }    

    /**
     * Whether The operation is registered in the messages listeners.
     *
     * @return boolean
     */
    private static function hasOperation( $operationId )
    {
        $message = SMTMemoryManager::fetch( SMTTestOnDemandEventConsumerDto::getClass(), $operationId );
        if ( $message != NULL )
        {
            return TRUE;
        }
        $message = SMTMemoryManager::fetch( SMTMeasureOnDemandEventConsumerDto::getClass(), $operationId );
        if ( $message != NULL )
        {
        	return TRUE;
        }
        
    	return FALSE;
    }    
    
    /**
     * Cleanup all conversations that are obsolete (not requested since 10 times the period of retrieval)
     * and the consumers attached to that conversation and not attached to another one.
     *
     * @throws Exception
     */
    public function purgeObsoleteConversations()
    {
        if ( SMTEventMessageManagerLockHandler::acquireConversationLock() )
        {
        	try
        	{
        	    $conversationMessageFilters = SMTMemoryManager::fetchAll( SMTConversationMessageFilterDto::getClass() );
        	    if ( $conversationMessageFilters != NULL && count( $conversationMessageFilters ) > 0 )
        	    {
        	        $consumersToDelete = array();
        	    	foreach ( $conversationMessageFilters as $conversationMessageFilter )
        	    	{
        	    		if ( $conversationMessageFilter != NULL && $conversationMessageFilter->isObsolete() )
        	    		{
        	    		    $conversationId = $conversationMessageFilter->getConversationId();
        	    		    $consumerId = $conversationMessageFilter->getConsumerId();
        	    		    
        	    		    //first remove the conversation
        	    		    SMTLogger::getInstance()->trace( "Purge obsolete conversation: ".$conversationId, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
        	    		    $conversationMessageFilter->delete();
        	    		    
        	    		    //if consumer is shared in another conversation don't delete it
        	    		    if ( in_array( $consumerId, $consumersToDelete ))
        	    		    {
        	    		        if( ( $key = array_search( $consumerId, $consumersToDelete) ) !== FALSE ) 
        	    		        {
        	    		        	unset( $consumersToDelete[$key]);
        	    		        }
        	    		    }   
        	    		    else
        	    		    {
        	    		        array_push( $consumersToDelete, $consumerId );
        	    		    }     	    		    
        	    		}
        	    	}
        	    	
        	    	foreach ( $consumersToDelete as $consumerId )
        	    	{
            	    	//clean up all the consumers attached to that obsolete conversation and not attached to another one
            	    	$this->unregisterActivityEventConsumer($consumerId);
            	    	$this->unregisterLinkTestUpdateEventConsumer($consumerId);
            	    	$this->unregisterOpticalAlarmEventConsumer($consumerId);
            	    	$this->unregisterSystemAlarmEventConsumer($consumerId);
            	    	$this->unregisterMeasureOndemandEventConsumer($consumerId);
            	    	$this->unregisterTestOndemandEventConsumer($consumerId);
        	    	}
        	    }
        	    SMTEventMessageManagerLockHandler::releaseConversationLock();
        	}
        	catch( \Exception $e )
        	{
        		SMTEventMessageManagerLockHandler::releaseConversationLock();
        		throw $e;
        	}        
        }
        else
        {
        	SMTLogger::getInstance()->trace("Couldn't acquire lock to purge Conversations ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
        }        
    }
    
    /**
     * Cleanup all activity event consumers that are obsolete (not requested since 5 times the period of retrieval).
     *
     * @throws Exception
     */
    public function purgeObsoleteActivityEventConsumers()
    {
    	if ( SMTEventMessageManagerLockHandler::acquireActivityLock() )
    	{
    		try
    		{
    			$eventConsumers = $this->fetchActivityEventConsumers();
    			if ( $eventConsumers != NULL )
    			{
    				foreach ( $eventConsumers as $eventConsumer )
    				{
    					if ( $eventConsumer != NULL && $eventConsumer->isObsolete() )
    					{
    					    SMTLogger::getInstance()->trace( "Purge obsolete activity events consumer: ".$eventConsumer->getId(), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    						SMTMemoryManager::delete( SMTActivityEventConsumerDto::getClass(), $eventConsumer->getId() );
    					}
    				}
    			}
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseActivityLock();
    			throw $e;
    		}
    
    		SMTEventMessageManagerLockHandler::releaseActivityLock();
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to purge activity event consumers", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}    	
    }       
    
    /**
     * Cleanup all link test update event consumers that are obsolete (not requested since 5 times the period of retrieval).
     *
     * @throws Exception
     */
    public function purgeObsoleteLinkTestUpdateEventConsumers()
    {
    	if ( SMTEventMessageManagerLockHandler::acquireLinkTestUpdateLock() )
    	{
    		try
    		{
    			$eventConsumers = $this->fetchLinkTestUpdateEventConsumers();
    			if ( $eventConsumers != NULL )
    			{
    				foreach ( $eventConsumers as $eventConsumer )
    				{
    					if ( $eventConsumer != NULL && $eventConsumer->isObsolete() )
    					{
    					    SMTLogger::getInstance()->trace( "Purge obsolete link test update events consumer: ".$eventConsumer->getId(), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    						SMTMemoryManager::delete( SMTLinkTestUpdateEventConsumerDto::getClass(), $eventConsumer->getId() );
    					}
    				}
    			}
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseLinkTestUpdateLock();
    			throw $e;
    		}
    
    		SMTEventMessageManagerLockHandler::releaseLinkTestUpdateLock();
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to purge link test update event consumers", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}
    }    
        
    /**
     * Cleanup all Optical alarm event consumers that are obsolete (not requested since 5 times the period of retrieval).
     *
     * @throws Exception
     */
    public function purgeObsoleteOpticalAlarmEventConsumers()
    {
    	if ( SMTEventMessageManagerLockHandler::acquireAlarmLock() )
    	{
    		try
    		{
    			$eventConsumers = $this->fetchOpticalAlarmEventConsumers();
    			if ( $eventConsumers != NULL )
    			{
    				foreach ( $eventConsumers as $eventConsumer )
    				{
    					if ( $eventConsumer != NULL && $eventConsumer->isObsolete() )
    					{
    					    SMTLogger::getInstance()->trace( "Purge obsolete optical alarm event consumer: ".$eventConsumer->getId(), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    						SMTMemoryManager::delete( SMTOpticalAlarmEventConsumerDto::getClass(), $eventConsumer->getId() );
    					}
    				}
    			}
    			SMTEventMessageManagerLockHandler::releaseAlarmLock();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseAlarmLock();
    			throw $e;
    		}    
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to purge Optical alarms event consumers ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}    	
    }    
        
    /**
     * Cleanup all System alarm event consumers that are obsolete (not requested since 5 times the period of retrieval).
     *
     * @throws Exception
     */
    public function purgeObsoleteSystemAlarmEventConsumers()
    {
    	if ( SMTEventMessageManagerLockHandler::acquireAlarmLock() )
    	{
    		try
    		{
    			$eventConsumers = $this->fetchSystemAlarmEventConsumers();
    			if ( $eventConsumers != NULL )
    			{
    				foreach ( $eventConsumers as $eventConsumer )
    				{
    					if ( $eventConsumer != NULL && $eventConsumer->isObsolete() )
    					{
    					    SMTLogger::getInstance()->trace( "Purge obsolete system alarm event consumer: ".$eventConsumer->getId(), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    						SMTMemoryManager::delete( SMTSystemAlarmEventConsumerDto::getClass(), $eventConsumer->getId() );
    					}
    				}
    			}
    			SMTEventMessageManagerLockHandler::releaseAlarmLock();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseAlarmLock();
    			throw $e;
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to purge System alarms event consumers ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}
    }    
    
    /**
     * Cleanup all test on demand operation status event consumers that are obsolete (not requested since 5 times the period of retrieval).
     * 
     * @throws Exception
     */
    public function purgeObsoleteTestOnDemandEventConsumers()
    {
        if ( SMTEventMessageManagerLockHandler::acquireTestOnDemandLock() )
        {
        	try
        	{
        		$operationEventConsumers = $this->fetchTestOnDemandEventConsumers();
        		if ( $operationEventConsumers != NULL )
        		{
        			foreach ( $operationEventConsumers as $operationEventConsumer )
        			{
        				if ( $operationEventConsumer != NULL && $operationEventConsumer->isObsolete() )
        				{
        				    SMTLogger::getInstance()->trace( "Purge obsolete test on demand event consumer: ".$operationEventConsumer->getId(), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
        					SMTMemoryManager::delete( SMTTestOnDemandEventConsumerDto::getClass(), $operationEventConsumer->getId() );
        				}
        			}
        		}
        		SMTEventMessageManagerLockHandler::releaseTestOnDemandLock();
        	}
        	catch( \Exception $e )
        	{
        		SMTEventMessageManagerLockHandler::releaseTestOnDemandLock();
        		throw $e;
        	}        	 
        }
        else
        {
        	SMTLogger::getInstance()->trace("Couldn't acquire lock to purge test on demand event consumers ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
        }        
    }
    
    /**
     * Cleanup all measure on demand operation status event consumers that are obsolete (not requested since 5 times the period of retrieval).
     *
     * @throws Exception
     */
    public function purgeObsoleteMeasureOnDemandEventConsumers()
    {
    	if ( SMTEventMessageManagerLockHandler::acquireMeasureOnDemandLock() )
    	{
    	    try
    	    {        	    	 
        	    $operationEventConsumers = $this->fetchMeasureOnDemandEventConsumers();
        	    if ( $operationEventConsumers != NULL )
        	    {
        	    	foreach ( $operationEventConsumers as $operationEventConsumer )
        	    	{
        	    		if ( $operationEventConsumer != NULL && $operationEventConsumer->isObsolete() )
        	    		{
        	    		    SMTLogger::getInstance()->trace( "Purge obsolete measure on demand event consumer: ".$operationEventConsumer->getId(), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
        	    		    SMTMemoryManager::delete( SMTMeasureOnDemandEventConsumerDto::getClass(), $operationEventConsumer->getId() );
        	    		}
        	    	}
        	    }
        	    SMTEventMessageManagerLockHandler::releaseMeasureOnDemandLock();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseMeasureOnDemandLock();
    			throw $e;
        	}        	
        }
        else
        {
        	SMTLogger::getInstance()->trace("Couldn't acquire lock to purge measure on demand event consumers ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
        }        
    }    

    /**
     * Fetch all the pending link test update event consumers for all consumers
     *
     * @throws Exception
     *
     * @return SMTActivityEventConsumerDto[]
     */
    private function fetchLinkTestUpdateEventConsumers()
    {
    	$eventConsumers = NULL;
    	if ( SMTEventMessageManagerLockHandler::acquireLinkTestUpdateLock() )
    	{
    		try
    		{
    			$eventConsumers = SMTMemoryManager::fetchAll( SMTLinkTestUpdateEventConsumerDto::getClass() );
    			SMTEventMessageManagerLockHandler::releaseLinkTestUpdateLock();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseLinkTestUpdateLock();
    			throw $e;
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to fetch link test update event consumers ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}
    	return $eventConsumers;
    }
    
    /**
     * Fetch all the pending activity event consumers for all consumers
     *
     * @throws Exception
     *
     * @return SMTActivityEventConsumerDto[]
     */
    private function fetchActivityEventConsumers()
    {
    	$eventConsumers = NULL;
    	if ( SMTEventMessageManagerLockHandler::acquireActivityLock() )
    	{
    		try
    		{
    			$eventConsumers = SMTMemoryManager::fetchAll( SMTActivityEventConsumerDto::getClass() );
    			SMTEventMessageManagerLockHandler::releaseActivityLock();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseActivityLock();
    			throw $e;
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to fetch activity event consumers ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}    	
    	return $eventConsumers;
    }
    
    /**
     * Fetch all the pending Optical alarm event consumers for all consumers
     *
     * @throws Exception
     *
     * @return SMTOpticalAlarmEventConsumerDto[]
     */
    private function fetchOpticalAlarmEventConsumers()
    {
    	$eventConsumers = NULL;
    	if ( SMTEventMessageManagerLockHandler::acquireAlarmLock() )
    	{
    		try
    		{
    			$eventConsumers = SMTMemoryManager::fetchAll( SMTOpticalAlarmEventConsumerDto::getClass() );
    			SMTEventMessageManagerLockHandler::releaseAlarmLock();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseAlarmLock();
    			throw $e;
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to fetch Optical alarm event consumers ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}    	
    	return $eventConsumers;
    }    
    
    /**
     * Fetch all the pending System alarm event consumers for all consumers
     *
     * @throws Exception
     *
     * @return SMTSystemAlarmEventConsumerDto[]
     */
    private function fetchSystemAlarmEventConsumers()
    {
    	$eventConsumers = NULL;
    	if ( SMTEventMessageManagerLockHandler::acquireAlarmLock() )
    	{
    		try
    		{
    			$eventConsumers = SMTMemoryManager::fetchAll( SMTSystemAlarmEventConsumerDto::getClass() );
    			SMTEventMessageManagerLockHandler::releaseAlarmLock();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseAlarmLock();
    			throw $e;
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to fetch System alarm event consumers ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}
    	return $eventConsumers;
    }    
    
    /**
     * Fetch all the pending test on demand event consumers for all consumers 
     * 
     * @throws Exception
     * 
     * @return SMTTestOnDemandEventConsumerDto[]
     */
    private function fetchTestOnDemandEventConsumers()
    {
        $eventConsumers = NULL;
        if ( SMTEventMessageManagerLockHandler::acquireTestOnDemandLock() )
        {
        	try
        	{
        		$eventConsumers = SMTMemoryManager::fetchAll( SMTTestOnDemandEventConsumerDto::getClass() );
        		SMTEventMessageManagerLockHandler::releaseTestOnDemandLock();
        	}
        	catch( \Exception $e )
        	{
        		SMTEventMessageManagerLockHandler::releaseTestOnDemandLock();
        		throw $e;
        	}
        }
        else
        {
        	SMTLogger::getInstance()->trace("Couldn't acquire lock to fetch test on demand event consumers ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
        }        
        return $eventConsumers;        
    }        
    
    /**
     * Fetch all the pending measure on demand event consumers for all consumers 
     * @throws Exception
     *
     * @return SMTMeasureOnDemandEventDto[]
     */
    private function fetchMeasureOnDemandEventConsumers()
    {
    	$eventConsumers = NULL;
    	if ( SMTEventMessageManagerLockHandler::acquireMeasureOnDemandLock() )
    	{
    		try
    		{
    			$eventConsumers = SMTMemoryManager::fetchAll( SMTMeasureOnDemandEventConsumerDto::getClass() );
    			SMTEventMessageManagerLockHandler::releaseMeasureOnDemandLock();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseMeasureOnDemandLock();
    			throw $e;
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to fetch measure on demand event consumers ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}    	
    	return $eventConsumers;
    }    
    
    /**
     * Get TestOnDemand message and remove TestOnDemand Event Consumer if operation ended
     * 
     * @param string $consumerId consumer id (operation Id)
     * @return SMTTestOnDemandEventConsumerDto
     */
    private function getTestOnDemandMessages( $consumerId )
    {
        $messages = array();
        if ( SMTEventMessageManagerLockHandler::acquireTestOnDemandLock() )
        {
            try 
            {
                /** @var $message SMTTestOnDemandEventConsumerDto */
                $eventConsumer = SMTMemoryManager::fetch( SMTTestOnDemandEventConsumerDto::getClass(), $consumerId );
                if ( $eventConsumer != NULL )
                {
                    $eventConsumer->touchAccess();
                    $hasOperationEnded = FALSE;
                	$events = $eventConsumer->getEvents();
                	for( $i=0; $i<count($events); $i++ )
                	{
                		$message = new SMTTestOnDemandMessageDto();
                		$message->setConsumerId($consumerId);
                		$message->setEvent( $events[$i]);
                		array_push( $messages, $message);
                		$hasOperationEnded = $events[$i]->hasOperationEnded();
                		
                		if ( $hasOperationEnded )
                		{
                			//operation has ended, stop loop
                			break;
                		}                		 
                	}
                	
                	if ( $hasOperationEnded )
                	{
                		//operation ended, remove the customer listener
                		$eventConsumer->delete();
                	}
                	else
                	{
                		$eventConsumer->clearEvents();
                		//commit events cleared and update last access timestamp
                		$eventConsumer->save();
                	}                	
                }                           
                
            	SMTEventMessageManagerLockHandler::releaseTestOnDemandLock();
            }
            catch( \Exception $e )
        	{
        		SMTEventMessageManagerLockHandler::releaseTestOnDemandLock();
        		throw $e;
        	}
        }
        return $messages;
    }
    /**
    * Get MeasureOnDemand message and remove MeasureOnDemand Event Consumer if operation ended
    * 
    * @param string $consumerId consumer id (operation Id)
    * 
    * @return SMTMeasureOnDemandEventConsumerDto
    */
    private function getMeasureOnDemandMessages( $consumerId )
    {
        $messages = array();
        if ( SMTEventMessageManagerLockHandler::acquireMeasureOnDemandLock() )
        {
            try
            {
                /** @var $message SMTMeasureOnDemandEventConsumerDto */                
                $eventConsumer = SMTMemoryManager::fetch( SMTMeasureOnDemandEventConsumerDto::getClass(), $consumerId );
                                
                if ( $eventConsumer != NULL )
                {
                    $eventConsumer->touchAccess();
                    //SMTLogger::getInstance()->trace( "SMTMeasureOnDemandEventConsumerDto".var_export( $eventConsumer, true), SMTLogger::DEBUG );
                    $events = $eventConsumer->getEvents();
                    if ( $events != NULL && count($events) > 0)
                    {
                        $hasOperationEnded = FALSE;
                        for( $i=0; $i<count($events); $i++ )
                        {
                        	$message = new SMTMeasureOnDemandMessageDto();
                        	$message->setConsumerId($consumerId);
                        	$message->setEvent( $events[$i]);
                        	array_push( $messages, $message);
                        	$hasOperationEnded = $events[$i]->hasOperationEnded(); 
                        	                    	
                        	if ( $hasOperationEnded )
                        	{
                        	    //operation has ended, stop loop
                        	    break;
                        	}
                        }              
    
                        if ( $hasOperationEnded )
                        {
                        	//operation ended, remove the customer listener
                        	$eventConsumer->delete();
                        }
                        else
                        {
                        	$eventConsumer->clearEvents();
                        	//commit events cleared and update last access timestamp
                        	$eventConsumer->save();
                        }
                    }
                    else
                    {
                        //commit last access timestamp update
                        $eventConsumer->save();
                    }                    
                }
                
                SMTEventMessageManagerLockHandler::releaseMeasureOnDemandLock();
            }
            catch( \Exception $e )
        	{
        	    SMTEventMessageManagerLockHandler::releaseMeasureOnDemandLock();
        	    throw $e;
            }
        }
        else
        {
        	SMTLogger::getInstance()->trace("Couldn't acquire lock to get measure on demand messages ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
        }        
        return $messages;
    }
        
    /**
     * Get Optical alarm messages and clear alarm EventConsumer
     * 
     * @param string $consumerId consumer Id
     * @return array of SMTOpticalAlarmMessageDto
     */
    private function getOpticalAlarmMessages( $consumerId )
    {
        $messages = array();
        if ( SMTEventMessageManagerLockHandler::acquireAlarmLock() )
        {
        	try
        	{
        		$eventConsumer = SMTMemoryManager::fetch( SMTOpticalAlarmEventConsumerDto::getClass(), $consumerId );
        		
        		//SMTLogger::getInstance()->trace( "SMTOpticalAlarmEventConsumerDto".var_export( $eventConsumer, true), SMTLogger::DEBUG );
        		if ( $eventConsumer != NULL )
        		{
        		    $eventConsumer->touchAccess();
        		    $events = $eventConsumer->getEvents();
        		    if ( $events != NULL && count($events) > 0)
        		    {
            		    for( $i=0; $i<count($events); $i++ )
            		    {
            		        $message = new SMTOpticalAlarmMessageDto();
            		        $message->setConsumerId($consumerId);
            		        $message->setEvent( $events[$i]);
            		        array_push( $messages, $message);
            		    }            		    
        				$eventConsumer->clearEvents();
        		    }
    				//commit events cleared and update last access timestamp
    				$eventConsumer->save();
        		}
        		SMTEventMessageManagerLockHandler::releaseAlarmLock();
        	}
        	catch( \Exception $e )
        	{
        		SMTEventMessageManagerLockHandler::releaseAlarmLock();
        		throw $e;
        	}
        }
        else
        {
        	SMTLogger::getInstance()->trace("Couldn't acquire lock to get optical alarm messages ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
        }        
        return $messages;        
    }

    /**
     * * Get System alarm messages and clear alarm EventConsumer
     * 
     * @param string $consumerId consumer Id
     * @return array of SMTSystemAlarmMessageDto
     */
    private function getSystemAlarmMessages( $consumerId )
    {
    	$messages = array();
    	if ( SMTEventMessageManagerLockHandler::acquireAlarmLock() )
    	{
    		try
    		{
    			$eventConsumer = SMTMemoryManager::fetch( SMTSystemAlarmEventConsumerDto::getClass(), $consumerId );    			    			
    			if ( $eventConsumer != NULL )
    			{
    			    $eventConsumer->touchAccess();
    				$events = $eventConsumer->getEvents();
    				if ( $events != NULL && count($events) > 0)
    				{
        				for( $i=0; $i<count($events); $i++ )
        				{
        					$message = new SMTSystemAlarmMessageDto();
        					$message->setConsumerId($consumerId);
        					$message->setEvent( $events[$i]);
        					array_push( $messages, $message);
        				}
        				
        				$eventConsumer->clearEvents();
    				}
    				//commit events cleared and update last access timestamp
    				$eventConsumer->save();
    			}
    			SMTEventMessageManagerLockHandler::releaseAlarmLock();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseAlarmLock();
    			throw $e;
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to get System alarm messages ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}
    	return $messages;
    }    
    
    /**
     * Get activity messages and clear Activity EventConsumer
     * 
     * @param string $consumerId consumer Id
     * @return array of SMTActivityMessageDto
     */    
    private function getActivityMessages( $consumerId )
    {
        $messages = array();
        if ( SMTEventMessageManagerLockHandler::acquireActivityLock() )
        {
        	try
        	{
        		$eventConsumer = SMTMemoryManager::fetch( SMTActivityEventConsumerDto::getClass(), $consumerId );        		
//         		SMTLogger::getInstance()->trace( "SMTActivityEventConsumerDto".var_export( $eventConsumer, true), SMTLogger::DEBUG );
        		if ( $eventConsumer != NULL )
        		{
        		    $eventConsumer->touchAccess();
        			$events = $eventConsumer->getEvents();
        			if ( $events != NULL && count($events) > 0)
        			{
            			for( $i=0; $i<count($events); $i++ )
            			{
            				$message = new SMTActivityMessageDto();
            				$message->setConsumerId($consumerId);
            				$message->setEvent( $events[$i]);
            				array_push( $messages, $message);
            			}
            		
            			$eventConsumer->clearEvents();
        			}
        			//commit events cleared and update last access timestamp 
        			$eventConsumer->save();
        		}        		
        		
        		SMTEventMessageManagerLockHandler::releaseActivityLock();
        	}
        	catch( \Exception $e )
        	{
        		SMTEventMessageManagerLockHandler::releaseActivityLock();
        		throw $e;
        	}
        }
        else
        {
        	SMTLogger::getInstance()->trace("Couldn't acquire lock to get activity messages ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
        }        
        return $messages;        
    }
    
    /**
     * Get link test update messages and clear link test update EventConsumer
     *
     * @param string $consumerId consumer Id
     * @return array of SMTLinkTestUpdateMessageDto
     */
    private function getLinkTestUpdateMessages( $consumerId )
    {
    	$messages = array();
    	if ( SMTEventMessageManagerLockHandler::acquireLinkTestUpdateLock() )
    	{
    		try
    		{
    			$eventConsumer = SMTMemoryManager::fetch( SMTLinkTestUpdateEventConsumerDto::getClass(), $consumerId );
    			if ( $eventConsumer != NULL )
    			{
    			    $eventConsumer->touchAccess();
    				$events = $eventConsumer->getEvents();
    				if ( $events != NULL && count($events) > 0)
    				{
        				for( $i=0; $i<count($events); $i++ )
        				{
        					$message = new SMTLinkTestUpdateMessageDto();
        					$message->setConsumerId($consumerId);
        					$message->setEvent( $events[$i]);
        					array_push( $messages, $message);
        				}
        
        				$eventConsumer->clearEvents();
    				}
    				//commit events cleared and update last access timestamp
    				$eventConsumer->save();
    			}
    
    			SMTEventMessageManagerLockHandler::releaseLinkTestUpdateLock();
    		}
    		catch( \Exception $e )
    		{
    			SMTEventMessageManagerLockHandler::releaseLinkTestUpdateLock();
    			throw $e;
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock to get linkTest update messages ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}
    	return $messages;
    }       
}

?>

