import sys
import dbus
import dbus.service
import dbus.mainloop.glib
import threading
import signal
import subprocess
import os
import bz2
import base64
import traceback
import linecache
import sys
import glob
import syslog
from gi.repository import GLib
from jdsu.mts.blockutils import mount, umount, mount_bind, mounted_point
from jdsu.mts.flashutils import mount_chroot, umount_chroot, chroot_exec, format_device
from jdsu.mts.bootutils import bootenv
from lxml import etree
from io import IOBase, StringIO

if sys.version_info[0] < 3:
    from urllib2 import ProxyHandler, build_opener, install_opener, urlopen, Request, HTTPError, URLError
    from ConfigParser import ConfigParser
    from Queue import Queue
else:
    from urllib.request import Request, urlopen, ProxyHandler, build_opener, install_opener
    from urllib.error import HTTPError, URLError
    from configparser import ConfigParser
    from queue import Queue
    file = IOBase

# ________________________TAR_MANAGMENT__________________________
import shutil
import tarfile

def split_url(url):
    u = url.split(":")
    index = int(u[0])
    url = ":".join(u[1:])
    return (url, index)


def synchronize(source, destination, overwrite=False):
    print("SYNC %s %s" % (source, destination))
    if os.path.isdir(source):
        if os.path.isdir(destination):
            for item in os.listdir(source):
                s = os.path.join(source, item)
                d = os.path.join(destination, item)
                synchronize(s, d)
        else:
            shutil.copytree(source, destination, symlinks=True)
    else:
        if not os.path.lexists(destination) or overwrite:
            if os.path.islink(source):
                print("COPY LINK %s" % source)
                linkto = os.readlink(source)
                os.symlink(linkto, destination)
            else:
                if not os.path.exists(os.path.dirname(destination)):
                    os.makedirs(os.path.dirname(destination))
                if os.path.exists(source):
                    shutil.copy2(source, destination)


class MyFileObj(file):
    def setCls(self, cls, fsize, name):
        self.__cls = cls
        self.__fsize = fsize
        self.__rname = name
        self.lastpercent = 1000

    def read(self, size):
        percent = self.tell() * 100 / self.__fsize
        if percent != self.lastpercent:
            self.__cls.expose("Upgrade", 66, self.__rname, "Installation", percent)
            self.lastpercent = percent
        return file.read(self, size)


def download_file(src, dest, proxy, chunk_size=8192, report_hook=None):
    with open(dest, "wb") as destination:
        if proxy is not None and proxy.strip() != "DIRECT":
            print("Enable proxy %s" % proxy)
            proxy_handler = ProxyHandler({'http': proxy})
        else:
            print("Disable proxy")
            proxy_handler = ProxyHandler({})
        opener = build_opener(proxy_handler)
        install_opener(opener)
        response = urlopen(src);
        total_size = response.info().get('Content-Length').strip()
        total_size = int(total_size)
        bytes_so_far = 0

        while 1:
            chunk = response.read(chunk_size)
            bytes_so_far += len(chunk)
            destination.write(chunk)

            if not chunk:
                break

            if report_hook:
                report_hook(bytes_so_far, chunk_size, total_size)

    return bytes_so_far

















class ReleaseXML(object):
    def __init__(self, release_file, url):
        self.__root = etree.parse(release_file)
        self.__url = url
        if url is not None:
            (self.__release, self.__slot) = split_url(url)
            self.__root.xpath("/root")[0].attrib['url'] = url
        else:
            url = self.__root.xpath("/root/@url")[0]
            (self.__release, self.__slot) = split_url(url)

    def xpath(self, path):
        return self.__root.xpath("/root/release/%s"%path)

    def url(self):
        return self.__url

    def save(self, fname):
        with open(fname, "w") as f:
            f.write(etree.tostring(self.__root))

    def serialize(self):
        return (etree.tostring(self.__root))

    def remote(self):
        (release, slot) = split_url(self.__url)
        return ("http://" in release)


def get_release_xml(url, proxy):
    (release, slot) = split_url(url)
    release_file = None
    valid = True
    if "http://" in release:
        # Download XML file
        if proxy is not None and proxy.strip() != "DIRECT":
            print("Enable proxy %s" % proxy)
            proxy_handler = ProxyHandler({'http': proxy})
        else:
            print("Disable proxy")
            proxy_handler = ProxyHandler({})
        opener = build_opener(proxy_handler)
        install_opener(opener)

        # Check mymirror.php
        base_url = release.replace("http://", "").split("/")[0]
        mymirror_url = "http://%s/mymirror.php" % (base_url)
        req = Request(mymirror_url)
        try:
            mirror = urlopen(req).read()
            release = release.replace(base_url, mirror)
            url = url.replace(base_url, mirror)
            print("Release with mirror: %s" % release)
        except HTTPError:
            pass

        release_file = "/var/volatile/release.xml"
        req = Request(release)
        try:
            with open(release_file, "wb") as local_file:
                f = urlopen(req)
                local_file.write(f.read())
        except HTTPError as e:
            print("HTTP Error:%s %s"%(e.code, release))
            PrintException()
            return None, None
        except URLError as e:
            print("URL Error: %s %s"%(e.reason, release))
            PrintException()
            return None, None

        # Download signature
        if not os.path.exists("/etc/release/no_check"):
            req = Request(release + ".sig")
            try:
                with open(release_file + ".sig", "wb") as local_file:
                    f = urlopen(req)
                    local_file.write(f.read())
            except HTTPError as e:
                print("HTTP Error:%s %s"%(e.code, release))
                PrintException()
                valid = False
            except URLError as e:
                print("URL Error:%s %s"%(e.reason, release))
                PrintException()
                valid = False
    else:  # USB Media
        release_file = release

    # Check signature
    if not os.path.exists("/etc/release/no_check"):
        try:
            print(subprocess.check_output(["gpg", "--verify", release_file + ".sig", release_file]))
        except Exception as e:
            print("BAD SIGNATURE %s" % str(e))
            PrintException()
            valid = False
    xmlrelease = ReleaseXML(release_file, url)
    if valid:
        try:
            hook = get_hooks(xmlrelease)
            run_hook(hook, 'inspect_hook')
        except Exception:
            pass
    return (xmlrelease, valid)


def current_release_xml():
    if not os.path.isfile("/etc/release/current.xml"):
        return None
    return ReleaseXML("/etc/release/current.xml", None)


def get_hooks(xmlrelease):
    h = None
    hooks = xmlrelease.xpath("hooks")
    if len(hooks) > 0:
        hk = bz2.decompress(base64.b64decode(hooks[0].text.encode("ASCII")))
        os.system("rm -fr /tmp/hook")
        os.system("mkdir /tmp/hook")
        sys.path.append("/tmp/hook")
        out = open("/tmp/hook/hook.py", "w")
        out.write(hk.decode("ASCII"))
        out.close()
        h = __import__("hook")
    return h


def run_hook(hook, name):
    print("Run hook %s" % name)
    try:
        if name in list(hook.__dict__.keys()):
            function = hook.__dict__[name]
            return function()
    except Exception as err:
        PrintException()
        return False
    print("No hook %s exists" % name)
    return True


def check_release(url, proxy):
    res = False
    (xmlrelease, valid) = get_release_xml(url, proxy)
    if not valid:
        return False
    try:
        hook = get_hooks(xmlrelease)
        res = run_hook(hook, 'pre_check')
    except Exception as err:
        print("Check release failed")
        PrintException()
    return res


def show_diff(url, proxy):
    def padd(v):
        t = v.split(".")
        t[0] = "%05i" % int(t[0])
        return ".".join(t)

    tmp = []
    current_xml = current_release_xml()
    (dist_xml, valid) = get_release_xml(url, proxy)

    if current_xml is None:
        for dist_item in dist_xml.xpath("content/item"):
            d = dist_item.attrib["date"].split("-")
            d.reverse()
            tmp.append("ISU: RPM     : \"%-20s\" > %15s %s" % (
                dist_item.attrib["name"], dist_item.attrib["version"].replace("V", ""), "/".join(d)))
        for dist_provide in dist_xml.xpath("content/provide"):
            d = dist_provide.attrib["date"].split("-")
            d.reverse()
            tmp.append("ISU: RELEASE : \"0\" ! %15s %s" % (dist_provide.attrib["version"].replace("V", ""), "/".join(d)))

    else:
        for dist_item in dist_xml.xpath("content/item"):
            ritem = current_xml.xpath("content/item[@name='%s']" % dist_item.attrib["name"])
            d = dist_item.attrib["date"].split("-")
            d.reverse()
            if len(ritem) == 0:
                tmp.append("ISU: RPM     : \"%-20s\" > %15s %s" % (
                    dist_item.attrib["name"], dist_item.attrib["version"].replace("V", ""), "/".join(d)))
                continue
            current_item = ritem[0]
            if padd(current_item.attrib["version"].replace("V", "")) > padd(dist_item.attrib["version"].replace("V", "")):
                tmp.append("ISU: RPM     : \"%-20s\" < %15s %s" % (
                    dist_item.attrib["name"], dist_item.attrib["version"].replace("V", ""), "/".join(d)))
            elif padd(current_item.attrib["version"].replace("V", "")) < padd(dist_item.attrib["version"].replace("V", "")):
                tmp.append("ISU: RPM     : \"%-20s\" > %15s %s" % (
                    dist_item.attrib["name"], dist_item.attrib["version"].replace("V", ""), "/".join(d)))
            elif current_item.attrib["date"] > dist_item.attrib["date"]:
                tmp.append("ISU: RPM     : \"%-20s\" < %15s %s" % (
                    dist_item.attrib["name"], dist_item.attrib["version"].replace("V", ""), "/".join(d)))
            elif current_item.attrib["date"] < dist_item.attrib["date"]:
                tmp.append("ISU: RPM     : \"%-20s\" > %15s %s" % (
                    dist_item.attrib["name"], dist_item.attrib["version"].replace("V", ""), "/".join(d)))
            else:
                tmp.append("ISU: RPM     : \"%-20s\" = %15s %s" % (
                    dist_item.attrib["name"], dist_item.attrib["version"].replace("V", ""), "/".join(d)))

        for current_provide in current_xml.xpath("content/provide"):
            dist_provide = dist_xml.xpath("content/provide[@name='%s']" % current_provide.attrib["name"])[0]
            d = dist_provide.attrib["date"].split("-")
            d.reverse()
            if int(current_provide.attrib["version"]) != int(dist_provide.attrib["version"]):
                tmp.append("ISU: RELEASE : \"%-20s\" ! %15s %s" % (
                    dist_provide.attrib["name"], dist_provide.attrib["version"].replace("V", ""), "/".join(d)))
            else:
                tmp.append("ISU: RELEASE : \"%-20s\" = %15s %s" % (
                    dist_provide.attrib["name"], dist_provide.attrib["version"].replace("V", ""), "/".join(d)))
    if not valid:
        tmp.append("ISU: RPM     : \"Corrupted Media !!!\"")

    order = ["Boo", "Lin", "Fil", "Ins", "Mic", "Fib"]
    order_list = []
    for i in order:
        item = None
        for s in tmp:
            if s.startswith("ISU: RPM     : \"%s" % i):
                item = s
                break
        if item is not None:
            order_list.append(item)
            tmp.remove(item)
    for rest in tmp:
        order_list.append(rest)
    return "\n".join(order_list)


def show_highlighted(url, proxy):
    if url == "current":
        xml = current_release_xml()
        valid = True
    else:
        (xml, valid) = get_release_xml(url, proxy)
    tmp = []
    if not valid:
        return "ERROR"

    for item in xml.xpath("content/item"):
        d = item.attrib["date"].split("-")
        d.reverse()
        tmp.append("ISU: RPM     : \"%-20s\" = %15s %s" % (
            item.attrib["name"], item.attrib["version"].replace("V", ""), "/".join(d)))

    for provide in xml.xpath("content/provide"):
        d = provide.attrib["date"].split("-")
        d.reverse()
        tmp.append("ISU: RELEASE : \"%-20s\" - %15s %s" % (
            provide.attrib["name"], provide.attrib["version"].replace("V", ""), "/".join(d)))
    tmp.append("OK")
    return "\n".join(tmp)




def PrintException():
    exc_type, exc_obj, tb = sys.exc_info()
    f = tb.tb_frame
    lineno = tb.tb_lineno
    filename = f.f_code.co_filename
    linecache.checkcache(filename)
    line = linecache.getline(filename, lineno, f.f_globals)
    print('EXCEPTION IN ({}, LINE {} "{}"): {}'.format(filename, lineno, line.strip(), exc_obj))
    traceback.print_exception(exc_type, exc_obj, tb, limit=5, file=sys.stdout)




class SmartManager(dbus.service.Object):
    version = 1.0

    def __init__(self, daemon=False, loop=900000, debug=False):
        self.__daemon = daemon
        self.__workqueue = Queue()
        self.__proxy = None
        self.__last_state = "NOP"
        # default values for configuration
        self.__use_background_mode = False
        self.__auto_switch = True
        self.__test_update_part = False
        self.__parts = [5, 6]
        # No smart library in python3
        if sys.version_info[0] < 3:
            self.upgrade_way = [self.__upgrade_by_rpm, self.__upgrade_by_tar]
        else:
            self.upgrade_way = [self.__upgrade_by_tar]
        ifa = "text"
        if os.path.exists('/etc/releasemanager.conf'):
            config = ConfigParser()
            config.read('/etc/releasemanager.conf')
            if config.has_option('GENERAL', 'parts'):
                parts = config.get('GENERAL', 'parts')
                self.__parts = map(int, parts.strip().split(','))
            if config.has_option('GENERAL', 'background_mode'):
                self.__use_background_mode = config.getboolean('GENERAL', 'background_mode')
            if config.has_option('GENERAL', 'auto_switch_part'):
                self.__auto_switch = config.getboolean('GENERAL', 'auto_switch_part')
            if config.has_option('GENERAL', 'test_update_part'):
                self.__test_update_part = config.getboolean('GENERAL', 'test_update_part')
            print("Config read : ")
            print(self.__parts)
            print(self.__use_background_mode)
            print(self.__auto_switch)
            print(self.__test_update_part)
        if self.__daemon:
            dbus.mainloop.glib.DBusGMainLoop(set_as_default=True)
            bus_name = dbus.service.BusName('org.jdsu.releasemanager', bus=dbus.SystemBus())
            dbus.service.Object.__init__(self, bus_name, "/org/jdsu/releasemanager")
            GLib.threads_init()
            self.__mainloop = GLib.MainLoop()
            with open("/proc/cmdline", "r") as f:
                bootargs = f.readline().split(' ')
                rootpart = [arg for arg in bootargs if arg.startswith("root=")]
                if len(rootpart) < 1:
                    print("Cannot determine root partition, fatal error, exiting")
                    exit(-1)
                rootpart = rootpart[0].split("=")[1]
                self.__flash_block_device = rootpart[:rootpart.rindex("p")]
                print("Root part is %s" % self.__flash_block_device)

        self.__release_monitoring = None
        self.__updated = False
        self.__timeout = loop
        if self.__daemon:
            self.__thread = threading.Thread(target=self.__worker)
            self.__thread.start()
            print("Daemon started\n")
        if not os.path.isdir("/var/lib/release-manager"):
            os.mkdir("/var/lib/release-manager")
        if os.path.isfile("/var/lib/release-manager/upgrade_status"):
            with open("/var/lib/release-manager/upgrade_status", "r") as f:
                if f.readline().strip() == "Success":
                    with open("/tmp/upgrade_success", "w") as t:
                        t.write("OK")
                else:
                    with open("/tmp/upgrade_failure", "w") as t:
                        t.write("FAIL")
            os.remove("/var/lib/release-manager/upgrade_status")

    def __worker(self):
        next_job = self.__workqueue.get()
        while type(next_job[0]) != str and next_job[0] != "EXIT":
            try:
                print("Worker: %s Start with param=%s" % (next_job[0].__name__, next_job[1]))
                next_job[0](**next_job[1])
                print("Worker: %s End" % next_job[0].__name__)
            except Exception as err:
                print("ERROR: %s" % str(err))
                PrintException()
            finally:
                self.__workqueue.task_done()
                next_job = self.__workqueue.get()

    def __del__(self):
        self.stop_daemon()
        if self.__daemon:
            print("Daemon Stoped")

    @dbus.service.signal(dbus_interface='org.jdsu.releasemanager', signature='s')
    def new_release(self, release):
        print("New release: %s" % str(release))

    def __check_new_release(self):
        if self.__release_monitoring is not None:
            oldxml = current_release_xml()
            (newxml, valid) = get_release_xml(self.__release_monitoring, self.__proxy)

            old_name = oldxml.xpath("@name")[0]
            new_name = newxml.xpath("@name")[0]
            print("Check release %s ?= %s" % (old_name, new_name))

            if old_name != new_name:
                self.new_release(new_name)

    def __timer(self):
        print("Timer")
        if self.__release_monitoring is not None:
            self.__workqueue.put((self.__check_new_release, {}))
            GLib.timeout_add(self.__timeout, self.__timer)

    def __set_release_monitoring(self, url):
        if len(url) > 0:
            if check_release(url, self.__proxy):
                print("Enable release monitoring on %s" % url)
                self.__release_monitoring = url
                self.__check_new_release()
                GLib.timeout_add(self.__timeout, self.__timer)
        else:
            print("Disable release monitoring")
            self.__release_monitoring = None

    @dbus.service.method(dbus_interface='org.jdsu.releasemanager', in_signature='s')
    def set_release_monitoring(self, url):
        print("CALL release-manager set_release_monitoring %s" % url)
        self.__set_release_monitoring(url)

    def __run_task(self, task, param={}):
        if self.__daemon:
            print("Enqueue %s with param=%s" % (task.__name__, param))
            self.__workqueue.put((task, param))
        else:
            print("%s Start with param=%s" % (task.__name__, param))
            task(**param)
            print("%s End" % task.__name__)

    def start_daemon(self):
        signal.signal(signal.SIGTERM, self.stop_daemon)
        try:
            self.__mainloop.run()
        except:
            self.stop_daemon()

    def stop_daemon(self):
        if self.__daemon:
            self.__workqueue.put(("EXIT"))
            self.__thread.join()
            self.__mainloop.quit()

    @dbus.service.signal(dbus_interface='org.jdsu.releasemanager', signature='s')
    def log(self, line):
        print(line)

    @dbus.service.signal(dbus_interface='org.jdsu.releasemanager', signature='s')
    def set_action(self, line):
        print(line)

    @dbus.service.signal(dbus_interface='org.jdsu.releasemanager', signature='sissi')
    def expose(self, topic, percent, subkey, subtopic, subpercent):
        print("%s %i %s %s %i" % (topic, percent, subkey, subtopic, subpercent))


    def __upgrade_by_rpm(self, xmlrelease, hook):
        import smart
        import smart.transaction
        import smart.const
        current_xml = current_release_xml()
        if current_xml:
            old_name = current_xml.xpath("@name")[0]
        else:
            old_name = "000000"
        new_name = xmlrelease.xpath("@name")[0]
        if old_name > new_name:
            print("This is a downgrade from %s to %s" % (old_name, new_name))
            upgrade = False
        else:
            print("This is an upgrade from %s to %s" % (old_name, new_name))
            upgrade = True
        if not (upgrade or run_hook(hook, 'rpm_downgrade')):
            self.set_action("Downgrade forbiden by rpm")
            return False
        if not run_hook(hook, 'pre_rpm_upgrade'):
            return False
        try:
            self.set_action("0: Set Release")
            if self.__daemon:
                smartobject = smart.init(command="channel", shell=False, interface="dbusiface")
            else:
                smartobject = smart.init(command="channel", shell=False)
            setattr(smartobject, "manager", self)
            smart.initDistro(smartobject)
            smart.initPlugins()
            smart.initPycurl()
            smart.initPsyco()
            url = xmlrelease.url()
            (release, slot) = split_url(url)
            rootpath = "/".join(release.split("/")[:-1])
            name = xmlrelease.xpath("@name")[0]
            id = xmlrelease.xpath("@id")[0]
            print("Set release %s" % (name))
            channels_deployed = smart.sysconf.keys("channels")
            if "rpmsys" not in channels_deployed:
                chan = smart.channel.parseChannelData({'type': 'rpm-sys'})
                smart.sysconf.set(("channels", "rpmsys"), chan)

            for obsolete in xmlrelease.xpath("repositories/obsolete"):
                alias = obsolete.attrib["alias"]
                smart.sysconf.remove("channels", alias)

            for repo in xmlrelease.xpath("repositories/rpm"):
                rid = repo.attrib["id"]
                rtype = repo.attrib["type"]
                rpath = repo.text
                uid = "%s-%s_%s" % (id, name, rid)
                upath = "%s/%s" % (rootpath, rpath)
                alias = "%s-%s" % (id, rid)
                if alias in channels_deployed:
                    smart.sysconf.remove("channels", alias)
                chan = smart.channel.parseChannelData(
                    {'type': 'rpm-md', 'baseurl': str(upath), 'name': uid, 'priority': 100})
                if not os.path.exists("/etc/release/no_check"):
                    chan["fingerprint"] = "624FC3EB316F2B2BCCD63AE68DBA543033386F18"

                smart.channel.createChannel(alias, chan)
                smart.sysconf.set(("channels", alias), chan)

            xmlrelease.save("/etc/release/current_set.xml")
            smartobject.saveSysConf()
            if self.__daemon:
                self.set_release_terminate(name)

            self.set_action("1: Update packages cache")
            smartobject_init()
            smartobject.rebuildSysConfChannels()
            smartobject.reloadChannels()
            failed = not smartobject.reloadChannels(None, caching=smart.const.NEVER)
            cache = smartobject.getCache()
            smartobject.saveSysConf()
            run_hook(hook, 'post_update')
            run_hook(hook, 'pre_upgrade')
            self.set_action("2: Upgrade packages cache")
            smartobject_init()
            try:
                cache = smartobject.getCache()
                trans = smart.transaction.Transaction(cache, smart.transaction.PolicyUpgrade)
                for pkg in cache.getPackages():
                    if pkg.installed:
                        trans.enqueue(pkg, smart.transaction.UPGRADE)
                trans.run()
                smartobject.commitTransaction(trans, confirm=False)
                smartobject.saveSysConf()
                res = True
            except Exception as err:
                print(str(err))
                return False

            if not smartobject_upgrade():
                return False
            run_hook(hook, 'post_upgrade')
            self.set_action("Upgrade Terminate")
        except Exception as err:
            PrintException()
            print("ABORT: %s" % err)
            smartobject.saveSysConf()
            smartobject.restoreMediaState()
            smartobject._pathlocks.unlockAll()
            del (smartobject)
            smartobject = None
            smart.deinit()
            print("Stop Smart")
            return False
        id = xmlrelease.xpath("@id")[0]
        xmlrelease.save("/etc/release/%s.xml" % id)
        os.system("rm -f /etc/release/current.xml")
        os.system("ln -s /etc/release/%s.xml /etc/release/current.xml" % id)
        smartobject.saveSysConf()
        smartobject.restoreMediaState()
        smartobject._pathlocks.unlockAll()
        del (smartobject)
        smartobject = None
        smart.deinit()
        print("Stop Smart")
        if not run_hook(hook, 'post_rpm_upgrade'):
            return False
        return True

    # End of SMART API
    ###

    def __upgrade_failed(self):
        if os.path.exists("/tmp/Disable_OnOff_Button"):
            os.remove("/tmp/Disable_OnOff_Button")
        with open("/var/lib/release-manager/upgrade_status", "w") as f:
            f.write("Failed")
        self.start_upgrades_terminate("FAIL")

    def __start_upgrades(self, url):
        self.__last_state = "UPGRADING"

        (xmlrelease, valid) = get_release_xml(url, self.__proxy)
        if not valid:
            self.__upgrade_failed()
            return

        with open("/tmp/Disable_OnOff_Button", "w") as f:
            f.write("LOCK")
        with open("/var/lib/release-manager/upgrade_status", "w") as f:
            f.write("Started")
        with open("/var/volatile/upgrade_in_progress", "w") as f:
            f.write("lock")

        hook = get_hooks(xmlrelease)

        if not run_hook(hook, 'pre_process'):
            self.__upgrade_failed()
            return

        # in background mode, do not change runlevel because apps are still working
        if not self.__use_background_mode:
            os.system("init 4; sleep 5")

        error = True
        for upgrade_method in self.upgrade_way:
            print("Try upgrade with %s"%upgrade_method.__name__)
            if upgrade_method(xmlrelease, hook):
                error = False
                break
            else:
                print("\t%s failed, Try next method"%upgrade_method.__name__)

        if not run_hook(hook, 'post_process'):
            error = True

        os.remove("/tmp/Disable_OnOff_Button")
        os.remove("/var/volatile/upgrade_in_progress")
        if error:
            self.start_upgrades_terminate("FAIL")
            with open("/var/lib/release-manager/upgrade_status", "w") as f:
                f.write("Failed")
        else:
            self.start_upgrades_terminate("PASS")
            with open("/var/lib/release-manager/upgrade_status", "w") as f:
                f.write("Success")

    @dbus.service.signal(dbus_interface='org.jdsu.releasemanager', signature='s')
    def start_upgrades_terminate(self, state):
        self.__last_state = state
        print("Upgrade process terminate: %s" % state)

    @dbus.service.method('org.jdsu.releasemanager', in_signature='s')
    def start_upgrade(self, url):
        print("CALL release-manager start_upgrade %s" % url)
        self.__run_task(self.__start_upgrades, {'url': url})

    @dbus.service.signal(dbus_interface='org.jdsu.releasemanager', signature='s')
    def set_release_terminate(self, name):
        print("Release changed to %s" % name)

    @dbus.service.method('org.jdsu.releasemanager', out_signature='b')
    def check_new_release(self):
        print("CALL release-manager check_new_release")
        return self.__check_new_release()

    @dbus.service.method('org.jdsu.releasemanager', out_signature='s')
    def get_release_info(self):
        print("CALL release-manager get_release_info")
        releasexml = current_release_xml()
        if releasexml is None:
            return "<release />"
        return releasexml.serialize()

    @dbus.service.method('org.jdsu.releasemanager', out_signature='s')
    def get_last_upgrade_state(self):
        print("CALL release-manager get_last_upgrade_state")
        return self.__last_state

    def __upgrade_by_tar(self, xmlrelease, hook):
        if not run_hook(hook, 'pre_tar_upgrade'):
            return False
        result = True
        tarf = xmlrelease.xpath("repositories/archive/@file")[0]
        name = xmlrelease.xpath("@name")[0]
        url = xmlrelease.url()
        remote = xmlrelease.remote()
        (xmlpath, slot) = split_url(url)
        path = "/".join(xmlpath.split("/")[:-1]) + "/" + tarf
        if remote:
            try:
                subprocess.check_output(["mount", "-t", "tmpfs", "-o", "size=256M", "tmpfs", "/mnt/tmp"])
            except:
                PrintException()
                return False
            lpath = "/mnt/tmp"
        else:
            lpath = "/".join(xmlpath.split("/")[:-1]) + "/"

        def chunk_report(bytes_so_far, chunk_size, total_size):
            percent = float(bytes_so_far) / total_size
            percent = round(percent * 100, 2)
            if int(percent) != int(chunk_report.lastpercent):
                self.expose("Upgrade", 33, name, "Download release", percent)
                chunk_report.lastpercent = percent

        chunk_report.lastpercent = 1000
        run_hook(hook, 'pre_tar_download')

        if remote:
            self.set_action("1: Download release")
            self.log("Download %s" % path)
            download_file(path, "%s/%s" % (lpath, tarf), self.__proxy, report_hook=chunk_report)

        self.set_action("2: Check release")
        try:
            if not os.path.exists("/etc/release/no_check"):
                if remote:
                    self.log("Download %s.sig" % path)
                    download_file(path + ".sig", "%s/%s.sig" % (lpath, tarf), self.__proxy, report_hook=chunk_report)

                self.expose("Upgrade", 50, name, "Check integrity", 0)
                print(
                    subprocess.check_output(["gpg", "--verify", "%s/%s.sig" % (lpath, tarf), "%s/%s" % (lpath, tarf)]))
                self.expose("Upgrade", 50, name, "Check integrity", 100)
        except subprocess.CalledProcessError as err:
            print("BAD SIGNATURE")
            if self.__daemon:
                self.upgrade_by_tar_terminate("BAD SIGNATURE")
            return
        try:
            try:
                bootenv.refresh()
                mmc_part = int(bootenv.mmc_part)
                if mmc_part == self.__parts[0]:
                    mmc_part_new = self.__parts[1]
                else:
                    mmc_part_new = self.__parts[0]
            except:
                mmc_part_new = self.__parts[1]

            print("mmc_part is %s" % mmc_part_new)

            # Unmount the partition about to be formatted (all mounts)
            for mount_point in mounted_point(self.__flash_block_device + "p%s" % mmc_part_new):
                umount(mount_point)

            run_hook(hook, 'pre_tar_install')
            self.set_action("3: Format partitions")
            if not os.path.exists("/mnt/fs"):
                os.makedirs("/mnt/fs")
            self.expose("Upgrade", 60, name, "Prepare filesystem %s" % mmc_part_new, 10)
            format_device(mmc_part_new)
            self.expose("Upgrade", 60, name, "Prepare filesystem", 50)
            mount(self.__flash_block_device + "p%s" % mmc_part_new, "/mnt/fs/", "ext4")
            self.expose("Upgrade", 60, name, "Prepare filesystem", 60)
            if not os.path.exists("/mnt/fs/user/disk"):
                os.makedirs("/mnt/fs/user/disk")
            self.expose("Upgrade", 60, name, "Prepare filesystem", 70)
            mount_bind("/user/disk", "/mnt/fs/user/disk")
            self.expose("Upgrade", 60, name, "Prepare filesystem", 100)
            self.set_action("4: Install release")

            run_hook(hook, 'pre_tar_untar')
            total_size = os.path.getsize("%s/%s" % (lpath, tarf))
            with MyFileObj("%s/%s" % (lpath, tarf)) as fileobj:
                fileobj.setCls(self, total_size, name)
                with tarfile.open(fileobj=fileobj, errorlevel=1) as tar:
                    tar.extractall("/mnt/fs")
            run_hook(hook, 'post_tar_untar')

            self.set_action("5: Setup release")
            shutil.copyfile("/etc/resolv.conf", "/mnt/fs/etc/resolv.conf")

            # Copy all file explicitly listed in /etc/release/backup
            run_hook(hook, 'pre_synchronize')
            self.set_action("6: Synchronize")
            backup = []
            if os.path.exists("/etc/release/backup"):
                for f in os.listdir("/etc/release/backup"):
                    with open("/etc/release/backup/%s" % f) as fi:
                        for line in fi:
                            backup.append(line.strip())
            if os.path.exists("/mnt/fs/etc/release/backup"):
                for f in os.listdir("/mnt/fs/etc/release/backup"):
                    with open("/mnt/fs/etc/release/backup/%s" % f) as fi:
                        for line in fi:
                            backup.append(line.strip())
            run_hook(hook, 'pre_synchronize_list')
            for bk in set(backup):
                override = False
                if bk.startswith("@"):
                    override = True
                    bk = bk[1:].strip()
                for fn in glob.glob(bk):
                    synchronize(fn, "/mnt/fs/%s" % fn, override)
            run_hook(hook, 'post_synchronize')

            self.set_action("7: Run post install (don't turn off device)")
            id = xmlrelease.xpath("@id")[0]
            xmlrelease.save("/mnt/fs/etc/release/%s.xml" % id)
            os.system("rm -f /mnt/fs/etc/release/current.xml")
            os.system("ln -s /etc/release/%s.xml /mnt/fs/etc/release/current.xml" % id)

            try:
                mount_chroot()
                os.system("mount > /tmp/mount.log")
                chroot_exec("/mnt/fs", ["/usr/sbin/run-postinsts"])
            except:
                result = False
                PrintException()
            finally:
                umount_chroot()
        except subprocess.CalledProcessError as err:
            result = False
            PrintException()
        except:
            result = False
            PrintException()
        finally:
            self.set_action("8: Clean installation")
            self.expose("Upgrade", 90, name, "Clean installation", 100)
            umount("/mnt/fs/user/disk")
            if not os.path.isdir("/mnt/fs/var/lib/release-manager"):
                os.mkdir("/mnt/fs/var/lib/release-manager")
            with open("/mnt/fs/var/lib/release-manager/upgrade_status", "w") as f:
                f.write("Success")
            umount("/mnt/fs")
            if remote:
                umount("/mnt/tmp")
            if result:
                run_hook(hook, 'pre_enable')
                self.set_action("8: Enable new installation")
                self.expose("Upgrade", 100, name, "Enable new installation", 100)
                if self.__auto_switch:
                    print("Setting permanent boot part")
                    bootenv.mmc_part = mmc_part_new
                if self.__test_update_part:
                    print("Setting temp boot part")
                    bootenv.mmc_part_volatile = mmc_part_new
                self.expose("Upgrade", 100, name, "Finished successful", 100)
                run_hook(hook, 'post_enable')
        if self.__daemon:
            self.set_action("Upgrade Terminate")
            if result:
                self.upgrade_by_tar_terminate("OK")
            else:
                self.upgrade_by_tar_terminate("FAIL")
        if not run_hook(hook, 'post_tar_upgrade'):
            return False
        return result

    @dbus.service.signal(dbus_interface='org.jdsu.releasemanager', signature='s')
    def upgrade_by_tar_terminate(self, name):
        print("Release changed to %s" % name)

    @dbus.service.method('org.jdsu.releasemanager')
    def upgrade_by_tar(self):
        print("CALL release-manager upgrade_by_tar")
        self.__run_task(self.__upgrade_by_tar)

    @dbus.service.method(dbus_interface='org.jdsu.releasemanager', in_signature='s', out_signature='s')
    def show_highlighted(self, url):
        print("CALL release-manager show_highlighted %s" % url)
        return show_highlighted(url, self.__proxy)

    @dbus.service.method(dbus_interface='org.jdsu.releasemanager', in_signature='s', out_signature='s')
    def show_diff(self, url):
        print("CALL release-manager show_diff %s" % url)
        return show_diff(url, self.__proxy)

    @dbus.service.method(dbus_interface='org.jdsu.releasemanager', in_signature='s', out_signature='b')
    def check_release(self, url):
        print("CALL release-manager check_release %s" % url)
        return check_release(url, self.__proxy)

    @dbus.service.method('org.jdsu.releasemanager')
    def exit(self):
        print("CALL release-manager exit")
        self.stop_daemon()

    def __set_proxy(self, proxy):
        print("SET PROXY: %s" % proxy)
        if len(proxy) > 1:
            self.__proxy = proxy
        else:
            self.__proxy = None

    @dbus.service.method(dbus_interface='org.jdsu.releasemanager', in_signature='s')
    def set_proxy(self, proxy):
        print("CALL release-manager set_proxy %s" % proxy)
        self.__set_proxy(proxy)

    @dbus.service.method(dbus_interface='org.jdsu.releasemanager', in_signature='s')
    def set_background_mode(self, state):
        in_background = bool(state)
        print("Set background mode: %s" % str(in_background))
        self.__use_background_mode = in_background

    def __is_partition_mounted(self, partition):
        with open('/etc/mtab', 'r') as fd:
            mounted_line = [line for line in fd if partition in line]
            if mounted_line:
                return True
            else:
                return False

