<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\settings;

use app\util\SMTDto;
use app\services\monitoring\SMTOtuMonitoringTestThresholdDto;
use app\sharedmemory\SMTMemoryManager;
use app\sharedmemory\SMTIMemorySupport;


/**
 * SmartOTU cache for settings with application scope.
 * Store SmartOtuSettings and SmartOtuThresholds in same cache.
 * 
 * @author Sylvain Desplat
 */
class SMTSmartOtuSettingsCache extends SMTDto implements SMTIMemorySupport
{
    /**
     * Only one instance stored in memory support
     */
    const ID = "0";
    
    /**
     * OTU parser port
     * 
     * @var number
     */
    private $otuPort = NULL;
    
    /**
     * OTU parser IP address (used for remote debug to set OTU address).
     * By default OTU address is set on local loop on 127.0.0.1
     *
     * @var string
     */
    private $otuIpAddress = NULL;    
    
    /**
     * OTU mode.
     *
     * @var string
     */
    private $otuMode = NULL;    
    
    /**
     * OTU type.
     *
     * @var string
     */
    private $otuType = NULL;
    
    /**
     * google map url retrieved from OTU.
     *
     * @var string
     */
    private $mapUrl = NULL;
    
    /**
     * The OTU log level
     * 
     * @var string SMTLogger::ERROR or SMTLogger::INFO or SMTLogger::DEBUG
     */
    private $logLevel = NULL;
    
    /**
     * Whether the OTU is in debug mode or NULL if the data wasn't yet retrieved
     * @var boolean
     */
    private $debugMode = NULL;
    
    /**
     * Session time out value.
     * 
     * @var number
     */
    private $sessionTimeOut = NULL;
    
    /**
     * OTU Date-time max drift allowed in comarison with connected web client. 
     * @var integer
     */
    private $dateTimeMaxDrift = NULL;
    
    /**
     * Whether the SMS option is available on the OTU or NULL if the data wasn't yet retrieved
     * @var boolean
     */
    protected $smsOptionValid = NULL;    

    /**
     * Whether attenuation monitoring critical thresholds are set in OTU
     */
    private $isCriticalThresholdsSet = FALSE;
    
    /**
     * Whether the REST API route cache has been checked
     * 
     * @var boolean
     */
    private $isRouteCacheChecked = FALSE;
    
    /**
     * Last reset of the OTU date-time 
     *
     * @var boolean
     */
    private $lastResetDateTime = 0;
    

    /**
     * Monitoring Acceptable Noise Margin in Db. 
     * Used to display that min noise level in trace viewer and to force user to set markers in traceviewer outside noise level.
     *  
     * @var float
     */
    private $monitoringAcceptableNoiseMarginDbDefault = SMTSmartOtuThresholds::DEFAULT_MONITORING_ACCEPTABLE_NOISE_MARGIN_DB_DEFAULT;    
    
    /**
     * Attenuation critical threshold (in Db)
     * 
     * @var float
     */
    private $attenuationCriticalThresholdDbDefault = SMTSmartOtuThresholds::DEFAULT_ATTENUATION_CRITICAL_THRESHOLD;
    
    /**
     * The default delta between TO and BACK TO thresholds for attenuation in Db
     * 
     * @var float
     */
    private $attenuationToBtoDeltaDbDefault = SMTSmartOtuThresholds::DEFAULT_MONITORING_ATTENUATION_TO_AND_BACK_TO_DELTA_DB_DEFAULT;
    
    //PEAK monitoring (NULL to force to retrieve value from ini file)
    private $peakToBtoDeltaDefault = NULL;
    private $monitorablePeakThresholdDb= NULL;
    private $missingPeakThresholdPtDefault= NULL;
    
    /**
     * @SMTAttributeInfosAnnotation(classname='app\serviceshelper\monitoring\SMTOtuMonitoringTestThresholdDto')
     * @var app\serviceshelper\monitoring\SMTOtuMonitoringTestThresholdDto
     */
    private $firstMarkerAttenuationThresholds = NULL;
    
    /**
     * @SMTAttributeInfosAnnotation(classname='app\serviceshelper\monitoring\SMTOtuMonitoringTestThresholdDto')
     * @var app\serviceshelper\monitoring\SMTOtuMonitoringTestThresholdDto
     */
    private $budgetVariationThresholds = NULL;
    
    /**
     * @SMTAttributeInfosAnnotation(classname='app\serviceshelper\monitoring\SMTOtuMonitoringTestThresholdDto')
     * @var app\serviceshelper\monitoring\SMTOtuMonitoringTestThresholdDto
     */
    private $peaksThresholdsDb= NULL;
    
    /**
     * @return string
     */
    public function getId()
    {
    	return self::ID;
    }
    
    /**
     * @param number $port
     */
    public function setOtuPort($port)
    {
    	$this->otuPort = $port;
    }
    
    /**
     * Returns the current OTU port or null if none is set
     *
     * @return number
     */
    public function getOtuPort()
    {
    	return $this->otuPort;
    }
    
    /**
     * Set whether attenuation monitoring critical thresholds are set in OTU
     * 
     * @param boolean $isCriticalThresholdsSet
     */
    public function setIsCriticalThresholdsSet($isCriticalThresholdsSet)
    {
    	$this->isCriticalThresholdsSet = $isCriticalThresholdsSet;
    }
    
    /**
     * Whether attenuation monitoring critical thresholds are set in OTU
     *
     * @return boolean
     */
    public function IsCriticalThresholdsSet()
    {
    	return $this->isCriticalThresholdsSet;
    }
    
    /**
     * Set whether the REST API route cache has been checked
     *
     * @param boolean $isRouteCacheChecked
     */
    public function setIsRouteCacheChecked($isRouteCacheChecked)
    {
    	$this->isRouteCacheChecked = $isRouteCacheChecked;
    }
    
    /**
     * Whether the REST API route cache has been checked
     *
     * @return boolean
     */
    public function isRouteCacheChecked()
    {
    	return $this->isRouteCacheChecked;
    }    
    
    /**
     * Set Last date-time reset
     *
     * @param int $lastResetDateTime
     */
    public function setLastResetDateTime($lastResetDateTime)
    {
    	$this->lastResetDateTime = $lastResetDateTime;
    }
    
    /**
     * Last date-time reset
     *
     * @return int
     */
    public function getLastResetDateTime()
    {
    	return $this->lastResetDateTime;
    }
    
    
    /**
     * Set the otu parser ip address in CACHE MEMORY. 
     * By default that value is the local loop except in the case of debug on a remote host.
     * @param string $otuIpAddress
     */
    public function setOtuIpAddress($otuIpAddress)
    {
    	$this->otuIpAddress = $otuIpAddress;
    }
    
    /**
     * Returns the otu parser ip address in CACHE MEMORY. 
     * By default that value is the local loop except in the case of debug on a remote host.
     *
     * @return string
     */
    public function getOtuIpAddress()
    {
    	return $this->otuIpAddress;
    }    
    
    /**
     * Set the otu mode in CACHE MEMORY.
     * 
     * @param string $otuMode
     */
    public function setOtuMode($otuMode)
    {
    	$this->otuMode = $otuMode;
    }
    
    /**
     * Returns the otu mode from CACHE MEMORY.
     *
     * @return string
     */
    public function getOtuMode()
    {
    	return $this->otuMode;
    }
    
    /**
     * Set the otu type in CACHE MEMORY.
     *
     * @param string $otuType
     */
    public function setOtuType($otuType)
    {
    	$this->otuType = $otuType;
    }
    
    /**
     * Returns the otu type from CACHE MEMORY.
     *
     * @return string
     */
    public function getOtuType()
    {
    	return $this->otuType;
    }    
    
    /**
     * Set the google map url in CACHE MEMORY.
     *
     * @param string $mapUrl
     */
    public function setMapUrl($mapUrl)
    {
    	$this->mapUrl = $mapUrl;
    }
    
    /**
     * Returns the google map url from CACHE MEMORY.
     *
     * @return string
     */
    public function getMapUrl()
    {
    	return $this->mapUrl;
    }    
    
    /**
     * Set log level
     *
     * @param string SMTLogger::ERROR or SMTLogger::INFO or SMTLogger::DEBUG
     */
    public function setLogLevel( $logLevel )
    {
    	$this->logLevel = $logLevel;
   	}
    
    /**
     *
     * @return string SMTLogger::ERROR or SMTLogger::INFO or SMTLogger::DEBUG
     */
    public function getLogLevel()
    {
    	return $this->logLevel;
    }
    
    /**
     *
     * @return boolean
     */
    public function isDebugMode()
    {
    	return $this->debugMode;
    }
    
    /**
     *
     * @param boolean $debugMode
     */
    public function setDebugMode( $debugMode )
    {
    	$this->debugMode = $debugMode;
    }
        
    /**
     *
     * @return number
     */
    public function getSessionTimeOut()
    {
    	return $this->sessionTimeOut;
    }
    
    /**
     *
     * @param number $sessionTimeOut
     */
    public function setSessionTimeOut( $sessionTimeOut )
    {
    	$this->sessionTimeOut = $sessionTimeOut;
    }
    
    /**
     *
     * @return integer
     */
    public function getDateTimeMaxDrift()
    {
    	return $this->dateTimeMaxDrift;
    }
    
    /**
     *
     * @param integer $dateTimeMaxDrift
     */
    public function setDateTimeMaxDrift( $dateTimeMaxDrift )
    {
    	$this->dateTimeMaxDrift = $dateTimeMaxDrift;
    }    
    
    /**
     * @param boolean $valid
     */
    public function setSmsOptionValid ( $smsOptionValid )
    {
    	$this->smsOptionValid = $smsOptionValid;
    }
    
    /**
     * Whether the SMS option is available on the OTU or NULL if the data wasn't yet retrieved
     * @return boolean or NULL
     */
    public function isSmsOptionValid()
    {
    	return $this->smsOptionValid;
    }    
    
    /**
     * @param app\serviceshelper\monitoring\SMTOtuMonitoringTestThresholdDto $peaksThresholdsDb
     */
    function setDefaultPeakThresholdsDb( $peaksThresholdsDb)
    {
    	$this->peaksThresholdsDb= $peaksThresholdsDb;
    }
    
    /**
     * @return \app\serviceshelper\monitoring\SMTOtuMonitoringTestThresholdDto
     */
    function getDefaultPeakThresholdsDb( )
    {
    	return $this->peaksThresholdsDb;
    }
        
    /**
     * @param app\serviceshelper\monitoring\SMTOtuMonitoringTestThresholdDto $firstMarkerAttenuationThresholds
     */
    function setDefaultFirstMarkerAttenuationThresholds( $firstMarkerAttenuationThresholds )
    {
    	$this->firstMarkerAttenuationThresholds = $firstMarkerAttenuationThresholds;
    }
        
    /**
     * @param app\serviceshelper\monitoring\SMTOtuMonitoringTestThresholdDto $budgetVariationThresholds
     */
    function setDefaultBudgetVariationThresholds( $budgetVariationThresholds )
    {
    	$this->budgetVariationThresholds = $budgetVariationThresholds;
    }            
    
    /**
     * @return \app\serviceshelper\monitoring\SMTOtuMonitoringTestThresholdDto
     */
    function getDefaultFirstMarkerAttenuationThresholds( )
    {
    	return $this->firstMarkerAttenuationThresholds;
    }
    
    /**
     * @return \app\serviceshelper\monitoring\SMTOtuMonitoringTestThresholdDto
     */
    function getDefaultBudgetVariationThresholds( )
    {
    	return $this->budgetVariationThresholds;
    }    
    
    /**
     * @return float
     */
    function getMonitoringAcceptableNoiseMarginDbDefault( )
    {
    	return $this->monitoringAcceptableNoiseMarginDbDefault;
    }

    /**
     * @param float $monitoringAcceptableNoiseMarginDbDefault
     */
    function setMonitoringAcceptableNoiseMarginDbDefault( $monitoringAcceptableNoiseMarginDbDefault )
    {
    	$this->monitoringAcceptableNoiseMarginDbDefault = $monitoringAcceptableNoiseMarginDbDefault;
    }    
    
    /**
     * @return float
     */
    function getAttenuationCriticalThresholdDbDefault( )
    {
    	return $this->attenuationCriticalThresholdDbDefault;
    }
    
    /**
     * @param float $monitoringAcceptableNoiseMarginDbDefault
     */
    function setAttenuationCriticalThresholdDbDefault( $attenuationCriticalThresholdDbDefault )
    {
    	$this->attenuationCriticalThresholdDbDefault = $attenuationCriticalThresholdDbDefault;
    }    
    
    /**
     * @return float
     */
    function getAttenuationToBtoDeltaDbDefault( )
    {
    	return $this->attenuationToBtoDeltaDbDefault;
    }
    
    /**
     * Register the delta between TO and BACK TO thresholds for attenuation (in Db) in cache and in settings
     *
     * @param SMTOtuMonitoringTestThresholdDto $attenuationToBtoDeltaDbDefault The delta between TO and BACK TO thresholds for attenuation (in Db)
     */
    public function setAttenuationToBtoDeltaDbDefault( $attenuationToBtoDeltaDbDefault  )
    {
        $this->attenuationToBtoDeltaDbDefault = $attenuationToBtoDeltaDbDefault;
    }
    
    /**
     * @return float
     */
    function getPeakToBtoDeltaDefault ()
    {
    	return $this->peakToBtoDeltaDefault;
    }
    
    /**
     * Register the delta between TO and BACK TO thresholds for peak (in Db) in cache and in settings
     *
     * @param float $peakToBtoDeltaDefault The delta between TO and BACK TO thresholds for peaks (in Db)
     */
    public function setPeakToBtoDeltaDefault ( $peakToBtoDeltaDefault )
    {
    	$this->peakToBtoDeltaDefault = $peakToBtoDeltaDefault ;
    }
    
    /**
     * @return float
     */
    function getMissingPeakThresholdPtDefault( )
    {
    	return $this->missingPeakThresholdPtDefault;
    }
    
    /**
     * Register the peaks shift limit in cache and in settings
     *
     * @param int $missingPeakThresholdPtDefault
     */
    public function setMissingPeakThresholdPtDefault( $missingPeakThresholdPtDefault)
    {
    	$this->missingPeakThresholdPtDefault= $missingPeakThresholdPtDefault;
    }
    
    /**
     * @return float
     */
    function getMonitorablePeakThresholdDb( )
    {
    	return $this->monitorablePeakThresholdDb;
    }
    
    /**
     * Register the threshold to detect peaks in cache and in settings
     *
     * @param float $monitorablePeakThresholdDb 
     */
    public function setMonitorablePeakThresholdDb( $monitorablePeakThresholdDb)
    {
    	$this->monitorablePeakThresholdDb= $monitorablePeakThresholdDb;
    }
    
    /**
     * Save Dto in shared memory
     *
     * @throws SMTAPCMemoryException
     */
    public function save()
    {
    	SMTMemoryManager::saveDto( $this );
    }
    
    /**
     * Delete Dto from shared memory
     *
     * @throws SMTAPCMemoryException
     */
    public function delete()
    {
    	SMTMemoryManager::deleteDto( $this );
    }
    
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }
    
    /**
     * Returns the dto class name.
     *
     * @return string the dto class name.
     */
    public static function getClass()
    {
    	return __CLASS__;
    }    
    
    /**
     * Otu settings cache is never obsolete: it must not be purged
     *
     * @return FALSE
     */
    public function isObsolete()
    {
    	return FALSE;
    }    
    
    /**
     * No obsolescence for those data: returns current time.
     * 
     * @see isObsolete()
     * @return number Timestamp of the last APC memory access (used to handle obsolescence of dtos in cache)
     */
    public function getLastAccess()
    {
        return microtime(TRUE);
    }
    
    /**
     * No obsolescence for those data: do nothing
     * 
     * @see isObsolete()
     *
    */
    public function touchAccess()
    {       
    }
    
    /**
     * @return app\settings\SMTSmartOtuSettingsCache
     */
    public static function getSettingsFromCacheMemory()
    {
    	return SMTMemoryManager::fetch( self::getClass(), self::ID );
    }
}
?>