<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\serviceshelper\system;

use app\SMTApplication;

use app\sharedmemory\SMTMemoryManager;

use app\serviceshelper\SMTServiceHelper;

use app\parser\SMTOtuApi;

/**
 *
 * @author Sylvain Desplat
 *
 */
class SMTOtu extends SMTServiceHelper
{
    const REBOOT = "REBOOT";
    const BLINK = "BLINK";
    const BLINK_ON = "ON";
    const BLINK_OFF = "OFF";
    const OTU_DATE_TIME_FORMAT = "Y,m,d,G,i,s";//YYYY,MM,DD,HH24,mm,SS
    const SLEEP_STOP_OTU_APPLICATION_US = 500000;
    
    /**
     * sleep 1s before reading ESR parser when changing date-time.
     *
     * @var int
     */
    const SLEEP_TIME_BEFORE_READING_ESR_US = 1000000;
    
    
    /**
     * Fetch OTU serial number
     */
    function getOtuSerialNumber()
    {
        $otuInfo = $this->sendReceive( SMTOtuApi::getOtuInfo() );
        list( $company, $otuType, $serialNumber, $otuMode, $otuReleaseVersion ) = explode(",", $otuInfo);
        //remove first 0
        return ltrim( $serialNumber, 0 );
    }
    
    
    /**
     * Reboot the OTU, set the OTU status to reboot, destroy the session.
     * 
     * @return array:string Array witht REBOOT=OK if successful; throws exceptions if failure
     * @throws \Exception if command fails
     */    
    function reboot()
    {
        //used to force the reboot by a system command
        $forceReboot = FALSE;
        
        //if SmartOTU is in degraded mode (cannot connect to OTU parser), force the reboot by a system command
        if ( !$this->getContext()->isOTUAvailable() )
        {
            $forceReboot = TRUE;
        }
        
        if ( $forceReboot != TRUE )
        {
            try 
            {
        		//Reboot the OTU from Otu application
        		$this->send( sprintf( SMTOtuApi::CMD_system_reboot ) );
            }
            catch ( \Exception $e )
            {
                $forceReboot = TRUE;
                $this->getContext()->traceException($e);
            }
        }        

		//set the OTU status to reboot and close the session
		SMTApplication::handleOTUReboot( $this->getContext() );
		
		//sleep 500ms to let the OTU application start stopping
		usleep( self::SLEEP_STOP_OTU_APPLICATION_US );
		
		//force system reboot.
		if ( $forceReboot )
		{
		    exec('sudo /sbin/reboot');
		}
    
		//return reboot OK in an array for JSON return string
    	return array( self::REBOOT => SMTOtuApi::RES_OK );
    }
    
    /**
     * Retrieves OTU date-time in a timestamp format
     * 
     * @return int Otu date-time in timestamp format.
     */
    function getOtuTimeStamp()
    {
        $otuDateTime = $this->sendReceive( sprintf( SMTOtuApi::CMD_get_otu_dateTime ) );
        $date = \DateTime::createFromFormat(self::OTU_DATE_TIME_FORMAT, $otuDateTime);
        return $date->getTimestamp();
    }
	
    /**
     * Set otu date time from timestamp and update SmartOTU timezone
     * 
     * @param number $otuTimeStamp New Otu time stamp including timezone (NOT IN UTC)
     * @param number $timezoneSec timezone in seconds
     * @param string $timezoneName timezone name
     * @param boolean $isTimezoneDst whether timezone is in DST period
     */
    function setOtuTimeStamp( $otuTimeStamp, $timezoneSec, $timezoneName, $isTimezoneDst )
    {
        $datetime = new \DateTime('@'.$otuTimeStamp);
        $YYYY= $datetime->format('Y');//Y,m,d,G,i,s
        $MM= $datetime->format('m');//Y,m,d,G,i,s
        $DD= $datetime->format('d');//Y,m,d,G,i,s
        $HH24= $datetime->format('G');//Y,m,d,G,i,s
        $mm= $datetime->format('i');//Y,m,d,G,i,s
        $SS= $datetime->format('s');//Y,m,d,G,i,s
                            
        $session = $this->getContext()->getSessionContext( FALSE );
        
        //set OTU date
        $this->getContext()->getOtuSocket()->send( sprintf( SMTOtuApi::CMD_set_otu_dateTime, $YYYY, $MM, $DD, $HH24, $mm, $SS ), self::SLEEP_TIME_BEFORE_READING_ESR_US );
                
        //update session last access ( used to track session timeout )
        $session->resetLastAccess();
        
        //update APC cache with new time
        SMTMemoryManager::touchAllDtos();
        
        //retrieve database connection
        $dbConnection = $this->getContext()->getDatabase();       
        
        //update timezone in database settings
        $dto = SMTSystemConfigurationDto::fetch($dbConnection);
        $dto->setTimezoneSec( $timezoneSec );
        $dto->setTimezoneName( $timezoneName );
        $dto->setTimezoneDst( $isTimezoneDst );
        $dto->save($dbConnection);
    }
    
    /**
     * Set STATUS LED blinking or not
     * @param boolean $blink
     */
    function blink($blink)
    {
    	$this->send( SMTOtuApi::setOtuShow($blink? self::BLINK_ON : self::BLINK_OFF) );
    }
    
    /**
     * Whether the STATUS LED is blinking
     * @return boolean
     */
    function isBlink()
    {
    	$blink = self::BLINK_OFF;
	    try 
	    {	
	    	$blink = $this->sendReceive( SMTOtuApi::getOtuShow() );
	    }
	    //catch the exception
	    catch ( \Exception $e )
	    {
	    	$this->getContext()->traceException($e);
	    }
    	return $blink == self::BLINK_ON;
    }
    
}
?>