<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\otdr;

use app\events\SMTEventFactory;

use app\http\SMTSessionDirectoryUtil;

use app\serviceshelper\monitoring\cache\SMTOperationManager;

use app\serviceshelper\monitoring\cache\SMTMeasureDataDto;

use app\otuMediation\SMTOtuMeasureOnDemandStatus;

use app\serviceshelper\monitoring\SMTMeasureOnDemandException;

use app\serviceshelper\monitoring\SMTMeasureOnDemandEventDto;

use app\util\SMTLogger;

use app\events\operations\SMTOperationEventDto;

use app\events\SMTEventMessageManager;

use app\parser\SMTOtuApi;
use app\serviceshelper\SMTServiceHelper;
use app\util\SMTUtil;

/**
 * Base class to start and query the status of an OTDR measure.
 * 
 * @author Sylvain Desplat
 */
class SMTCommonMeasurement extends SMTServiceHelper
{
    const MEASURE_ON_DEMAND_LOCK_FILE = "/../../../tmp/measureOnDemand.lock";
    
    /**
     * Default fiber index
     * @var int
     */
    const DEFAULT_FIBER_INDEX = 1.465;
    
    /**
     * Measure on demand lock file
     * 
     * @var string
     */
    private static $measureOnDemandLockFile = NULL; 
       
    /**
     * Check that no measure is running before starting a new one
     * 
     * @param string $portNumber
     * @param string $operationId
     * 
     * @throws SMTMeasureOnDemandException if a measurement is already running
     */
    protected function checkNoMeasurementRunning( $portNumber, $operationId )
    {
        $measurementStatus = $this->getMeasurementStatus( $operationId );
        
        //if a measure on demand is running throw an exception
        if ( !$measurementStatus->hasOperationEnded() )
        {
            //invalid status
            $measurementStatus->setStatus( SMTOperationEventDto::ERROR_ONE_MEASURE_ON_DEMAND );
            SMTEventMessageManager::getInstance()->sendOperationEvent( $measurementStatus );
            throw new SMTMeasureOnDemandException( SMTOperationEventDto::ERROR_ONE_MEASURE_ON_DEMAND );            
        }       

        //ensure that no measurement on demand message not sent to consumer is still present
        SMTEventMessageManager::getInstance()->purgeObsoleteMeasureOnDemandEventConsumers();
    }    
    
    /**
     * Get measure status from OTU parser
     * 
     * @param number $portNumber 
     * @param string $operationId 
     * @return SMTMeasureOnDemandEventDto;
     * 
     * @throws SMTMeasureOnDemandException if monitoring measurement fails
     */
    public function getMeasurementStatus( $operationId )
    {
	    //get measurement status:
	    $status = $this->sendReceive( SMTOtuApi::getMeasurementStatus() );
	    
   		$measurementStatus = $this->processMeasurementStatus( $operationId, $status);
        
        return $measurementStatus;
    }        
    
    /**
     * Stop measure on demand
     * 
     * @param string $operationId
     * @throws Exception
     */
    public function stopMeasure( $operationId )
    {
        if ( $this->acquireMeasureOnDemandLock() )
        {
            try 
            {
                $measureStatusDto = $this->getMeasurementStatus( $operationId );
                //stop measure if it is in waiting, init or in_progress
                if ( $measureStatusDto->getStatus() == SMTOperationEventDto::INIT || $measureStatusDto->getStatus() == SMTOperationEventDto::IN_PROGRESS || $measureStatusDto->getStatus() == SMTOperationEventDto::WAITING )
                {
                    $this->send( SMTOtuApi::getStopMeasurement() );
                }
                $this->releaseMeasureOnDemandLock();
            }
            catch( \Exception $e)
            {
                 $this->releaseMeasureOnDemandLock();
                 throw $e;   
            }
        }
        else
        {
        	$this->getContext()->getLogger()->trace("Couldn't acquire lock to stop measure for operation Id:".$operationId, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTMeasureOnDemandException( SMTOperationEventDto::FAILED, MSG_ERROR_STOP_MEASURE ) ;
        }
        
    }
    
    /**
     * Get measurement status
     *
     * @param string $operationId
     * 
     * @return SMTMeasureOnDemandEventDto;
     *
     * @throws SMTMeasureOnDemandException if monitoring measurement fails
     */
    public function getMeasurementAddStatus( $operationId )
    {
    	//get measurement addition status:
   		$status = $this->sendReceive( SMTOtuApi::getMeasurementAddStatus() );
    
   		$measurementStatus = $this->processMeasurementStatus( $operationId, $status);
   		   		
   		if ( !SMTOtuMeasureOnDemandStatus::isMeasurementAdditionStatusOK( $status ) )
   		{
   			throw new SMTMeasureOnDemandException( $measurementStatus->getStatus() );
   		}   		
    
    	return $measurementStatus;
    }    
    
    /**
     * Cache measure info in APC
     * 
     * @param string $operationId
     * @param int $portId
     * @param string $traceDirectory
     * @param string $traceName
     * @param int $linkId
     */
    protected function cacheMeasureInfo( $operationId, $portId, $traceDirectory, $traceName, $linkId = NULL )
    {
        SMTOperationManager::addMeasureInfo($operationId, $portId, $traceDirectory, $traceName, $linkId );
    }
    
    /**
     * Retrieve measure info from APC cache
     * 
     * @return SMTMeasureDataDto
     */
    public function getMeasureInfo( $operationId )
    { 
    	return SMTOperationManager::getMeasureInfo($operationId);
    }
    
    /**
     * Delete measure info from APC cache
     *
     * @return SMTMeasureDataDto
     */
    public function deleteMeasureInfo( $operationId )
    {
    	return SMTOperationManager::deleteMeasureInfo( $operationId );
    }    
    
    /**
     * Process measurement status:
     * - check is status is valid
     * - notify event message manager of the change of measure status
     *
     * @param string $operationId
     * @param string $status The result of command status to analyse
     * 
     * @return SMTMeasureOnDemandEventDto;
     *
     * @throws SMTMeasureOnDemandException if monitoring measurement fails
     */
    private function processMeasurementStatus( $operationId, $otuStatus )
    {    
        $measurementStatus = SMTEventFactory::createMeasureOnDemandEvent( NULL, $operationId, SMTOperationEventDto::NO_PROGRAMMED );                         
        
    	if ( SMTOtuMeasureOnDemandStatus::isValidMeasureStatus( $otuStatus ) )
    	{
    	    $decodedStatus = SMTOtuMeasureOnDemandStatus::decodeOtuStatus( $otuStatus );
    	    $measurementStatus->setStatus( $decodedStatus );
    	    

    	    $measureCache = $this->getMeasureInfo( $operationId );
    	    if ( $measureCache != NULL )
    	    {    	    
    	        $measurementStatus->setPortNumber( $measureCache->getPortId() );
    	        $measurementStatus->setLinkId( $measureCache->getLinkId() );
    	        
    	        //retrieve port number on which measure was launched
    	        $portNumber = $measureCache->getPortId();
    	        $otdrTraceName = SMTSessionDirectoryUtil::buildOtdrFileName($portNumber, $operationId);    	         
    	        $measurementStatus->setOtdrTraceName($otdrTraceName);
    	        
    	        //check if measure status has changed before sending a message
    	        if ( $decodedStatus != $measureCache->getStatus() )
        	    {
        		    SMTEventMessageManager::getInstance()->sendOperationEvent( $measurementStatus );
        	    }
        	    $measureCache->setStatus( $decodedStatus );
        	    $measureCache->save();
    	    }
    	}
    	else
    	{
    		//invalid status
    		$measurementStatus->setStatus( SMTOperationEventDto::INVALID_STATUS );
    		SMTEventMessageManager::getInstance()->sendOperationEvent( $measurementStatus );
    		throw new SMTMeasureOnDemandException( SMTOperationEventDto::INVALID_STATUS  );
    	}
    
    	return $measurementStatus;
    }    
    
    private static function getMeasureOnDemandLockFile()
    {
    	if ( self::$measureOnDemandLockFile == NULL )
    	{
    		$handle = fopen( __DIR__.self::MEASURE_ON_DEMAND_LOCK_FILE, "w+");
    		 
    		self::$measureOnDemandLockFile = ( $handle != FALSE)? $handle : NULL;
    	}
    	return self::$measureOnDemandLockFile;
    }
    
    protected function acquireMeasureOnDemandLock()
    {
        $retries = 0;
        $max_retries = 30;
        
        // keep trying to get a lock as long as possible ( max 10s)
        do
        {
        	if ($retries > 0)
        	{
        		usleep( 1000 * ( $retries*$retries ) );
        	}
        	$retries += 1;
        } while (!flock(self::getMeasureOnDemandLockFile(), LOCK_EX) && $retries <= $max_retries);
        
        return ($retries < $max_retries);        
    }
    
    protected function releaseMeasureOnDemandLock()
    {
    	return flock( self::$measureOnDemandLockFile, LOCK_UN );
    }    
    
    /**
     * Launch asynchronous measure polling
     *
     * @param string $phpFileToExecute
     * @param string $operationClass
     * @param string $operationId
     * @param string $linkId NOT NULL for measure on link 
     */
    static function executeAsynchronousMeasureProcess( $phpFileToExecute, $operationClass, $operationId, $linkId = NULL )
    {
    	$output = array();
    	if ( isset( $linkId ) )
    	{
    	    passthru(SMTUtil::PHP_CLIENT_EXECUTABLE.$phpFileToExecute.' -- '.$operationClass.','.$operationId.','.$linkId.' >> '.SMTLogger::getLogFilePath().' 2>&1 & ', $output );
    	}
    	else
    	{
    	    passthru(SMTUtil::PHP_CLIENT_EXECUTABLE.$phpFileToExecute.' -- '.$operationClass.','.$operationId.' >> '.SMTLogger::getLogFilePath().' 2>&1 & ', $output );
    	}
    }
}
?>