<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\monitoring;

use app\events\SMTEventFactory;

use app\serviceshelper\monitoring\cache\SMTOperationManager;

use app\otuMediation\SMTOtuOperationStatus;

use app\util\SMTLogger;

use app\events\operations\SMTOperationEventDto;

use app\events\SMTEventMessageManager;

use app\parser\SMTOtuApi;
use app\serviceshelper\SMTServiceHelper;
use app\util\SMTUtil;

/**
 * Start and query the status of a test on demand
 * 
 * @author Sylvain Desplat
 */
class SMTTestOnDemand extends SMTServiceHelper
{
    const TEST_ON_DEMAND_LOCK_FILE = "/../../../tmp/testOnDemand.lock";
    
    /**
     * Test on demand lock file
     * 
     * @var string
     */
    private static $testOnDemandLockFile = NULL; 
    
    /**
     * Start test
     * 
     * @param number $testId
     * @param number $linkId
     * @param boolean $monitorTest whether we want to monitor the execution of the test. TRUE by default.
     * 
     * @return SMTTestOnDemandEventDto;
     * 
     * @throws SMTTestOnDemandException if monitoring test launch has failed
     */
    function startTest( $testId, $linkId, $monitorTest = TRUE )
    {                               
        $measureStatus = SMTEventFactory::createTestOnDemandEvent($testId, NULL, SMTOperationEventDto::NO_PROGRAMMED );
        
        if ( $this->acquireTestOnDemandLock() )
        {
            $operationId = SMTTestOnDemandEventDto::generateTestOnDemandOperationId();
            
            //check that no test is running
            //if a test is already running, abort start test operation
            try
            {
                $testStatus = $this->checkNoTestRunning( $testId, $operationId );
            }
            catch ( \Exception $e )
            {
                $this->releaseTestOnDemandLock();
            	$this->getContext()->getLogger()->traceException($e);
            	throw $e;
            }                        
            
            try 
            {
                $testStatus = $this->sendReceive( SMTOtuApi::getExecuteTestCommand( $testId, SMTOtuApi::RES_ALL ) );//SMTOtuApi::RES_WEB //SMTOtuApi::RES_ALL          
            } 
            catch ( \Exception $e ) 
            {
                //test on demand start failed; don't monitor test progress.
                $monitorTest = FALSE;
                
                //nothing to do, try to get info with getTestStatus
                $this->getContext()->getLogger()->traceException($e);
            }    	
            
            try 
            {
                $measureStatus = $this->getTestStatus( $testId, $operationId );
                $this->releaseTestOnDemandLock();
                
                //test was added successfully, start the polling in an asynchronous process if required 
                if ( $monitorTest )
                {
                    $this->cacheTestInfo($operationId, $testId, NULL, NULL);
                    
                    $testOnDemandClass = SMTTestOnDemandEventDto::getClass();
                    $testOnDemandClass = str_replace("\\", "\\\\", $testOnDemandClass);
                    //passthru(SMTUtil::PHP_CLIENT_EXECUTABLE.__DIR__.'/testondemandProcess.php -- '.$testOnDemandClass.','.$operationId.','.$testId.','.$linkId.' >> '.SMTLogger::getLogFilePath().' 2>&1 & ', $output );
                    
                    self::executeAsynchronousTestProcess( __DIR__.'/testondemandProcess.php', $testOnDemandClass, $operationId, $testId, $linkId);
                }
            }
            catch( \Exception $e )
            {                
                $this->releaseTestOnDemandLock();
                throw  $e;
            }
        }
        else
        {
        	SMTLogger::getInstance()->trace("Couldn't acquire lock for test: ".$testId, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTTestOnDemandException( SMTOperationEventDto::FAILED, "Couldn't acquire lock to start test"  );
        }
        
        return $measureStatus;
    }    
    
    /**
     * Launch asynchronous test polling 
     * 
     * @param string $phpFileToExecute
     * @param string $OperationClass
     * @param string $operationId
     * @param string $testId
     * @param string $resourceId
     */
    private static function executeAsynchronousTestProcess( $phpFileToExecute, $OperationClass, $operationId, $testId, $resourceId )
    {
        $output = array();        
        passthru(SMTUtil::PHP_CLIENT_EXECUTABLE.$phpFileToExecute.' -- '.$OperationClass.','.$operationId.','.$testId.','.$resourceId.' >> '.SMTLogger::getLogFilePath().' 2>&1 & ', $output );
    }
    
    /**
     * Cache test info to monitor test
     *
     * @param string $operationId
     * @param int $testId
     * @param string $traceDirectory
     * @param string $traceName
     */
    protected function cacheTestInfo( $operationId, $testId, $traceDirectory, $traceName )
    {
    	SMTOperationManager::addTestInfo($operationId, $testId, $traceDirectory, $traceName);
    }
    
    /**
     * Retrieve Test info from APC cache
     *
     * @return SMTTestDataDto
     */
    public function getTestInfo( $operationId )
    {
    	return SMTOperationManager::getTestInfo($operationId);
    }
    
    /**
     * Delete Test info from APC cache
     *
     * @return SMTMeasureDataDto
     */
    public function deleteTestInfo( $operationId )
    {
    	//retrieve Test main settings from memory:
    	return SMTOperationManager::deleteTestInfo( $operationId );
    }    
    
    /**
     * Check that no test is running before starting a new one
     * 
     * @param string $testId
     * @param string $operationId
     * 
     * @throws SMTTestOnDemandException if a test is already running
     */
    private function checkNoTestRunning( $testId, $operationId )
    {
        $testStatus = $this->getTestStatus( $testId, $operationId );
        
        //if a test on demand is running throw an exception
        if ( !$testStatus->hasOperationEnded() )
        {
            //invalid status
            $testStatus->setStatus( SMTOperationEventDto::ERROR_ONE_TEST_ON_DEMAND );
            SMTEventMessageManager::getInstance()->sendOperationEvent( $testStatus );
            throw new SMTTestOnDemandException( SMTOperationEventDto::ERROR_ONE_TEST_ON_DEMAND );            
        }       

        //ensure that no test on demand event not sent to consumer is still present
        SMTEventMessageManager::getInstance()->purgeObsoleteTestOnDemandEventConsumers();
    }    
    
    /**
     * Get launched test status
     * 
     * @param number $testId
     * @param string $operationId 
     * @return SMTTestOnDemandEventDto;
     * 
     * @throws SMTTestOnDemandException if monitoring test fails
     */
    function getTestStatus( $testId, $operationId )
    {
        $measureStatus = SMTEventFactory::createTestOnDemandEvent($testId, $operationId, SMTOperationEventDto::NO_PROGRAMMED );
                
		//get test status:
		$testStatus = $this->sendReceive( SMTOtuApi::getTestStatus() );

		if ( SMTOtuOperationStatus::isValidOperationStatus( $testStatus ) )
		{
		    $decodedTestStatus = SMTOtuOperationStatus::decodeOtuStatus( $testStatus );
			$measureStatus->setStatus( $decodedTestStatus );
			
			//check if test status has changed before sending a message
			$testCache = $this->getTestInfo( $operationId );
			if ( $testCache != NULL )
			{
				if ( $decodedTestStatus != $testCache->getStatus() )
				{
					SMTEventMessageManager::getInstance()->sendOperationEvent( $measureStatus );
				}
				$testCache->setStatus( $decodedTestStatus );
				$testCache->save();
			}
		}
		else
		{
		    //invalid status
		    $measureStatus->setStatus( SMTOperationEventDto::INVALID_STATUS );
		    SMTEventMessageManager::getInstance()->sendOperationEvent( $measureStatus );
			throw new SMTTestOnDemandException( SMTOperationEventDto::INVALID_STATUS );
		}
        
        return $measureStatus;
    }        

    private static function getTestOnDemandLockFile()
    {
    	if ( self::$testOnDemandLockFile == NULL )
    	{
    		$handle = fopen( __DIR__.self::TEST_ON_DEMAND_LOCK_FILE, "w+");
    		 
    		self::$testOnDemandLockFile = ( $handle != FALSE)? $handle : NULL;
    	}
    	return self::$testOnDemandLockFile;
    }
    
    private function acquireTestOnDemandLock()
    {
        $retries = 0;
        $max_retries = 30;
        
        // keep trying to get a lock as long as possible ( max 10s)
        do
        {
        	if ($retries > 0)
        	{
        		usleep( 1000 * ( $retries*$retries ) );
        	}
        	$retries += 1;
        } while (!flock(self::getTestOnDemandLockFile(), LOCK_EX) && $retries <= $max_retries);
        
        return ($retries < $max_retries);        
    }
    
    private function releaseTestOnDemandLock()
    {
    	return flock( self::$testOnDemandLockFile, LOCK_UN );
    }    
}
?>