<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of VIAVI
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from VIAVI
// Copyright JDSU 2015. All rights reserved.
// *********************************************************
namespace app\serviceshelper\monitoring;

require_once 'lib/libkml.php';

use libKML\KML;
use libKML\features\Placemark;
use libKML\features\Feature;
use libKML\geometries\LineString;
use libKML\field_types\Coordinates;

use app\parser\SMTOtuApi;
use app\util\SMTUtil;
use app\util\SMTLogger;
use app\util\SMTKmlException;
use app\util\SMTIOException;
use app\http\SMTContext;
use app\http\SMTSessionDirectoryUtil;
use app\serviceshelper\SMTServiceHelper;
use app\serviceshelper\alarm\SMTAlarmSeverity;
use app\settings\SMTSmartOtuSettings;
use app\serviceshelper\maintenance\SMTLicence;
/**
 *
 * @author Sylvain Desplat
*/
class SMTGpsRoute extends SMTServiceHelper
{
    const GPS_ROUTE_DIR = '/otu/reference/gps_route/';    
    const KML_ALARM_FILE_NAME = "alarm_%d_%d";
    const KML_ICON_ID = "alarm_style_%d"; 
    const ALARM_URL = "http://%s/app.php#SYSTEM_SUMMARY!";
    const ALARM_URL_HTTPS = "https://%s/app.php#SYSTEM_SUMMARY!";
    const ALARM_DESCRIPTION = "%s: %s<br/>\n%s: %s<br/>\n%s: %s<br/>\nGPS: %f %f<br/>\n%s: %s<br/>\nOTU: %s<br/>\n%s";
    const GOOGLE_MAPS_URL = "http://maps.google.com/maps?q=%f,%f";
    const MAP_URL_INDEX = 0;
    const MAP_URL_COORDINATE_FORMAT_INDEX = 1;
    const PHP_STRING = '<?php';
    const HTML_STRING = '<html';
    
    /**
     * The kml file name
     * 
     * @var string
     */
    private $fileName = "";
    
    /**
     * The gps file name for OTU
     *
     * @var string
     */
    private $gpsFileName = "";    
    
    /**
     * The link identifier
     *
     * @var integer
     */
    private $linkId = 0;
    
    /**
     * 
     * @var libKML\KML 
     */
    private $kml = NULL;

    /**
     *
     * @var libKML\features\Placemark
     */
    private $placemark = NULL;

    /**
     *
     * @var libKML\geometries\LineString
     */
    private $lineString = NULL;    
    
    
    /**
     * read kml and save the retrieved coordinates in a gps file under format
     * [x1,y1,x2,y2,x3,y3]
     * 
     * @param string $fileName
     * @param integer $linkId link id
     * @return The gps file name after processing and used by OTU
     */
    function processKml( $fileName, $linkId)
    {
    	if ( self::checkXMLFile(self::GPS_ROUTE_DIR.$fileName) )
    	{
        	$this->fileName = $fileName;
        	$this->linkId = $linkId;
        	$this->renameKml( $linkId );
        	$this->readKml();
        	$this->extractLineString();
        	$this->generateGpsFileForOtu();
	        //if all operations are successful, we update the link in OTU with the gps file name
    	    $this->send( SMTOtuApi::updateGpsFileNameCommand( $linkId, $this->gpsFileName ) );
        	SMTLogger::getInstance()->trace("Successfully saved gps:".$this->gpsFileName, SMTLogger::INFO);
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Failed to check KML file : ".$fileName, SMTLogger::ERROR);
    		throw new SMTIOException( SMTIOException::COULD_NOT_CREATE_FILE, "Couldn't validate KML file content : ".$fileName);
    	}
        return $this->gpsFileName;
    }
    
    /**
     * Search kml file name from the linkId
     * @param integer $linkId
     * @return string
     */
    static function getKmlFileName($linkId)
    {
    	$fileName = null;
    	//retrieve file name
    	$files = glob( self::GPS_ROUTE_DIR."*".'_'.$linkId.'*.kml');
    	//search last backup file
    	if ( $files != FALSE && !empty( $files ) )
    	{
    		foreach ( $files as $file )
    		{
    			$fileName = $file;
    			break;
    		}
    	}
    	return $fileName;
    }
    
    /**
     * Delete kml file and remove it from link.
     * 
     * @param string $fileName
     * @param integer $linkId link id 
     */
    function deleteKml( $fileName, $linkId )
    {
    	$this->linkId = $linkId;
    	$this->fileName = $fileName;
    	
    	//we dissociate first the gps file from the link on OTU
    	$this->send( SMTOtuApi::deleteGpsFileCommand( $linkId ) );
    	
    	$kmlFile = self::retrieveKmlFullPath( $fileName );
    	if ( $kmlFile != NULL )
    	{
    	    unlink($kmlFile);
    	}
    	
    	$otuFile = self::GPS_ROUTE_DIR.$fileName;
    	if ( $otuFile != NULL )
    	{
    		unlink($otuFile);
    	}
    	
    	SMTLogger::getInstance()->trace("Successfully removed gps file ".$fileName." for link:".$linkId, SMTLogger::INFO);
    }
    
    /**
     * Rename KML file name to kml name without special characters + linkId
     *
     * @param integer $linkId link id 
     */
    function renameKml( $linkId )
    {
    	$letter1 = chr(rand(97,122));
    	$letter2 = chr(rand(97,122));
        $new_Name = preg_replace('/[^-\w\.]+/', '_', $this->fileName, -1, $count);        
        $new_Name = pathinfo($new_Name, PATHINFO_FILENAME).'_'.$linkId.$letter1.$letter2.'.kml';
        SMTLogger::getInstance()->trace("Rename kml file from: ".$this->fileName.' to:'.$new_Name, SMTLogger::INFO);
        if ( rename(self::GPS_ROUTE_DIR.$this->fileName, self::GPS_ROUTE_DIR.$new_Name) === TRUE)
        {
            $this->fileName = $new_Name;
        }
        else
        {
        	throw new SMTIOException( SMTIOException::COULD_NOT_CREATE_FILE, "Couldn't rename file from:".$this->fileName." to:".$new_Name );
        }
    }
    
    /**
     * Read KML file and create kml object
     * 
     */
    function readKml()
    {        
        $kml_data = file_get_contents(self::GPS_ROUTE_DIR.$this->fileName);
        $this->kml = KML::createFromText($kml_data);
        SMTLogger::getInstance()->trace("Read kml file: ".$this->fileName, SMTLogger::INFO);
    }
    
    /**
     * Extract the LineString from the given kml
     * 
     * @return integer Returns the number of LineString found within a Placemark.
     */
    function extractLineString()
    {
        $all_features = $this->kml->getAllFeatures();
        $placemarkCount = 0;
        $lineStringCount = 0;
        $this->placemark = NULL;
        $this->lineString = NULL;
        
        foreach($all_features as $feature) 
        {
            //search one Linestring within a placemark 
            if ( $feature instanceof Placemark )
            {                
                $placemarkCount++;
                $geometry = $feature->getGeometry();
                if ($geometry instanceof LineString)
                {
                    $lineStringCount++;
                    if ( $lineStringCount == 1 )
                    {
                        $this->placemark = $feature;
                        $this->lineString = $geometry;
                        SMTLogger::getInstance()->trace("Found 1 line string in placemark: ".$this->placemark->getName(), SMTLogger::INFO);
                    }
                }
            }            
        }
        if ( $placemarkCount < 1 )
        {
        	SMTLogger::getInstance()->trace("No placemark found", SMTLogger::ERROR);
        	throw new SMTKmlException( SMTKmlException::NO_PLACEMARK);
        }        
        if ( $lineStringCount < 1 )
        {
            SMTLogger::getInstance()->trace("No line string found", SMTLogger::ERROR);
            throw new SMTKmlException( SMTKmlException::NO_LINE_STRING);
        }
        if ( $lineStringCount > 1 )
        {
        	SMTLogger::getInstance()->trace("More than 1 line string found: ".$lineStringCount, SMTLogger::ERROR);
        	throw new SMTKmlException( SMTKmlException::TOO_MANY_LINE_STRING);
        }        
    }
        
    /**
     * Generate a GPS file from the LineString found in the given kml file
     * 
     * @return boolean Whether the generation of the file was successfull
     */
    function generateGpsFileForOtu()
    {
        $file_contents = '';
        $lineStringCoordinateCount = 0;
        $this->gpsFileName = pathinfo($this->fileName, PATHINFO_FILENAME);                
        
        if ( ($this->placemark != NULL) && ($this->lineString != NULL) )
        {
            $coordinates = $this->lineString->getCoordinates();
            $lineStringCoordinateCount = count($coordinates);
            if ( $lineStringCoordinateCount > 0 ) 
            {
            	$coordinates_strings = array();            
            	foreach($coordinates as $coordinate)  
            	{
            		$coordinates_strings[] = implode(",", array($coordinate->getLongitude(), $coordinate->getLatitude()));
            	}
            	$file_contents= '[';
            	$file_contents.= implode(",", $coordinates_strings);
            	$file_contents.= ']'."\n";            	 
            }
        }
        
        if ( $lineStringCoordinateCount == 0)
        {
            SMTLogger::getInstance()->trace("No linestring coordinate found in file ".$this->fileName, SMTLogger::ERROR);
            $success = FALSE;
            throw new SMTKmlException( SMTKmlException::NO_LINE_STRING_COORDINATES);
        }
        else
        {
            if ( !( file_put_contents( self::GPS_ROUTE_DIR.$this->gpsFileName, $file_contents, LOCK_EX ) !== FALSE) )
            {                
                SMTLogger::getInstance()->trace("Failed to save coordinates in file ".$this->gpsFileName, SMTLogger::ERROR);
                throw new SMTKmlException( SMTKmlException::COULD_NOT_CREATE_GPS_FILE);
            }            
            else
            {
                SMTLogger::getInstance()->trace("Successfully saved gps file with content:".$file_contents, SMTLogger::DEBUG);
            }
        }
    }
    
    /**
     * Build kml url name from gps file name for download
     * 
     * @param string $gpsFileName
     */
    static function buildkmlUrl( $gpsFileName)
    {
        $kmlUrl = $gpsFileName.'.kml';
        $kmlUrl = "/maintenance/kml?fileName=".urlencode($kmlUrl);
        return $kmlUrl;
    }
    
    /**
     * Build kml file name from gps file name
     * @param string $fileName
     */    
    static function retrieveKmlFullPath( $fileName )
    {
        $kmlPath = self::GPS_ROUTE_DIR.$fileName.'.kml';
        return ( file_exists( $kmlPath)? $kmlPath: NULL );
    }
    
    /**
     * Create google maps url
     * 
     * @param float $gpsX
     * @param float $gpsY
     * @return string
     */
    public static function createGoogleMapsUrl($gpsX,$gpsY)
    {        
        $mapUrl = SMTSmartOtuSettings::getMapUrl();
        return sprintf( $mapUrl, $gpsY, $gpsX );
    }
    
    /**
     * Retrieve google maps url from OTU if GPS coordinates are used
     *
     * @param boolean $isValidGps
     */
    public function updateGoogleMapsUrl( $isValidGps )
    {
        if ( $isValidGps )
        {
            $mapUrlString = $this->sendReceive( SMTOtuApi::getMapUrlCommand(), FALSE );
            $mapUrlArray = SMTUtil::splitStringByComma($mapUrlString);
            //turn http://maps.google.com/maps/place/%.5f,%.5f into http://maps.google.com/maps/place/%f,%f
            $mapUrl = str_replace($mapUrlArray[self::MAP_URL_COORDINATE_FORMAT_INDEX], "%f", $mapUrlArray[self::MAP_URL_INDEX]);
            //update maps url in settings
            SMTSmartOtuSettings::setMapUrl($mapUrl);
        }
    }
    
    /**
     * Build kml alarm file
     * 
     * @param integer $alarmId
     * @param integer $linkId
     * @param float $gpsX
     * @param float $gpsY
     * @param string $description
     * @param integer $severity
     * @param boolean $cleared
     * @param integer $date
     * @param string $linkName
     * @param string $serverName
     * @param string $hostName
     * @throws SMTKmlException
     * @return string
     */
    static function createKmlAlarm( $alarmId, $linkId, $gpsX, $gpsY, $description, $severity, $cleared, $date, $linkName, $serverName, $hostName, SMTContext $context )
    {
        $isHighSecurity = SMTLicence::isHighSecurityMode($context);
        $fileName = sprintf( self::KML_ALARM_FILE_NAME, $linkId, $alarmId);
        SMTSessionDirectoryUtil::checkFilesCountInTempDir();
    	$kmlPath = SMTSessionDirectoryUtil::SMART_OTU_TEMP_DIR.$fileName.'.kml';    	    	
    	$date_Formatted = date("Y-m-d H:i:s",$date);
    	$severityImageUrl = SMTAlarmSeverity::getSeverityImageUrl($severity, $cleared, $serverName, $isHighSecurity);
    	$severityString = SMTAlarmSeverity::getSeverityTranslation($severity, $cleared);
    	$iconId = sprintf( self::KML_ICON_ID, $severity);
    	$placemarkId = $linkId.'_'.$alarmId;
  	    $alarmUrl = $isHighSecurity? sprintf( self::ALARM_URL_HTTPS, $serverName): sprintf( self::ALARM_URL, $serverName);    	
    	$linkNameEncoded = htmlspecialchars ($linkName, ENT_QUOTES );    	
    	$descriptionAlarm = sprintf( self::ALARM_DESCRIPTION, OTU_RES_SEVERITY, $severityString, OTU_RES_DESCRIPTION, $description, OTU_RES_LINK, $linkNameEncoded, $gpsY, $gpsX, OTU_RES_TIMESTAMP, $date_Formatted, $hostName, $alarmUrl);
    	$alarmDescriptionEncoded = htmlspecialchars ($descriptionAlarm, ENT_QUOTES );
    	
    	$file_contents = self::createDocument($gpsX, $gpsY, $placemarkId, $alarmDescriptionEncoded, $linkNameEncoded, $iconId, $severityImageUrl);
    	
    	if ( !( file_put_contents( $kmlPath, $file_contents, LOCK_EX ) !== FALSE) )
    	{
    		SMTLogger::getInstance()->trace("Failed to save alarm coordinates in kml file ".$kmlPath, SMTLogger::ERROR);
    		throw new SMTKmlException( SMTKmlException::COULD_NOT_CREATE_GPS_FILE);
    	}
    	return ( file_exists( $kmlPath)? $kmlPath: NULL );
    }
    
    private static function createDocument($gpsX, $gpsY, $placemarkId, $descriptionAlarm, $linkName, $iconId, $severityImageUrl)
    {    	
    	$file_contents='<?xml version="1.0" encoding="UTF-8"?>'."\n";
    	$file_contents.='<kml xmlns="http://www.opengis.net/kml/2.2" xmlns:gx="http://www.google.com/kml/ext/2.2" xmlns:kml="http://www.opengis.net/kml/2.2" xmlns:atom="http://www.w3.org/2005/Atom">'."\n";
    	$file_contents.='<Document>'."\n";
    	$file_contents.='<name>'.sprintf( OTU_ALARM, $linkName ).'</name>'."\n";
    	$file_contents.=self::createIconAlarm($iconId, $severityImageUrl);
    	$file_contents.=self::createPlacemark($gpsX, $gpsY, $placemarkId, $descriptionAlarm, $linkName, $iconId);    	    	
    	$file_contents.='</Document>'."\n";
    	$file_contents.='</kml>'."\n";
    	
    	return $file_contents;
    }
    
    private static function createIconAlarm($iconId, $severityImageUrl)
    {
        $file_contents = '';
        if ( $severityImageUrl != NULL )
        {
        	$file_contents='<Style id="'.$iconId.'">'."\n";
        	$file_contents.='<IconStyle id="icon_'.$iconId.'">'."\n";
        	$file_contents.='<Icon>'."\n";
        	$file_contents.='<href>'.$severityImageUrl.'</href>'."\n";
        	$file_contents.='<scale>1.0</scale>'."\n";
        	$file_contents.='</Icon>'."\n";
        	$file_contents.='</IconStyle>'."\n";
        	$file_contents.='</Style>'."\n";
        }
    	return $file_contents;
    }
    
    private static function createPlacemark($gpsX, $gpsY, $placemarkId, $descriptionAlarm, $linkName, $iconId)
    {
        $file_contents='<Placemark id="'.$placemarkId.'">'."\n";
        $file_contents.='<name>'.$linkName.'</name>'."\n";
        $file_contents.='<description>'.$descriptionAlarm.'</description>'."\n";
        $file_contents.='<styleUrl>#'.$iconId.'</styleUrl>'."\n";
        $file_contents.='<Point>'."\n";
        $file_contents.='<coordinates>'.$gpsX.','.$gpsY.',0</coordinates>'."\n";
        $file_contents.='</Point>'."\n";
        $file_contents.='</Placemark>'."\n";
        return $file_contents;
    }
    
    /**
     * Check xml file content only if we are in Enhanced Security
     * @param string $xmlFilename Path to the XML file
     * @param string $version 1.0
     * @param string $encoding utf-8
     * @return bool
     */
    public static function checkXMLFile($xmlFilename, $version = '1.0', $encoding = 'utf-8')
    {
    	$valid = TRUE;
   		$xmlContent = file_get_contents($xmlFilename);
   		$valid = self::isXMLContentValid($xmlContent, $version, $encoding);

    	return $valid;
    }
    
    /**
     * Check that given file has a valid XML content
     * 
     * @param string $xmlContent A well-formed XML string
     * @param string $version 1.0
     * @param string $encoding utf-8
     * @return bool
     */
    public static function isXMLContentValid($xmlContent, $version = '1.0', $encoding = 'utf-8')
    {
    	$xmlValid = TRUE;
    	if (trim($xmlContent) == '') 
    	{
    		return false;
    	}
    	
    	libxml_use_internal_errors(true);
    	
    	$doc = new \DOMDocument($version, $encoding);
    	$doc->loadXML($xmlContent);
    	
    	$errors = libxml_get_errors();
    	libxml_clear_errors();
    	
    	$xmlValid = empty($errors);
    	if ( $xmlValid )
    	{
    		$xmlValid = (stripos($xmlContent, self::PHP_STRING) === FALSE) && (stripos($xmlContent, self::HTML_STRING) === FALSE) ;
    	}
    	
    	return $xmlValid;
    }    
    
    static function stripNonXmlDoc($xml)
    {
    }
    
    
}
?>