<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\services\autotest;


use app\util\SMTStatusDto;

use app\serviceshelper\autotest\SMTAutotest;

use app\services\autotest\otu\SMTOtuAutotestDto;

use app\util\SMTLogger;

use app\settings\SMTSmartOtuSettings;

use app\parser\SMTOtuApi;
use app\services\SMTIService;
use app\services\SMTService;
use app\services\security\SMTAccessControl;

use app\view\SMTTestRestApiPage;
use app\util\SMTIOException;
use app\view\SMTFileDownloadPage;
use app\error\SMTAccessForbiddenRestException;


/**
 * Service to retrieve OTU autotest.
 *
 * @author Sylvain Desplat
 */
class SMTAutotestService extends SMTService implements SMTIService
{        
    /**
     *
     * @url POST /all
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchAutotests()
    {
        $autotestListDto = new SMTAutotestListDto();
        try
        {
            $autotestHelper = new SMTAutotest();
            $autotestHelper->setContext( $this->getContext() );
            $autotests = $autotestHelper->fetchAutotests();
            
            $autotestListDto->setAutotestList($autotests);

            $autotestSetup = new SMTAutotestSetupDto();
            $autotestDateMinutes = intval( $autotestHelper->getAutotestDateInDayInMinutes() );
            $autotestSetup->setAutotestDateMinutes($autotestDateMinutes);
            $autotestEnabled = $autotestHelper->isAutotestOn();
            $autotestSetup->setEnabled( $autotestEnabled );
            
            $autotestListDto->setAutotestSetup( $autotestSetup );
        }
        catch ( \Exception $e )
        {
            $this->handleException($e);
        }
        
        return $autotestListDto->getJsonData();
    }
    
    /**
     *
     * @url PUT /
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function updateAutotestSettings($request_data)
    {
        try
        {
            $autotestSetUpDto = SMTAutotestSetupDto::getInstance( $request_data );
            
            $autotestHelper = new SMTAutotest();
            $autotestHelper->setContext( $this->getContext() );
            $autotestHelper->updateAutotestSettings( $autotestSetUpDto );

            //fetch autotest new values:
            $autotestDateMinutes = $autotestHelper->getAutotestDateInDayInMinutes();
            $autotestSetUpDto->setAutotestDateMinutes( intval( $autotestDateMinutes) );
            $autotestEnabled = $autotestHelper->isAutotestOn();
            $autotestSetUpDto->setEnabled( $autotestEnabled );
        }
        catch ( \Exception $e )
        {
            $this->handleException($e);
        }
        
        return $autotestSetUpDto->getJsonData();
    }
    
    /**
     *
     * @url PUT /start
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function startAutotest()
    {
        $result = new SMTStatusDto();
    	try
    	{
    	    $this->getContext()->getLogger()->trace("Start manual autotest..." );
    	    $this->send( SMTOtuApi::startManualAutotestCommand() );
    	    $result->setStatusOK();
    	}
    	catch ( \Exception $e )
    	{
            $result->setStatusKO();
            $this->handleException( $e );
    	}
    
    	return $result;
    }
    
    /**
     *
     * @url POST /
     * @status 200
     *
     * Access granted to everybody (used by OTU application), but control on host name sending the request: only loopback address allowed.
     */
    function sendAutotest( $request_data )
    {
        $result = new SMTStatusDto();
    	try
    	{
    		$request = $this->getContext()->getHttpRequest();
    		   		
    		if ( $request->isLocalHTTPQuery() || SMTSmartOtuSettings::isDebugMode() )
    		{
    			$this->getContext()->getLogger()->trace("receiving autotest report..." );
    			$this->getContext()->getLogger()->trace( print_r( $request_data, true ),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    		
    			$autotestOtuDto = SMTOtuAutotestDto::getInstance( $request_data );
    			$this->getContext()->getLogger()->trace( "DTO autotest:".print_r( $autotestOtuDto, true ),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );

    			$autotestHelper = new SMTAutotest();
    			$autotestHelper->setContext( $this->getContext() );
    			$activityDto = $autotestHelper->processAutotest( $autotestOtuDto );
    			$result->setStatusOK();
    		}
    		else
    		{
    			$this->getContext()->getLogger()->trace( sprintf("Remote host %s attempt to send autotest: forbidden!",$request->getClientIpAddress() ), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    			throw new SMTAccessForbiddenRestException( MSG_ERROR_REMOTE_HOST_ATTEMPT_TO_SEND_DATA, "Host: ".$request->getClientIpAddress() );
    		}
    	}
    	catch ( \Exception $e )
    	{
            $result->setStatusKO();
            $this->handleException( $e );
    	}
    
    	return $result->getJsonData();
    }    
 
    /**
     * Request to download the autotest summary file
     *
     * @url GET /download/
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function downloadAutotestSummaryFile( $request_data )
    {
    	try
    	{
    		$autotestSummaryTimeStamp= $request_data['autotestSummaryTimeStamp'];
    		
    		//fetch autotest file name
    		$autotestHelper = new SMTAutotest();
    		$autotestHelper->setContext( $this->getContext() );
    		$autotest = $autotestHelper->fetchAutotest($autotestSummaryTimeStamp);
    		if ( $autotest != null )
    		{
	    		$autotestSummaryFileName = $autotest->getAutotestSummaryFileName();
	    		
	    		if ( ($autotestSummaryFileName!= NULL) && file_exists($autotestSummaryFileName) )
	    		{
	    			$page = new SMTFileDownloadPage($autotestSummaryFileName);
	    			$this->getContext()->getHttpResponse()->send( $page );
	    		}
	    		else
	    		{
	    			$e = new SMTIOException( SMTIOException::FILE_NOT_FOUND );
	    			$this->handleException($e);
	    		}
    		}
    		else
    		{
    			$e = new SMTIOException( SMTIOException::FILE_NOT_FOUND );
    			$this->handleException($e);
    		}
    	}
    	catch( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    }
        
    /**
     *
     * @url GET /test/fetchall
     */
    function testFetchAutotests()
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$url = '/autotest/all';
        	$page = new SMTTestRestApiPage($url, "", 'POST');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }
}