<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of VIAVI Solutions
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from VIAVI Solutions
// Copyright JDSU 2017. All rights reserved.
// *********************************************************
namespace app\services\alarm;



use app\database\SMTIExecuteInTransaction;

use app\database\SMTSmartOtuDB;

use app\serviceshelper\alarm\SMTAlarmSpecificProblemCode;

/**
 * Peak optical alarm Dto
 * 
 * @author Sylvain Desplat
 * @SMTClassTableNameAnnotation(tablename='optical_alarm', tablekey='id') 
 */
class SMTPeakOpticalAlarmDto extends SMTOpticalAlarmDto
{    
    /**
     * @SMTAttributeInfosAnnotation(classname='app\services\alarm\SMTPeakOpticalAlarmEventDto', id='id', tablename='peak_optical_alarm_event')
     * @var SMTPeakOpticalAlarmEventDto
     */
    protected $currentAlarmEvent;
        
    const FETCH_PEAK_OPTICAL_ALARM = "SELECT optical_alarm.* FROM optical_alarm, peak_optical_alarm_event WHERE optical_alarm.id=:id AND optical_alarm.current_alarm_event_id=peak_optical_alarm_event.id";
    
    const FETCH_PEAK_OPTICAL_ALARM_BY_DESCRIMINATOR = "SELECT optical_alarm.* FROM optical_alarm, peak_optical_alarm_event WHERE discriminator=:alarmDiscriminator AND optical_alarm.current_alarm_event_id=peak_optical_alarm_event.id";
    
    const FETCH_PEAK_OPTICAL_ALARM_BY_TESTID = "SELECT optical_alarm.* FROM optical_alarm, peak_optical_alarm_event WHERE test_id=:testId AND optical_alarm.current_alarm_event_id=peak_optical_alarm_event.id";
    
    const FETCH_ALL_PEAK_OPTICAL_ALARM = "SELECT optical_alarm.* FROM optical_alarm, peak_optical_alarm_event WHERE optical_alarm.current_alarm_event_id=peak_optical_alarm_event.id";
    
    /**
     * Fetch the peak optical alarm for the given alarm descriptor
     *
     * @param \SQLite3 $dbConnection
     * @param string $alarmDiscriminator
     * @param boolean $joinFetch FALSE by default: TRUE to force the retrieval of joined properties (optical alarm last event)
     *
     * @return SMTPeakOpticalAlarmDto
     */
    public static function fetch( \SQLite3 $dbConnection, $alarmDiscriminator, $joinFetch = FALSE )
    {
        $query=self::FETCH_PEAK_OPTICAL_ALARM_BY_DESCRIMINATOR;
        $statement = $dbConnection->prepare( $query);
        
        self::handleStatementError( $dbConnection, $statement, "Prepare select", $query, NULL, __FILE__,__METHOD__,__LINE__ );
        
        $bindParameters = ':alarmDiscriminator=>'.$alarmDiscriminator;
        $statement->bindValue(':alarmDiscriminator', $alarmDiscriminator, SQLITE3_TEXT );
        $resultSet = $statement->execute();
        
        self::handleStatementError( $dbConnection, $resultSet, "Select", $query, $bindParameters, __FILE__,__METHOD__,__LINE__ );
        
    	$opticalAlarm = parent::getSingleInstanceFromResultSet($resultSet, get_class() );
    	
    	//translates alarm specific problem
    	if ( $opticalAlarm != NULL )
    	{
    	    $opticalAlarm->setSpecificProblemText( SMTAlarmSpecificProblemCode::decodeOpticalAlarmSpecificProblem( $opticalAlarm->getSpecificProblemCode(), $opticalAlarm->isCleared() ) );
    	}
    	
    	if ( $opticalAlarm != NULL && $opticalAlarm instanceof SMTPeakOpticalAlarmDto)
    	{
    	    self::decodeSpecificProblemAndJoinFetchAlarmEvent($dbConnection, $opticalAlarm, $joinFetch );
    	}
    	
    	return $opticalAlarm;
    }       
    
    /**
     * Fetch the peak optical alarm for the given testId
     *
     * @param \SQLite3 $dbConnection
     * @param string $testId
     * @param boolean $joinFetch FALSE by default: TRUE to force the retrieval of joined properties (optical alarm last event)
     *
     * @return array of SMTPeakOpticalAlarmDto or NULL if no alarm found
     */
    public static function fetchFromTest( \SQLite3 $dbConnection, $testId, $joinFetch = FALSE )
    {
    	$opticalAlarms = array();
    	
    	$query=self::FETCH_PEAK_OPTICAL_ALARM_BY_TESTID;
    	$statement = $dbConnection->prepare( $query);
    
    	self::handleStatementError( $dbConnection, $statement, "Prepare select", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    
    	$bindParameters = ':testId=>'.$testId;
    	$statement->bindValue(':testId', $testId, SQLITE3_TEXT );
    	$resultSet = $statement->execute();
    
    	self::handleStatementError( $dbConnection, $resultSet, "Select", $query, $bindParameters, __FILE__,__METHOD__,__LINE__ );
    
    	$opticalAlarms = parent::getListFromResultSet($resultSet, get_class() );
    	
    	foreach ( $opticalAlarms as &$alarm )
    	{
    		//translates alarm specific problem
    		if ( $alarm != NULL )
    		{
    			self::decodeSpecificProblemAndJoinFetchAlarmEvent($dbConnection, $alarm, $joinFetch );
    		}
    	}
    	 
    	return $opticalAlarms;
    }    
    
    /**
     * Fetch the peak optical alarm with its event for the given alarm id
     *
     * @param \SQLite3 $dbConnection
     * @param int $id alarm id
     *
     * @return SMTPeakOpticalAlarmDto
     */
    public static function joinFetch( \SQLite3 $dbConnection, $id )
    {
    	$query=self::FETCH_PEAK_OPTICAL_ALARM;
    	$statement = $dbConnection->prepare( $query);
    
    	self::handleStatementError( $dbConnection, $statement, "Prepare select", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    
    	$bindParameters = ':id=>'.$id;
    	$statement->bindValue(':id', $id, SQLITE3_TEXT );
    	$resultSet = $statement->execute();
    
    	self::handleStatementError( $dbConnection, $resultSet, "Select", $query, $bindParameters, __FILE__,__METHOD__,__LINE__ );
    
    	$opticalAlarm = parent::getSingleInstanceFromResultSet($resultSet, get_class() );
    	 
    	if ( $opticalAlarm != NULL )
    	{
    	    self::decodeSpecificProblemAndJoinFetchAlarmEvent($dbConnection, $opticalAlarm, TRUE );
    	}
    	 
    	return $opticalAlarm;
    }    
    
    /**
     * Fetch all the peak optical alarms
     *
     * @param \SQLite3 $dbConnection
     * @param boolean $joinFetch FALSE by default: TRUE to force the retrieval of joined properties (optical alarm last event)
     *
     * @return array SMTPeakOpticalAlarmDto
     */
    public static function fetchAll( \SQLite3 $dbConnection, $joinFetch = FALSE )
    {
        $opticalAlarms = array();
        
    	$query=self::FETCH_ALL_PEAK_OPTICAL_ALARM;
    	$statement = $dbConnection->prepare( $query);
    
    	self::handleStatementError( $dbConnection, $statement, "Prepare select", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    
    	$resultSet = $statement->execute();
    
    	self::handleStatementError( $dbConnection, $resultSet, "Select", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    
    	$opticalAlarms = parent::getListFromResultSet($resultSet, get_class() );
    	
    	foreach ( $opticalAlarms as &$alarm )
    	{
    	    //translates alarm specific problem
    	    if ( $alarm != NULL )
    	    {
                self::decodeSpecificProblemAndJoinFetchAlarmEvent($dbConnection, $alarm, $joinFetch );
    	    }
    	}
    	 
    	return $opticalAlarms;
    }    
    
    /**
     * 
     * @param \SQLite3 $dbConnection
     * @param SMTPeakOpticalAlarmDto $opticalAlarm
     * @param string $joinFetch FALSE by default: TRUE to force the retrieval of joined properties (optical alarm last event)
     */
    private static function decodeSpecificProblemAndJoinFetchAlarmEvent( \SQLite3 $dbConnection, SMTPeakOpticalAlarmDto $opticalAlarm, $fetchAlarmEvent = FALSE)
    {
        $opticalAlarmEvent = NULL;
        
        //Join fetch peak optical alarm event
        if ( $fetchAlarmEvent )
        {
        	$opticalAlarmEvent = SMTPeakOpticalAlarmEventDto::fetch($dbConnection, $opticalAlarm->getId());
        	$opticalAlarm->setCurrentAlarmEvent( $opticalAlarmEvent );        	
        }
    }    
    
    /**
     * Save dto in database and save joined dtos (alarm event)
     *
     * @param SMTSmartOtuDB $dbConnection
     */
    public function save( SMTSmartOtuDB $dbConnection)
    {
    	$dbConnection->runInTransaction( new SMTSavePeakAlarm( $this ) );
    }
    
    /**
     * Save dto in database and save joined dtos (alarm event)
     * 
     * @param SMTSmartOtuDB $dbConnection
     * 
     */
    function performSavePeakAlarm( SMTSmartOtuDB $dbConnection )
    {
        parent::save($dbConnection );
        //update alarm event if it exists
        if ( $this->getCurrentAlarmEvent() != NULL )
        {
            $this->getCurrentAlarmEvent()->setAlarmId( $this->id );
            $this->getCurrentAlarmEvent()->save($dbConnection);
            
            //update current alarm event in alarm
            $this->updateSingleProperty( $dbConnection, "currentAlarmEvent" );                    
        }
    }
    
    /**
     *
     * @param SMTPeakOpticalAlarmEventDto $currentAlarmEvent
     */
    public function setCurrentAlarmEvent( SMTPeakOpticalAlarmEventDto $currentAlarmEvent )
    {
    	$this->currentAlarmEvent = $currentAlarmEvent;
    }
    /**
     *
     * @return SMTPeakOpticalAlarmEventDto
     */
    public function getCurrentAlarmEvent()
    {
    	return $this->currentAlarmEvent;
    }   
    
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }
}

/**
 * Save peak alarm in a transaction
 *
 * @author Sylvain Desplat
 */
class SMTSavePeakAlarm implements SMTIExecuteInTransaction
{
	/**
	 *
	 * @var SMTPeakOpticalAlarmDto
	 */
	var $alarm;

	/**
	 *
	 * @param SMTAlarmDto $alarm
	 */
	function __construct( SMTPeakOpticalAlarmDto $alarm )
	{
		$this->alarm = $alarm;
	}

	/**
	 *
	 */
	public function run( SMTSmartOtuDB $dbConnection )
	{
		$this->alarm->performSavePeakAlarm($dbConnection);
	}

	/**
	 * @return SMTResultSetDto
	 */
	public function getResultSetDto()
	{
		return $this->alarm;
	}
}
?>