<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\util;

use app\settings\SMTSmartOtuSettings;

date_default_timezone_set('UTC');

/**
 * Utility file to log parser logs in /var/log/otu_query 
 * and other logs and errors to /var/log/smart_otu file
 * 
 * @author Sylvain Desplat
 */
class SMTLogger
{
    const LOG_DIR = "/var/log/";   
    const QUERY_LOG_FILE = "otu_query.log";
    const SMART_OTU_LOG_FILE = "smart_otu.log";
    const DEBUG = "debug";
    const INFO = "info";
    const ERROR = "error";
    const PINFO = "pinfo";
    
    private $queryLogFileHandle;
    private $logFileHandle;
    private $logLevel;
    private static $loggerInstance = NULL;
    
    private function __construct()
    {
        $this->queryLogFileHandle = ( ( $queryHandle = fopen( self::getQueryLogFilePath(),"a") ) !== false )? $queryHandle : NULL;
        $this->logFileHandle = ( ( $handle = fopen( self::getLogFilePath(),"a" ) ) !== false )? $handle : NULL;
        $this->logLevel = SMTSmartOtuSettings::getLogLevel();
    }
    
    function __destruct()
    {
        if (isset ($this->queryLogFileHandle) )
        {
            fclose( $this->queryLogFileHandle );
            $this->queryLogFileHandle = NULL;
        }        
        
        if (isset ($this->logFileHandle) )
        {
            fclose( $this->logFileHandle );
            $this->logFileHandle = NULL;
        }
    }
    
    /**
     * Returns log manager.
     * 
     * @return \app\util\SMTLogger
     */
    public static function getInstance()
    {
        if ( self::$loggerInstance == NULL )
        {
            self::$loggerInstance = new SMTLogger();
        }
        return self::$loggerInstance;
    }    
    
    /**
     * Log message in OTU query file
     *
     * @param string $message
     * @param string $level by default trace level = self::INFO
     * @param string $queryDate The date of the query execution (used for logs) or NULL
     * @param $resultMessage string the result of the query (NULL for a command)
     */
    function traceOtuQuery( $message, $level = self::INFO, $queryDate= NULL, $resultMessage = NULL )
    {
    	if ( ($level == self::DEBUG) && ($this->logLevel != self::DEBUG) )
    	{
    		return;
    	}
    	
    	//can append if that method is called after destructor has been invoked
    	if ( $this->queryLogFileHandle == NULL)
    	{
    		$templogFileHandle = fopen( self::getQueryLogFilePath(),"a");
    		if ( $templogFileHandle !== FALSE )
    		{
    			fwrite( $templogFileHandle, self::formatQueryMessage( $message, $level, $queryDate, $resultMessage) );
    			fclose( $templogFileHandle );
    		}
    	}
    	else
    	{
    		fwrite( $this->queryLogFileHandle, self::formatQueryMessage( $message, $level, $queryDate, $resultMessage) );
    	}
    }
    
    
    /**
     * Log message in log file
     *
     * @param string $message
     * @param string $level by default trace level = self::INFO 
     * @param string $phpFileName The php file name. NULL by default
     * @param string $methodName The php method. NULL by default
     * @param string $lineNumber The php line of code. NULL by default
     */
    function trace( $message, $level = self::INFO, $phpFileName = NULL, $methodName = NULL, $lineNumber = NULL )
    {
        $level = self::checkLogLevel( $level );
        
        if ( ($level == self::ERROR) || ($level == self::PINFO) ||
             ( $level == self::INFO && ( $this->logLevel == self::INFO || $this->logLevel == self::DEBUG ) ) ||
             ( $level == self::DEBUG && $this->logLevel == self::DEBUG)
                )
        {
            //can append if that method is called after destructor has been invoked
            if ( $this->logFileHandle == NULL)
            {
            	$templogFileHandle = fopen( self::getLogFilePath(),"a");
            	if ( $templogFileHandle !== FALSE )
            	{
            		fwrite( $templogFileHandle, self::formatMessage( $message, $level, $phpFileName, $methodName, $lineNumber ) );
            		fclose( $templogFileHandle );
            	}
            
            }
            else
            {
        	    fwrite( $this->logFileHandle, self::formatMessage( $message, $level, $phpFileName, $methodName, $lineNumber ) );
            }
        }
    }
    
    /**
     * EPT 6775 Hide function parameters in stack trace
     * 
     * @param string $stackTrace
     * @return string Stack trace without parameters
     */
    static function filterStackTrace($stackTrace)
    {
    	return preg_replace('/(?m)^(.*?:.*?)\(.*?$/', '${1}()',$stackTrace);
    }
        
    /**
     * Log exception in log file 
     *
     * @param \Exception $e
     */
    function traceException( \Exception $e )
    {
        if ( $e instanceof SMTSocketException || ( ($e instanceof SMTException) && ($e->getDetailedMessage() != NULL) ) )
        {
        	$message = $e->getMessage().": ".$e->getDetailedMessage();
        }
        else
        {
        	$message = $e->getMessage();
        }
        
        //can append if that method is called after destructor has been invoked
        if ( $this->logFileHandle == NULL)
        {
        	$templogFileHandle = fopen( self::getLogFilePath(),"a");
        	if ( $templogFileHandle !== FALSE )
        	{    	        
                fwrite( $templogFileHandle, self::formatMessage( $message, self::ERROR ) );
                $stackTrace = self::filterStackTrace($e->getTraceAsString());
                fwrite( $templogFileHandle, $stackTrace."\n" );
        		fclose( $templogFileHandle );
        	}        
        }
        else
        {                
            fwrite( $this->logFileHandle, self::formatMessage( $message, self::ERROR ) );
            $stackTrace = self::filterStackTrace($e->getTraceAsString());
            fwrite( $this->logFileHandle, $stackTrace."\n" );
        }
    }    
    
    /**
     * Whether log level is set to DEBUG
     *
     * @return boolean
     */
    public static function isDebugLogLevel()
    {
    	return self::getInstance()->logLevel == self::DEBUG;
    }    
    
    /**
     * Date string in "absolute time": not in UTC
     * 
     * @return string
     */
    public static function getDate()
    {
        return date("D j H:i:s");
    }
    
    /**
     * Format log message
     * 
     * @param string $message
     * @param string $level level can be self::INFO or self::PINFO or self::ERROR or self::DEBUG  
     * @param $phpFileName The php file name. NULL by default
     * @param $methodName The php method. NULL by default
     * @param $lineNumber The php line of code. NULL by default
     * @param string $date The date to put on logs or NULL for current date
     * 
     * @return string
     */
    public static function formatMessage( $message, $level, $phpFileName = NULL, $methodName = NULL, $lineNumber = NULL)
    {
        $levelString = "[".strtoupper( self::checkLogLevel( $level ) )."]";
        $fileMethodString = ( isset( $methodName)? " Method: ".$methodName : (isset( $phpFileName)? " File: ".$phpFileName : "" ) );
        $lineNumberString = ( isset( $lineNumber)? " Line: ".$lineNumber : "" );
        
        return self::getDate()."\t".$levelString.$fileMethodString.$lineNumberString."\t".trim( $message )."\n";
    }

    /**
     * Format query message
     *
     * @param string $message
     * @param string $level level can be self::INFO or self::PINFO or self::ERROR or self::DEBUG
     * @param string $queryDate The date to put on logs or NULL for current date
     * @param string $resultMessage result of the query
     * 
     * @return string
     */
    public static function formatQueryMessage( $message, $level, $queryDate = NULL, $resultMessage = NULL)
    {
    	$levelString = "[".strtoupper( self::checkLogLevel( $level ) )."]";
    	$currentDate = self::getDate();
    	if ($resultMessage != NULL)
    	{
    		return (($queryDate== NULL )? $currentDate:$queryDate)."\t".$levelString."\t".trim( $message )."\n".$currentDate."\t".$levelString."\t".trim( $resultMessage)."\n";
    	}
    	else 
    	{
    		return $currentDate."\t".$levelString."\t".trim( $message )."\n";
    	}
    }
    
    /**
     * Check log level value. If $level value is invalid, uses level self::INFO
     * 
     * @param string $level
     * @return string If $level value is invalid, uses default level self::INFO.
     */
    private static function checkLogLevel( $level )
    {      
        if ( $level == self::INFO || $level == self::ERROR || $level == self::DEBUG || $level == self::PINFO )
        {
        	return $level;
        }
        return self::INFO;
    }
    
    /**
     * Returns the smart otu log file path
     * @return string
     */
    static function getLogFilePath()
    {
        return self::LOG_DIR.self::SMART_OTU_LOG_FILE;
    }
    
    /**
     * Returns the otu query log file path
     * @return string
     */
    static function getQueryLogFilePath()
    {
    	return  self::LOG_DIR.self::QUERY_LOG_FILE;
    }    
}



?>