<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\media;

use app\util\SMTInvalidValueException;

use app\serviceshelper\monitoring\SMTLinkTest;

use app\serviceshelper\system\SMTOtu;

use app\database\SMTSmartOtuDB;

use app\serviceshelper\system\SMTSystemConfigurationDto;

use app\serviceshelper\SMTServiceHelper;

use app\parser\SMTOtuApi;
use app\services\setup\SMTConfEmailDto;
use app\services\setup\SMTEmailDto;
use app\services\setup\SMTEmailPortFilterDto;
use app\services\setup\SMTTestEmailDto;
use app\util\SMTUtil;

include_once( "phputf8/ascii.php" );

define("MSG_TEST_EMAIL","Test sent from OTU: %s");

/**
 *
 * @author Sylvain Desplat
 *
 */
class SMTEmail extends SMTServiceHelper
{    
    /**
     * Email media.
     *
     * @var string
     */
    const EMAIL_MEDIA = 'EMAIL';    
    /**
     * Email media for smartOtu is limited to ethernet.
     *
     * @var string
     */
    const EMAIL_MEDIA_ETH = 'ETH';
    
    const MIN_EMAIL_INDEX = 1;
    
    const MAX_EMAIL_INDEX = 16;
    
    const DEFAULT_JDSU_VIAVI_EXTENSION = '@viavisolutions.com';
    
    const DEFAULT_SMTP_PORT_NOT_SET = 0;
    const DEFAULT_SMTP_PORT = 25;
    const DEFAULT_SMTP_SECURITY_NONE = 'NONE';
    const SMTP_SECURITY_SSLTLS = 'SSLTLS';
    const SMTP_SECURITY_STARTTLS = 'STARTTLS';

    const SMTP_SEPARATOR = '::';
    const EMAIL_SENT_SUCCESSFULLY = 'Completed';    
    const EMAIL_SENT_FAILED = 'SMTP error';
    

    const ADDRESS_FROM_INDEX = 3;
    const ADDRESS_TO_INDEX = 7;
    const SUBJECT_INDEX = 8;
    const MEDIA_INDEX = 0;
    const LOGIN_INDEX = 1;
    const PASSWORD_INDEX = 2;
    const SMTP_INDEX= 4;
    const SMTP_PORT_INDEX= 5;
    const SMTP_SECURITY_PROTOCOL_INDEX= 6;
    const ATTACH_FILE_INDEX = 9;
    const PORT_FILTER = 10;
    const PORT_FILTER_BUFFER = 11;
    
    const EXIM_LOG_FILE_PATH = '/var/log/exim/mainlog';
	
    /**
     * Create empty Dto Email to be sent to the client
     * @return SMTEmailDto
     */
    function newEmail()
    {
		$emailDto = new SMTEmailDto();
		$emailDto->setValid( FALSE);
		$hostName = $this->sendReceive( SMTOtuApi::CMD_get_otu_name );
		$asciiName = $hostName;
			
		if ( !utf8_is_ascii( $hostName ) )
		{
			//turn latin characters with accents to ascii equivalent
			$asciiName = utf8_accents_to_ascii( $hostName );
				
			//if asciiName is not an ascii string, strip non ascii characters and controls
			if ( !utf8_is_ascii( $asciiName ) )
			{
				$asciiName = utf8_strip_non_ascii_ctrl($asciiName);
			}
		}
		// Removes all special chars
		$asciiName = preg_replace('/[^A-Za-z0-9\_\.\-]/', '', $asciiName);
		
		$emailDto->setAddressFrom( $asciiName.self::DEFAULT_JDSU_VIAVI_EXTENSION );
		//Language for OTU alerts is limited to english; don't retrieve the "Alarm" string from resources.
		$emailDto->setSubject( 'Alarm '.$hostName  );
		
		$portFilter = new SMTEmailPortFilterDto();
		$emailDto->setPortFilter( $portFilter );
		
		//retrieve the number of ports available:
		$linkHelper = new SMTLinkTest();
		$linkHelper->setContext( $this->getContext() );		
		$portCount = $linkHelper->getPortsCount();
		$portFilter->setPortCount($portCount);

    	return $emailDto;
    }
    
    /**
     * Update emails config on OTU and in database
     * 
     * @param SMTConfEmailDto $emailConfDto
     * @param SMTSmartOtuDB $dbConnection
     */
    function updateEmails( SMTConfEmailDto $emailConfDto, SMTSmartOtuDB $dbConnection )
    {
        $emailsIndexToRemove = self::MIN_EMAIL_INDEX;         
        
        
        //update emails in OTU
        foreach ( $emailConfDto->getEmails() as $emailDto )
        {
            $emailsIndexToRemove++;
            $portFilter = $emailDto->getPortFilter();
            
            if (self::isEmailSMTPSecurityProtocolValid( $emailConfDto->getSmtpSecurityProtocol()))
                {
                $command = SMTOtuApi::setEmailWithAttachedFileCommand( $emailDto->getIndex(), $emailConfDto->getLogin(), $emailConfDto->getPassword(), $emailDto->getAddressFrom(), $emailConfDto->getSmtpServer(), $emailConfDto->getSmtpPort(), $emailConfDto->getSmtpSecurityProtocol(), $emailDto->getAddressTo(), $emailDto->getSubject(), $emailConfDto->isAttachFile(), $portFilter->hasFilter(), $portFilter->getPortBuffer() );
        	    $this->send( $command );
                }
             else 
             {
                 throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "smtpSecurityProtocol", $smtpSecurityProtocol );
             }
        }
        
        for ( $index = $emailsIndexToRemove; $index <= self::MAX_EMAIL_INDEX; $index++ )
        {
            $command = SMTOtuApi::setEmailWithAttachedFileCommand( $index, "", "", "", "", self::DEFAULT_SMTP_PORT_NOT_SET, self::DEFAULT_SMTP_SECURITY_NONE, "", "", FALSE, FALSE,SMTEmailPortFilterEncoderDecoder::EMPTY_EMAIL );
            $this->send( $command );
        }
        
        $updateSmtpServerStoredInDatabase = FALSE;
        
        //retrieve SMTP server config from database
        $systemConfiguration = SMTSystemConfigurationDto::fetch( $dbConnection );
        
        //update smtp server settings in database
        //update attachFile in database if it has changed
        if ( $systemConfiguration->isSmtpAttachFile()!= $emailConfDto->isAttachFile() )
        {
        	$updateSmtpServerStoredInDatabase = TRUE;
        	$systemConfiguration->setSmtpAttachFile( $emailConfDto->isAttachFile() );
        }
        //update login in database if it has changed
    	if ( strcmp( $systemConfiguration->getSmtpLogin(), $emailConfDto->getLogin() ) != 0 )
    	{
    		$updateSmtpServerStoredInDatabase = TRUE;
    		$systemConfiguration->setSmtpLogin( $emailConfDto->getLogin() );
    	}
    	//update password in database if it has changed
    	if ( strcmp( $systemConfiguration->getSmtpPassword(), $emailConfDto->getPassword() ) != 0 )
    	{
    		$updateSmtpServerStoredInDatabase = TRUE;
    		$systemConfiguration->setSmtpPassword( $emailConfDto->getPassword() );
    	}
        //update smtp server in database if it has changed
        //smtp server can be null to use exim in automatic mode
    	if ( strcmp( $systemConfiguration->getSmtpServer(), $emailConfDto->getSmtpServer() ) != 0 )
    	{
    		$updateSmtpServerStoredInDatabase = TRUE;
    		$systemConfiguration->setSmtpServer( $emailConfDto->getSmtpServer() );
    	}
    	
    	if ( $systemConfiguration->getSmtpPort() != $emailConfDto->getSmtpPort() )
    	{
    		$updateSmtpServerStoredInDatabase = TRUE;
    		$systemConfiguration->setSmtpPort( $emailConfDto->getSmtpPort() );
    	}

    	if ( $systemConfiguration->getSmtpSecurityProtocol() != $emailConfDto->getSmtpSecurityProtocol() )
    	{
    		$updateSmtpServerStoredInDatabase = TRUE;
    		$systemConfiguration->setSmtpSecurityProtocol( $emailConfDto->getSmtpSecurityProtocol() );
    	}    	
        
        if ( $updateSmtpServerStoredInDatabase )
        {
        	$systemConfiguration->save( $dbConnection );
        }
    }
    
    /**
     * Build Email test message 
     * @return string
     */
    function buildTestEmailMessage()
    {
    	$testString = sprintf(MSG_TEST_EMAIL, "");
        try 
        {
            $otu = new SMTOtu();
            $otu->setContext( $this->getContext() );
		    $serialNumber = $otu->getOtuSerialNumber();
		    $testString = sprintf(MSG_TEST_EMAIL, $serialNumber);
        }
        //catch exception: continue to sent if the OTU name retrieval failed
        catch( \Exception $e )
        {
            $this->getContext()->getLogger()->traceException($e);
        }
        return $testString;        
    }
    
    /**
     * Fetch Emails configurations
     *
     * @param SMTSmartOtuDB $dbConnection
     */
    function retrieveEmails( SMTSmartOtuDB $dbConnection = NULL )
    {
    	$emailConfDto = new SMTConfEmailDto();
    	$closeDbConnection = FALSE;
    	
    	try 
    	{
	    	if ( $dbConnection == NULL )
	    	{
	    	    $dbConnection = $this->getContext()->getDatabase();
			    $closeDbConnection = TRUE;
	    	}
	
			//retrieve email settings from database
			$systemConfiguration = SMTSystemConfigurationDto::fetch( $dbConnection );
    	}
    	catch(\Exception $e)
    	{
    		$this->getContext()->getLogger()->traceException($e);
    		//create default system configuration
    		$systemConfiguration = new SMTSystemConfigurationDto();
    	}

		//retrieve the number of ports available:
		$linkHelper = new SMTLinkTest();
		$linkHelper->setContext( $this->getContext() );
		$portCount = $linkHelper->getPortsCount();
		
		for( $index=SMTEmail::MIN_EMAIL_INDEX; $index<=SMTEmail::MAX_EMAIL_INDEX; $index++)
		{
    		$emailDto = $this->retrieveEmail( $index, $emailConfDto, $portCount, $systemConfiguration, $dbConnection );
    		if ( $emailDto != NULL )
    		{
    		    $emailConfDto->addEmail( $emailDto );
    		}
    		else
    		{
    		    //don't fetch more emails when an empty email is found
    		    break;
    		}
		}
    			 
		$emailConfDto->setActive( $this->isEmailMediaOn() );
		if ( $closeDbConnection )
		{
		    $dbConnection->close();
		}
    
    	return $emailConfDto;
    	}    
    
	/**
	 * Fetch an email with the given index and fill additional infos (login, passord, smtp server) if not already set.
	 *
	 * @param string $emailIndex
	 * @param \app\services\setup\SMTConfEmailDto $emailConfDto
	 * @param SMTSystemConfigurationDto $systemConfiguration
	 * @param SMTSmartOtuDB $dbConnection
	 * 
	 * @return \app\services\setup\SMTEmailDto
	 */
	function retrieveEmail( $emailIndex, SMTConfEmailDto $emailConfDto, $portCount, SMTSystemConfigurationDto $systemConfiguration, SMTSmartOtuDB $dbConnection = NULL )
	{
	    $emailDto = NULL;
	    $updateSmtpServerStoredInDatabase = FALSE;
	    $emailString = $this->sendReceive( SMTOtuApi::getEmailWithAttachedFileCommand( $emailIndex ), FALSE );		 	   
	    
		//look for  ,"  sequence to explode the string into an array $conf
	    //remove spaces in array values
	    $confArray = SMTUtil::splitStringByComma( $emailString );
			
		$isEmailConfigValid = self::isEmailConfigValid( $confArray );
		if ( $isEmailConfigValid )
		{
    		$emailDto = new SMTEmailDto();
    		$emailDto->setValid( $isEmailConfigValid );
    		$emailDto->setIndex( intval($emailIndex) );
    		$emailDto->setAddressFrom( $confArray[self::ADDRESS_FROM_INDEX] );
    		$emailDto->setAddressTo( $confArray[self::ADDRESS_TO_INDEX] );
    		$emailDto->setSubject( $confArray[self::SUBJECT_INDEX] );
    		
    		//email port filter
    		$portFilter = new SMTEmailPortFilterDto();    		    		
    		$filter = (strcmp($confArray[self::PORT_FILTER], SMTOtuApi::RES_YES) == 0);
    		$portFilter->setHasFilter($filter);
    		$portFilter->setPortBuffer($confArray[self::PORT_FILTER_BUFFER], $portCount);
    		
    		$emailDto->setPortFilter( $portFilter );
		}						
		
		//update attachFile in database if it has changed
		if ( $isEmailConfigValid && isset( $confArray[self::ATTACH_FILE_INDEX] ) && 
		        strcmp( $confArray[self::ATTACH_FILE_INDEX], $emailConfDto->isAttachFile()? SMTOtuApi::RES_YES: SMTOtuApi::RES_NO ) != 0 )
		{
		    $attachFile = strcmp($confArray[self::ATTACH_FILE_INDEX], SMTOtuApi::RES_YES) == 0;
		    $emailConfDto->setAttachFile( $attachFile );
		    if ( strcmp( $systemConfiguration->isSmtpAttachFile()? SMTOtuApi::RES_YES: SMTOtuApi::RES_NO, $confArray[self::ATTACH_FILE_INDEX]) != 0 )
		    {
//     		    SMTLogger::getInstance()->trace("systemConfiguration->isSmtpAttachFile()");
    			$updateSmtpServerStoredInDatabase = TRUE;
    			$systemConfiguration->setSmtpAttachFile( $emailConfDto->isAttachFile() );
		    }
		}		
		//update dto from database if no data is retrieved from OTU
		else
		{
			$emailConfDto->setAttachFile( $systemConfiguration->isSmtpAttachFile() );
		}
		
		//update dto if it has not already been set
		if($emailConfDto->getLogin()=='')
		{
    		//login retrieved from OTU, update it in DTO and database if it has changed
    		if ( $isEmailConfigValid && isset( $confArray[self::LOGIN_INDEX] ) && $confArray[self::LOGIN_INDEX] != '' )
    		{
    			$emailConfDto->setLogin( $confArray[self::LOGIN_INDEX] );
    			if ( strcmp( $systemConfiguration->getSmtpLogin(), $confArray[self::LOGIN_INDEX]) != 0 )
    			{
//     			    SMTLogger::getInstance()->trace("systemConfiguration->getSmtpLogin()");
    			    $updateSmtpServerStoredInDatabase = TRUE;
    			    $systemConfiguration->setSmtpLogin( $confArray[self::LOGIN_INDEX] );
    			}
    		}
    		//update dto from database if no data is retrieved from OTU
    		else
    		{
    			$emailConfDto->setLogin( $systemConfiguration->getSmtpLogin() );
    		}
    	}
		//update dto if it has not already been set
		if ( $emailConfDto->getPassword()=='' )
		{
		    //smtp password retrieved from OTU, update it in DTO and database if it has changed
    		if ( $isEmailConfigValid && isset( $confArray[self::PASSWORD_INDEX] ) && $confArray[self::PASSWORD_INDEX] != '' )
    		{
    			$emailConfDto->setPassword( $confArray[self::PASSWORD_INDEX] );
    			if ( strcmp( $systemConfiguration->getSmtpPassword(), $confArray[self::PASSWORD_INDEX]) != 0 )
    			{
//     			    SMTLogger::getInstance()->trace("systemConfiguration->getSmtpPassword()");
    			    $updateSmtpServerStoredInDatabase = TRUE;
    				$systemConfiguration->setSmtpPassword( $confArray[self::PASSWORD_INDEX] );
    			}
    		}
    		//update dto from database if no data is retrieved from OTU
    		else
		    {
		    	$emailConfDto->setPassword( $systemConfiguration->getSmtpPassword() );
		    }		    
		}		 
		
		//update dto if it has not already been set
		if ( $emailConfDto->getSmtpServer()=='' )
		{
		    //smtp server retrieved from OTU, update it in DTO and database if it has changed
		    if ( isset( $confArray[self::SMTP_INDEX] ) && $confArray[self::SMTP_INDEX] != '' )
		    {
		    	$emailConfDto->setSmtpServer( $confArray[self::SMTP_INDEX] );
		    	if ( strcmp( $systemConfiguration->getSmtpServer(), $confArray[self::SMTP_INDEX]) != 0 )
		    	{
		    		$updateSmtpServerStoredInDatabase = TRUE;
// 		    		SMTLogger::getInstance()->trace("systemConfiguration->getSmtpServer()");
		    		$systemConfiguration->setSmtpServer( $confArray[self::SMTP_INDEX] );
		    	}
		    }
		    //update dto from database if no data is retrieved from OTU
		    else
		    {    		   
			    $emailConfDto->setSmtpServer( $systemConfiguration->getSmtpServer() );
		    }
		}
		
		//update dto if it has not already been set
		if ( ( $emailConfDto->getSmtpPort()== '' ) || $emailConfDto->getSmtpPort() == self::DEFAULT_SMTP_PORT_NOT_SET )
		{
			//smtp server port retrieved from OTU, update it in DTO and database if it has changed
			if ( isset( $confArray[self::SMTP_PORT_INDEX] ) && ($confArray[self::SMTP_PORT_INDEX] != '') && $confArray[self::SMTP_PORT_INDEX] > 0 )
			{
				$port = $confArray[self::SMTP_PORT_INDEX];
				if (isset($port) && SMTUtil::isInt($port) && $port > 0 )
				{				    
					$emailConfDto->setSmtpPort( $port );
				}
				else
				{
					$emailConfDto->setSmtpPort( self::DEFAULT_SMTP_PORT );
				}

				if ( strcmp( $systemConfiguration->getSmtpPort(), $emailConfDto->getSmtpPort() ) != 0 )
				{
// 				    SMTLogger::getInstance()->trace("systemConfiguration->getSmtpPort()");
					$updateSmtpServerStoredInDatabase = TRUE;
					$systemConfiguration->setSmtpPort( $confArray[self::SMTP_PORT_INDEX] );
				}
			}
			//update dto from database if no data is retrieved from OTU
			else
			{
				$emailConfDto->setSmtpPort( $systemConfiguration->getSmtpPort() );
			}
		}
		
		//update dto if it has not already been set
		if ( ( $emailConfDto->getSmtpSecurityProtocol()== '' ) || ($emailConfDto->getSmtpSecurityProtocol() == self::DEFAULT_SMTP_SECURITY_NONE) )
		{
			//smtp server port retrieved from OTU, update it in DTO and database if it has changed
			if ( isset( $confArray[self::SMTP_SECURITY_PROTOCOL_INDEX] ) && ($confArray[self::SMTP_SECURITY_PROTOCOL_INDEX] != '') && self::isEmailSMTPSecurityProtocolValid( $emailConfDto->getSmtpSecurityProtocol()) )
			{
				$securityProtocol = $confArray[self::SMTP_SECURITY_PROTOCOL_INDEX];
				if ( isset($securityProtocol) )
				{
					$emailConfDto->setSmtpSecurityProtocol( $securityProtocol );
				}
				else
				{
					$emailConfDto->setSmtpSecurityProtocol( self::DEFAULT_SMTP_SECURITY_NONE );
				}
		
				if ( strcmp( $systemConfiguration->getSmtpSecurityProtocol(), $emailConfDto->getSmtpSecurityProtocol() ) != 0 )
				{
					// 				    SMTLogger::getInstance()->trace("systemConfiguration->getSmtpPort()");
					$updateSmtpServerStoredInDatabase = TRUE;
					$systemConfiguration->setSmtpSecurityProtocol( $confArray[self::SMTP_SECURITY_PROTOCOL_INDEX] );
				}
			}
			//update dto from database if no data is retrieved from OTU
			else
			{
				$emailConfDto->setSmtpSecurityProtocol( $systemConfiguration->getSmtpSecurityProtocol() );
			}
		}		
		
		if ( $updateSmtpServerStoredInDatabase )
		{
			try 
			{
	// 		    SMTLogger::getInstance()->trace("systemConfiguration->save()");
			    $systemConfiguration->save( $dbConnection );
			}
			catch( \Exception $e )
			{
				$this->getContext()->getLogger()->traceException($e);
			}
		}		
		
		return $emailDto;
	}
		
	/**
	 * Retrieve exim logs.
	 * 
	 * @return string NULL, if logs couldn't be retrieved
	 */
	static function retrieveTestEmailLogs()
	{
		//to access exim log file, change exim dir exec flag: chmod 755 /var/log/exim
		//sleep 3 seconds before tryning to fetch logs		
		$count = 0;
		sleep(3);
		$timeStamp = filemtime( self::EXIM_LOG_FILE_PATH );
		
		//look for logs content up to 10 times
		//let 2s after the last file change because the log file is filled during a few hundred of ms
		while ( ( ( $timeStamp === FALSE ) || ( time() - $timeStamp > 2 ) ) && $count < 10 )
		{
			sleep(1);
			$timeStamp = filemtime( self::EXIM_LOG_FILE_PATH );
			$count++;
		}
		//read log file
		$content = file_get_contents( self::EXIM_LOG_FILE_PATH );
		
		return ($content !== FALSE)? $content : NULL;
	}
	
	/**
	 * Perform a Test of Email media: send an Email
	 * 
	 * @param $emailNumber Index of email to test
	 * 
	 * @return SMTTestEmailDto Test email result dto
	 */
	function testEmail( $emailNumber )
	{
		$emailConfDto = new SMTConfEmailDto();
		$testEmailDto = new SMTTestEmailDto();
		
		$dbConnection = $this->getContext()->getDatabase();
		//retrieve SMTP server config from database
		$systemConfiguration = SMTSystemConfigurationDto::fetch( $dbConnection );
		
		$emailDto = $this->retrieveEmail( $emailNumber, $emailConfDto, 1, $systemConfiguration, $dbConnection );
		$dbConnection->close();

		if ( $emailDto->isValid() )
		{
			$testString = $this->buildTestEmailMessage();
			
			$cmd_test_media = sprintf( SMTOtuApi::CMD_test_email, $emailNumber, self::EMAIL_MEDIA_ETH , $testString, $emailConfDto->getLogin(), $emailConfDto->getPassword(), $emailDto->getAddressFrom(), $emailConfDto->getSmtpServer(), $emailConfDto->getSmtpPort(), $emailConfDto->getSmtpSecurityProtocol(), $emailDto->getAddressTo(), $emailDto->getSubject() );
			$this->send( $cmd_test_media );

			//cancel EPT#5455 don't check exim mail logs to know whether mail was successfully sent 
			//to access exim log file, change exim dir exec flag: chmod 755 /var/log/exim 
// 			$eximLogContent = self::retrieveTestEmailLogs();
// 			if ( $eximLogContent != NULL )
// 			{
// 				$testEmailDto->setEximLogContent($eximLogContent);
// 				$testEmailDto->setSuccess( $eximLogContent != NULL
// 						&& (strpos($eximLogContent, self::EMAIL_SENT_SUCCESSFULLY) !== FALSE)
// 						&& (strpos($eximLogContent, self::EMAIL_SENT_FAILED) == FALSE)
// 				);
// 			}
// 			//if logs couldn't be retrieved, consider mail was sucessfully sent.
// 			//This can append when the mail has not yet been sent 
// 			//because the alarm has not yet been processed be the OTU "com" graph 
// 			else
// 			{
				$testEmailDto->setSuccess(TRUE);
			//}
		}
		else
		{
			throw new SMTMediaException( SMTMediaException::ERROR_INVALID_CONFIG );
		}
		
		return $testEmailDto;
	}	
	
	
	/**
	 *
	 * Set Email media ON or OFF
	 * @param boolean $on Whether email media must be activated or not
	 */
	function setEmailMedia( $on )
	{
		$this->send( SMTOtuApi::setEmailMedia( $on ) );
		//update media period and email max retry
		if ( $on )
		{
		    $this->send( SMTOtuApi::setMediaRetryPeriod( SMTOtuApi::MEDIA_RETRY_DELAY ) );
		    $this->send( SMTOtuApi::setEmailMaxRetry( SMTOtuApi::MEDIA_RETRY_COUNT ) );
		}
	}	
	
	
	/**
	 *
	 * Retrieve if Email media is ON
	 *
	 */
	function isEmailMediaOn()
	{
		//notification status on ALT1
		$mediaConfig = $this->sendReceive( SMTOtuApi::getEmailMedia() );
		$mediaConfig = explode(",", $mediaConfig );
	
		$notificationStatus = ($mediaConfig[self::MEDIA_INDEX] == self::EMAIL_MEDIA);

		return $notificationStatus;
	}
	
	/**
	 *
	 * Validate the Email configuration according to media (ETH) and destination address
	 *
	 * @param array $emailArray the Email configuration array to check
	 *
	 * returns true if the Email configuration is valid
	 */
	private static function isEmailConfigValid( $emailArray )
	{
		$isEmailConfigValid = FALSE;
		if ( $emailArray[self::MEDIA_INDEX]!= SMTOtuApi::RES_UNKNOWN && $emailArray[self::MEDIA_INDEX] != SMTOtuApi::RES_NONE && $emailArray[self::MEDIA_INDEX] != "" && $emailArray[self::ADDRESS_TO_INDEX] != "" )
		{
			$isEmailConfigValid = TRUE;
		}
		return $isEmailConfigValid;
	}
	
	/**
	 *
	 * Validate the Email security protocol
	 *
	 * @param $securityProtocol string the Email security protocol to check
	 *
	 * returns true if the Email security protocol is valid
	 */
	public static function isEmailSMTPSecurityProtocolValid( $securityProtocol )
	{
		$isEmailSecurityProtocolValid = FALSE;
		if ( ($securityProtocol == self::DEFAULT_SMTP_SECURITY_NONE) || ($securityProtocol == self::SMTP_SECURITY_SSLTLS) || ($securityProtocol== self::SMTP_SECURITY_STARTTLS ) )
		{
			$isEmailSecurityProtocolValid = TRUE;
		}
		return $isEmailSecurityProtocolValid;
	}	
	
}
?>