<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\serviceshelper\maintenance;

use app\services\maintenance\SMTBackupDto;

use app\util\SMTIOException;

use app\parser\SMTOtuApi;

use app\serviceshelper\SMTServiceHelper;

use app\util\SMTLogger;

/**
 * Handles OTU backup/restore function
 *
 * @author Sylvain Desplat
 */
class SMTOtuBackupRestore extends SMTServiceHelper
{
    const OTU_BACKUP_DIR = "/acterna/user/harddisk/";
    const OTU_BACKUP_NAME_PREFIX = "smartotu-backup-";
    const OTU_BACKUP_NAME_SUFFIX = ".tar.gz";
    const OTU_BACKUP_NAME_EMPTY_SERIAL_NUMBER = "00000";
    
    /**
     * Create a backup of SmartOTU configuration and monitoring on OTU disk and return the backup full path name.
     * 
     * @return string Backup full path on OTU
     * @throws SMTIOException If backup cannot be created
     */
    function retrieveSmartOtuBackup()
    {
        $backupName = NULL;
        try
        {
            $backupName = $this->sendReceive( SMTOtuApi::getSmartOtuBackupCommand() );
            $this->getContext()->getLogger()->trace( sprintf("Successfully created backup, with name: %s.", $backupName) );        
        }
        catch(\Exception $e )
        {
        	$this->getContext()->getLogger()->traceException($e);
        	$e = new SMTIOException( SMTIOException::COULD_NOT_CREATE_BACKUP );
        	throw $e;
        }
        
        return $backupName;
    }
    
    /**
     * Launch the restore of theOTU configuration, monitoring and SmartOTU password file
     * 
     * @param string $backupName the backup name
     * @throws SMTIOException If backup archive was not found
     */
    function restoreSmartOtuBackup( $backupName )
    {
        //check if backup is available on disk
        $backupPath = self::checkBackupFileName( $backupName );
        if ( $backupPath != NULL )
        {
            //start the restore
            $this->send( SMTOtuApi::smartOtuRestoreCommand( $backupPath ) );
            
            //the Otu is going to reboot
            $this->getContext()->setOTUReboot();
            
            $this->getContext()->getLogger()->trace( sprintf("Successfully launched SmartOTU restore, with backup: %s.", $backupName), SMTLogger::INFO );
        }
        else
        {
            throw new SMTIOException( SMTIOException::NO_BACKUP_FOUND );            
        }
    }
    
    /**
     * Return last available backup name
     * 
     * @return \app\services\maintenance\SMTBackupDto
     */
    public static function getLastAvailableBackupNameDto()
    {
        $backupDto = new SMTBackupDto();
        $backupDto->setBackupFileName( self::getCurrentBackupName() );
        
        return $backupDto;
    }
    
    /**
     * Retrieve the full backup file name; returns NULL if not found and throw an exception if the serial number in the archive is not the serial number of the OTU.
     * @param string $backupName the backup name
     * @return the full backup file name; returns NULL if not found.
     */
    public static function checkBackupFileName( $backupName )
    {        
    	$fileName = NULL;
    	
    	//retrieve serial number from backup name
    	if ( preg_match_all('!\d+!', $backupName, $matches) != FALSE )
    	{    	
        	$serialNumberFromName = intval( $matches[0][0] );        	
        	
        	if ( !isset( $serialNumberFromName ) )
        	{
        	    //we allow to restore a backup with a different serial number
//         	    $otuInfo = $this->sendReceive( SMTOtuApi::getOtuInfo() );
//         	    list( $company, $otuType, $serialNumber, $otuMode, $otuSoftwareRelease ) = explode(",", $otuInfo);
//         	    if ( intval( $serialNumberFromName ) != intval( $serialNumber ) )
//         	    {
//         	        throw new SMTIOException( SMTIOException::INVALID_FILE );
//         	    }
//         	}
//         	else
//         	{
        	    throw new SMTIOException( SMTIOException::INVALID_FILE );
        	}
    	}
    	else
    	{
    	    throw new SMTIOException( SMTIOException::INVALID_FILE );
    	}
    	
    	//retrieve file name
   		$resultArray = glob( self::OTU_BACKUP_DIR.$backupName );
    
		//first file is the file to download; if glob fails, returns false
		if ( $resultArray != FALSE && !empty( $resultArray ) )
		{
			$fileName = $resultArray[0];
		}

    	return $fileName;
    }
    
    /**
     * Retrieve the name of the current (last) backup archive copied on disk
     *
     * @return current backup or NULL if none is found
     */
    public static function getCurrentBackupName()
    {
    	$currentBackupName = NULL;        
    	
    	//retrieve file name
    	$files = glob( self::OTU_BACKUP_DIR.self::OTU_BACKUP_NAME_PREFIX."*".self::OTU_BACKUP_NAME_SUFFIX );    	
    	
    	//search last backup file
    	if ( $files != FALSE && !empty( $files ) )
    	{
    	    usort($files, function($a, $b) 
    	        {
    	    	    return (filemtime($a) < filemtime($b)) ? -1 : 1;
    	        } );
    		$currentBackupName = basename( $files[ count($files) - 1 ] );
    		//if selected backup is the automatic backup performed before restore, ignore it
    		if ( strpos( $currentBackupName, self::OTU_BACKUP_NAME_PREFIX.self::OTU_BACKUP_NAME_EMPTY_SERIAL_NUMBER ) !== FALSE )
    		{
    		    if ( count($files) > 1 )
    		    {
    		        $currentBackupName = basename( $files[ count($files) - 2 ] );
    		    }
    		    else
    		    {
    		        $currentBackupName = NULL;
    		    }
    		}
    	}
    
    	return $currentBackupName;
    }    
}
?>