<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\alarm;

use app\database\SMTSmartOtuDB;

use app\services\alarm\SMTAttenuationOpticalAlarmDto;

use app\services\alarm\SMTSystemAlarmEventDto;

use app\services\alarm\SMTAttenuationOpticalAlarmEventDto;

use app\serviceshelper\otau\SMTMacroSwitch;

use app\util\SMTLogger;

use app\services\alarm\otu\SMTOtuAlarmDto;

use app\services\alarm\otu\SMTOtuSystemAlarmDto;

use app\services\alarm\otu\SMTOtuOpticalAlarmDto;

use app\services\alarm\SMTSystemAlarmDto;
use app\services\alarm\otu\SMTOtuOpticalAlarmPeakDto;
use app\services\alarm\SMTPeakOpticalAlarmDto;
use app\services\alarm\SMTPeakOpticalAlarmEventDto;


define( "ATTENUATION_ALARM_DISCRIMINATOR_FORMAT", "test=%d".SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR.SMTAlarmSpecificProblemDiscriminatorCode::ATTENUATION_ALARM );
define( "PEAK_ALARM_DISCRIMINATOR_FORMAT", "test=%d".SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR.SMTAlarmSpecificProblemDiscriminatorCode::PEAK_ALARM.SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR."peak=%0.2f");

/**
 * Create an optical or system alarm.
 * 
 * @author Sylvain Desplat
 */
class SMTAlarmFactory
{
	/**
	 * Build alarm trace identifier = attenuation alarm identifier
	 * @param int $testId
	 * @return string
	 */
    public static function getDiscriminatorForOtdrTrace($testId)
    {
    	return sprintf(ATTENUATION_ALARM_DISCRIMINATOR_FORMAT,$testId);
    }
    
    /**
     * Forge OTU alarm DTO from the given alarm array built from alarm encoded in json string
     * 
     * @param array $alarmInfoArray
     * 
     * @return SMTOtuAlarmDto
     */
    public static function forgeOtuAlarmDto( array &$alarmInfoArray )
    {        
        if ( array_key_exists( SMTAlarmResourceCode::OTU_ALARM_CODE_TEXT_KEY, $alarmInfoArray )  )
        {
            $resourceType = $alarmInfoArray[ SMTAlarmResourceCode::OTU_ALARM_CODE_TEXT_KEY];
        }        
        //if resource type was not found, try with a key with first letter upper case 
        else
        {
            $resourceType = $alarmInfoArray[ ucfirst( SMTAlarmResourceCode::OTU_ALARM_CODE_TEXT_KEY ) ];
        }
        
        switch ( $resourceType )
        {
        	case SMTAlarmResourceCode::OTU_OPTICAL_ALARM_CODE_TEXT:
                 return SMTOtuOpticalAlarmDto::getInstance($alarmInfoArray);  	    
        	break;
        	case SMTAlarmResourceCode::OTU_SYSTEM_ALARM_CODE_TEXT:
        		return SMTOtuSystemAlarmDto::getInstance($alarmInfoArray);
        	break;
        	default:
        	    SMTLogger::getInstance()->trace( sprintf("Couldn't forge OTU alarm with unknown resource type: %s. Alarm details: %s.", $resourceType, print_r($alarmInfoArray, TRUE) ), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__ );
        	    throw new SMTAlarmException( SMTAlarmException::ERROR_ALARM_RETRIEVAL_FAILED, sprintf("Couldn't forge OTU alarm with unknown resource type: %s. Alarm details: %s.", $resourceType, print_r($alarmInfoArray, TRUE) ) );
        	    return NULL;
        	break;        	            	
        }
    }
    
    /**
     * Update attenuation optical alarm from the given otu optical alarm and returns the new event
     * 
     * @param SMTSmartOtuDB $dbConnection
     * @param SMTOtuAlarmDto $otuAlarm the otu alarm to process
     * @param SMTAttenuationOpticalAlarmDto $alarm Alarm to update or NULL if it is a new alarm
     * @param string $alarmDiscriminator The alarm discriminator
     * àparam boolean $newAlarm Whether the alarm is new, without its current event and not yet saved 
     * 
     * @throws SMTAlarmException
     * 
     * @return SMTAttenuationOpticalAlarmEventDto the alarm event created
     */
    public static function updateAttenuationOpticalAlarmDto( SMTSmartOtuDB $dbConnection, SMTOtuOpticalAlarmDto $otuAlarm, SMTAttenuationOpticalAlarmDto $alarm = NULL, $alarmDiscriminator, $newAlarm = TRUE ) 
    {
        SMTLogger::getInstance()->trace( sprintf("Optical alarm attenuation with transition: %b. Alarm specific problem: %s.", SMTOtuAlarmSeverity::hasSeverityTransition( $otuAlarm->getOtuSeverity() ), $otuAlarm->getAttenuation()->getSpecificProblemCode()), SMTLogger::INFO );      	   
	    
	    // if attenuation optical alarm DTO exists
	    if ( $alarm != NULL )
	    {
		    SMTLogger::getInstance()->trace( sprintf("Create new attenuation event of severity: %s. Alarm specific problem: %s.", $otuAlarm->getAttenuation()->getOtuSeverity(), $otuAlarm->getAttenuation()->getSpecificProblemCode()),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
		    	
			//create the event
			$alarmEvent = new SMTAttenuationOpticalAlarmEventDto();
			$alarmEvent->setAlarmId( $alarm->getId() );
			$alarmEvent->setSeverity( $otuAlarm->getAttenuation()->getSeverityConvertedToSmartOtu() );
			$alarmEvent->setTime( $otuAlarm->getGenerationDateUTC( $dbConnection ) );    				
			$alarmEvent->setProblemConfirmed( $otuAlarm->isConfirmed() );
			
			if ( $otuAlarm->getAttenuation()->isValidDistance() )
			{
			    $alarmEvent->setDistanceM( $otuAlarm->getAttenuation()->getDistance() );
			}
			$alarmEvent->setDistanceValid( $otuAlarm->getAttenuation()->isValidDistance() );
			if ( $otuAlarm->getAttenuation()->isValidLevel() ) 
			{
			    $alarmEvent->setLevelDb( $otuAlarm->getAttenuation()->getLevel() );
			}
			$alarmEvent->setLevelValid( $otuAlarm->getAttenuation()->isValidLevel() );    	
			if ( $otuAlarm->getAttenuation()->isValidGps() )
			{
				$alarmEvent->setGpsX($otuAlarm->getAttenuation()->getGpsX() );
				$alarmEvent->setGpsY($otuAlarm->getAttenuation()->getGpsY() );			
			}
			$alarmEvent->setGpsValid( $otuAlarm->getAttenuation()->isValidGps() );
			if ( $otuAlarm->getDeviation()->isFirstMarkerDeviationValid() )
			{
			    $alarmEvent->setFirstMarkerDeviationDb($otuAlarm->getDeviation()->getFirstMarkerDeviation() );
			}
			$alarmEvent->setFirstMarkerDeviationValid( $otuAlarm->getDeviation()->isFirstMarkerDeviationValid() );
			if ( $otuAlarm->getDeviation()->isLinkLossDeviationValid() )
			{
			    $alarmEvent->setLinkLossDeviationDb($otuAlarm->getDeviation()->getLinkLossDeviation() );
			}			
			$alarmEvent->setLinkLossDeviationValid( $otuAlarm->getDeviation()->isLinkLossDeviationValid() );

			//Attach the event to the alarm:
			// - Update the alarm current event only if the event changes the properties of the alarm 
			//   (not a persistent failure, an unknown cause...) or if the alarm is new
			if ( $newAlarm || 
			        ( SMTAlarmSpecificProblemCode::isOpticalAlarmSpecificProblemCode( $otuAlarm->getAttenuation()->getSpecificProblemCode() ) ) &&
			        $otuAlarm->isValidAttenuationSeverity()
			        )
			        
			{
			    //attach the new event to the alarm
			    $alarm->setCurrentAlarmEvent( $alarmEvent );
			    
			    //update alarm UTC date time in seconds
			    $alarm->setLastUpdate( $alarmEvent->getTime() );					    
			    			    
			    $severity = $otuAlarm->getAttenuation()->getSeverityConvertedToSmartOtu();
			    
			    //update the alarm severity if it is an alarm severity
			    if ( SMTAlarmSeverity::isAlarmSeverity( $severity ) )
			    {
			    	$alarm->setSeverity( $severity );
			    }
			    //if it's a clear event, set the alarm as cleared
			    else if ( $severity == SMTAlarmSeverity::SEVERITY_NONE )
			    {
			    	SMTLogger::getInstance()->trace( sprintf("Clear the attenuation alarm: %s. Alarm specific problem: %s.", $alarmDiscriminator, $alarm->getSpecificProblemCode()), SMTLogger::INFO );
			    	$alarm->setClearStatus( TRUE );
			    }			    

			    //update alarm specific problem
			    $alarm->setSpecificProblemCode( $otuAlarm->getAttenuation()->getSpecificProblemCode() );			    			    
			    $alarm->updateSpecificProblemText();
			}
			else if ( ($otuAlarm->getAttenuation() != NULL) && ($otuAlarm->getAttenuation()->getSpecificProblemCode() == SMTAlarmSpecificProblemCode::OTU_OPT_ALARM_BACK_TO_NORMAL_STATE_CODE) )
			{
				$severity = $otuAlarm->getAttenuation()->getSeverityConvertedToSmartOtu();
				if ( $severity == SMTAlarmSeverity::SEVERITY_NONE )
				{
					SMTLogger::getInstance()->trace( sprintf("Force Clear the attenuation alarm: %s. Alarm specific problem: %s.", $alarmDiscriminator, $alarm->getSpecificProblemCode()), SMTLogger::INFO );
					$alarm->setClearStatus( TRUE );
				}
			}
		}
		else
		{		    
    		SMTLogger::getInstance()->trace("Alarm not processed by SmartOtu: Unknow otu alarm: ".print_r( $otuAlarm->getJsonData(), TRUE ), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		//throw new SMTAlarmException( SMTAlarmException::ERROR_ALARM_RETRIEVAL_FAILED, "Alarm not processed by SmartOtu: invalid alarm".$otuAlarm->getCode()." file: ".$otuAlarm->getAcknowledgeString() );
		}    
    
    	return $alarmEvent;
    }        
    
    /**
     * Create a new attenuation optical alarm from the given otu optical alarm without its current event
     *
     * @param SMTSmartOtuDB $dbConnection
     * @param SMTOtuAlarmDto $otuAlarm
     * @param string $alarmDiscriminator
     *
     * @throws SMTAlarmException
     *
     * @return SMTAttenuationOpticalAlarmDto the alarm created
     */
    public static function createAttenuationOpticalAlarmDto( SMTSmartOtuDB $dbConnection, SMTOtuOpticalAlarmDto $otuAlarm, $alarmDiscriminator )
    {
    	SMTLogger::getInstance()->trace( sprintf("Create Optical alarm attenuation with transition: %b. Alarm specific problem: %s.", SMTOtuAlarmSeverity::hasSeverityTransition( $otuAlarm->getOtuSeverity() ), $otuAlarm->getAttenuation()->getSpecificProblemCode()), SMTLogger::INFO );
    
    	//process attenuation alarm
    	//If optical alarm is new and if there is a transition of attenuation alarm severity, create a new alarm.
    	if ( SMTOtuAlarmSeverity::hasSeverityTransition( $otuAlarm->getOtuSeverity() ) )
    	{
    		SMTLogger::getInstance()->trace( sprintf("Create new attenuation alarm with discriminator: %s. Alarm specific problem: %s.", $alarmDiscriminator, $otuAlarm->getAttenuation()->getSpecificProblemCode()), SMTLogger::INFO );
    		$newAlarm = TRUE;
    		 
    		$alarm = new SMTAttenuationOpticalAlarmDto();
    		$alarm->setTestId( $otuAlarm->getTestId() );    		
    		$alarm->setDiscriminator( $alarmDiscriminator );
    		$alarm->setAlarmType( SMTAlarmEventTypeCode::OTU_ALARM_EVENT_TYPE_QUALITY_OF_SERVICE_ALARM );    		  	
    		$alarm->setLastUpdate( $otuAlarm->getGenerationDateUTC( $dbConnection ) );    		
    		
  		    $severity = $otuAlarm->getSeverityConvertedToSmartOtu();    		    
    				
    		//update the alarm severity if it is an alarm severity
    		if ( SMTAlarmSeverity::isAlarmSeverity( $severity ) )
    		{
    			$alarm->setSeverity( $severity );
    			$alarm->setClearStatus( FALSE );
    		}
    		//if it is is a clear event, set the alarm as cleared
    		else if ( $severity == SMTAlarmSeverity::SEVERITY_NONE )
    		{
    			SMTLogger::getInstance()->trace( sprintf("Clear the attenuation alarm: %s. Alarm specific problem: %s.", $alarmDiscriminator, $alarm->getSpecificProblemCode()), SMTLogger::INFO );
    			$alarm->setClearStatus( TRUE );
    		}
    		
    		$alarm->setSpecificProblemCode( $otuAlarm->getAttenuation()->getSpecificProblemCode() );
    		$alarm->updateSpecificProblemText();
    	}    	 
    	else
    	{
    	    SMTLogger::getInstance()->trace("Alarm event with no severity change was not processed by SmartOtu: alarm: ".print_r( $otuAlarm->getJsonData(), TRUE ), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    	}
    
    	return $alarm;
    }
    
    /**
     * Create a new peak optical alarm from the given otu optical alarm without its current event
     *
     * @param SMTSmartOtuDB $dbConnection
     * @param SMTOtuAlarmDto $otuAlarm
     * @param SMTOtuOpticalAlarmPeakDto $peakAlarmDto 
     * @param string $alarmDiscriminator
     *
     * @throws SMTAlarmException
     *
     * @return SMTPeakOpticalAlarmDto the alarm created
     */
    public static function createPeakOpticalAlarmDto( SMTSmartOtuDB $dbConnection, SMTOtuOpticalAlarmDto $otuAlarm, SMTOtuOpticalAlarmPeakDto $peakAlarmDto, $alarmDiscriminator )
    {
    	SMTLogger::getInstance()->trace( sprintf("Create peak optical alarm with transition: %b. Alarm specific problem: %s.", SMTOtuAlarmSeverity::hasSeverityTransition( $peakAlarmDto->getOtuSeverity() ), SMTAlarmSpecificProblemCode::OTU_OPT_ALARM_PEAK_CODE), SMTLogger::INFO );
    	
    	//process peak alarm
    	//If optical alarm is new and if there is a transition of peak alarm severity, create a new alarm.
    	if ( SMTOtuAlarmSeverity::hasSeverityTransition( $peakAlarmDto->getOtuSeverity() ) )
    	{
    		SMTLogger::getInstance()->trace( sprintf("Create new peak alarm with discriminator: %s. Alarm specific problem: %s.", $alarmDiscriminator, SMTAlarmSpecificProblemCode::OTU_OPT_ALARM_PEAK_CODE), SMTLogger::INFO );
    		$newAlarm = TRUE;
    		
    		$alarm = new SMTPeakOpticalAlarmDto();
    		$alarm->setTestId( $otuAlarm->getTestId() );
    		$alarm->setDiscriminator( $alarmDiscriminator );
    		$alarm->setAlarmType( SMTAlarmEventTypeCode::OTU_ALARM_EVENT_TYPE_QUALITY_OF_SERVICE_ALARM );
    		$alarm->setLastUpdate( $otuAlarm->getGenerationDateUTC( $dbConnection ) );
    		$alarm->setSpecificProblemCode( SMTAlarmSpecificProblemCode::OTU_OPT_ALARM_PEAK_CODE);
    		
    		$severity = $peakAlarmDto->getSeverityConvertedToSmartOtu();
    		
    		//update the alarm severity if it is an alarm severity
    		if ( SMTAlarmSeverity::isAlarmSeverity( $severity ) )
    		{
    			$alarm->setSeverity( $severity );
    			$alarm->setClearStatus( FALSE );
    		}
    		//if it is is a clear event, set the alarm as cleared
    		else if ( $severity == SMTAlarmSeverity::SEVERITY_NONE )
    		{
    			SMTLogger::getInstance()->trace( sprintf("Clear the peak alarm: %s. Alarm specific problem: %s.", $alarmDiscriminator, $alarm->getSpecificProblemCode()), SMTLogger::INFO );
    			$alarm->setClearStatus( TRUE );
    		}    		    		
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Alarm event with no severity change was not processed by SmartOtu: alarm: ".print_r( $otuAlarm->getJsonData(), TRUE ), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    	}
    	
    	return $alarm;
    }
    
    /**
     * Update peak optical alarm from the given otu optical alarm and returns the new event
     *
     * @param SMTSmartOtuDB $dbConnection
     * @param SMTOtuAlarmDto $otuAlarm the otu alarm to process
     * @param SMTPeakOpticalAlarmDto $alarm Alarm to update or NULL if it is a new alarm
     * @param string $alarmDiscriminator The alarm discriminator
     * àparam boolean $newAlarm Whether the alarm is new, without its current event and not yet saved
     *
     * @throws SMTAlarmException
     *
     * @return SMTPeakOpticalAlarmEventDto the alarm event created
     */
    public static function updatePeakOpticalAlarmDto( SMTSmartOtuDB $dbConnection, SMTOtuOpticalAlarmDto $otuAlarm, SMTOtuOpticalAlarmPeakDto $peakAlarmDto, SMTPeakOpticalAlarmDto $alarm = NULL, $alarmDiscriminator, $newAlarm = TRUE )
    {
    	SMTLogger::getInstance()->trace( sprintf("Peak Optical alarm with transition: %b. Alarm specific problem: %s.", SMTOtuAlarmSeverity::hasSeverityTransition( $peakAlarmDto->getOtuSeverity() ), SMTAlarmSpecificProblemCode::OTU_OPT_ALARM_PEAK_CODE), SMTLogger::INFO );
    	
    	// if peak optical alarm DTO exists
    	if ( $alarm != NULL )
    	{
    		SMTLogger::getInstance()->trace( sprintf("Create new peak event of severity: %s. Alarm specific problem: %s.", $peakAlarmDto->getOtuSeverity(), SMTAlarmSpecificProblemCode::OTU_OPT_ALARM_PEAK_CODE),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    		
    		//create the event
    		$alarmEvent = new SMTPeakOpticalAlarmEventDto();
    		$alarmEvent->setAlarmId( $alarm->getId() );
    		$alarmEvent->setSeverity( $peakAlarmDto->getSeverityConvertedToSmartOtu() );
    		$alarmEvent->setTime( $otuAlarm->getGenerationDateUTC( $dbConnection ) );
    		$alarmEvent->setProblemConfirmed( $otuAlarm->isConfirmed() );
    		
    		$alarmEvent->setReferenceTopDistanceM( $peakAlarmDto->getReferenceTopDistance() );
    		$alarmEvent->setTopDistanceM( $peakAlarmDto->getTopDistance() );
    		$alarmEvent->setBottomDistanceM( $peakAlarmDto->getBottomDistance() );
    		$alarmEvent->setPeakName($peakAlarmDto->getPeakName());

    		if ( $peakAlarmDto->isValidLevel() )
    		{
    			$alarmEvent->setLevelDb( $peakAlarmDto->getLevel() );
    		}
    		$alarmEvent->setLevelValid( $peakAlarmDto->isValidLevel() );
    		if ( $otuAlarm->getDeviation()->isFirstMarkerDeviationValid() )
    		{
    			$alarmEvent->setFirstMarkerDeviationDb($otuAlarm->getDeviation()->getFirstMarkerDeviation() );
    		}
    		$alarmEvent->setFirstMarkerDeviationValid( $otuAlarm->getDeviation()->isFirstMarkerDeviationValid() );
    		if ( $otuAlarm->getDeviation()->isLinkLossDeviationValid() )
    		{
    			$alarmEvent->setLinkLossDeviationDb($otuAlarm->getDeviation()->getLinkLossDeviation() );
    		}
    		$alarmEvent->setLinkLossDeviationValid( $otuAlarm->getDeviation()->isLinkLossDeviationValid() );
    		
    		//Attach the event to the alarm:
    		// - Update the alarm current event only if the event changes the properties of the alarm
    		//   (not a persistent failure, an unknown cause...) or if the alarm is new
    		if ( $newAlarm 
    				|| $otuAlarm->isValidPeakSeverity($peakAlarmDto) )
    				
    		{
    			//attach the new event to the alarm
    			$alarm->setCurrentAlarmEvent( $alarmEvent );
    			
    			//update alarm UTC date time in seconds
    			$alarm->setLastUpdate( $alarmEvent->getTime() );
    			
    			$severity = $peakAlarmDto->getSeverityConvertedToSmartOtu();
    			
    			//update the alarm severity if it is an alarm severity
    			if ( SMTAlarmSeverity::isAlarmSeverity( $severity ) )
    			{
    				$alarm->setSeverity( $severity );
    			}
    			//if it's a clear event, set the alarm as cleared
    			else if ( $severity == SMTAlarmSeverity::SEVERITY_NONE )
    			{
    				SMTLogger::getInstance()->trace( sprintf("Clear the peak alarm: %s. Alarm specific problem: %s.", $alarmDiscriminator, $alarm->getSpecificProblemCode()), SMTLogger::INFO );
    				$alarm->setClearStatus( TRUE );
    			}
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Alarm not processed by SmartOtu: Unknow peak otu alarm: ".print_r( $otuAlarm->getJsonData(), TRUE ), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		//throw new SMTAlarmException( SMTAlarmException::ERROR_ALARM_RETRIEVAL_FAILED, "Alarm not processed by SmartOtu: invalid alarm".$otuAlarm->getCode()." file: ".$otuAlarm->getAcknowledgeString() );
    	}
    	
    	return $alarmEvent;
    }        
    
    
    /**
     * Create or update a system alarm from the given otu system alarm
     *
     * @param SMTSmartOtuDB $dbConnection
     * @param SMTOtuSystemAlarmDto $otuAlarm
     * @param SMTSystemAlarmDto $alarm Alarm to update or NULL if it is a new alarm
     * @param string $alarmDiscriminator
     *
     * @throws SMTAlarmException
     *
     * @return SMTSystemAlarmDto the alarm updated or created
     */
    public static function createOrUpdateSystemAlarmDto( SMTSmartOtuDB $dbConnection, SMTOtuSystemAlarmDto $otuAlarm, SMTSystemAlarmDto $alarm = NULL, $alarmDiscriminator )
    {
    	SMTLogger::getInstance()->trace( sprintf("System alarm with transition: %b. Alarm specific problem: %s.", SMTOtuAlarmSeverity::hasSeverityTransition( $otuAlarm->getOtuSeverity() ), $otuAlarm->getSpecificProblemCode()), SMTLogger::INFO );
    
		//If system alarm is new and if there is a transition of alarm severity, create a new alarm.
		if ( $alarm == NULL && SMTOtuAlarmSeverity::hasSeverityTransition( $otuAlarm->getOtuSeverity() ) )
		{
			SMTLogger::getInstance()->trace( sprintf("Create new system alarm with discriminator: %s. Alarm specific problem: %s.", $alarmDiscriminator, $otuAlarm->getSpecificProblemCode()), SMTLogger::INFO );

			$alarm = new SMTSystemAlarmDto();
			$alarm->setClearStatus( FALSE );
			$alarm->setDiscriminator( $alarmDiscriminator );
			$alarm->setSpecificProblemId( $otuAlarm->getSpecificProblemId() );
			$alarm->setOtuAdditionalInfoId( $otuAlarm->getAdditionalInfo() );
			$alarm->setAlarmType( $otuAlarm->getEventTypeCode() );
			$alarm->setAdditionText( self::buildSystemAlarmAdditionalInfo( $otuAlarm ) );
			$alarm->setSpecificProblemCode( $otuAlarm->getSpecificProblemCode() );			
		}

		// if system alarm DTO can be created or exists
		if ( $alarm != NULL )
		{
			SMTLogger::getInstance()->trace( sprintf("Create new system event of severity: %s. Alarm specific problem: %s.", $otuAlarm->getOtuSeverity(), $otuAlarm->getSpecificProblemCode()),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );

			$severity = $otuAlarm->getSeverityConvertedToSmartOtu();
			
			//create the event
			$alarmEvent = new SMTSystemAlarmEventDto();
			$alarmEvent->setAlarmId( $alarm->getId() );
			$alarmEvent->setSeverity( $severity );
			$alarmEvent->setTime( $otuAlarm->getGenerationDateUTC( $dbConnection ) );

			//attach the event to the alarm
			//last event is always the real status of the system alarm (no persistent failure....)
			$alarm->setCurrentAlarmEvent( $alarmEvent );

			//update the alarm UTC date time
			$alarm->setLastUpdate( $alarmEvent->getTime() );
			
			//update alarm specific problem: no update needed: if specific problem changes, it's a new alarm

			//update the alarm severity if it is an alarm severity
			if ( SMTAlarmSeverity::isAlarmSeverity( $severity ) )
			{
				$alarm->setSeverity( $severity );
			}
			//if it is is a clear event, set the alarm as cleared
			else if ( $severity == SMTAlarmSeverity::SEVERITY_NONE )
			{
				SMTLogger::getInstance()->trace( sprintf("Clear the system alarm: %s. Alarm specific problem: %s.", $alarmDiscriminator, $alarm->getSpecificProblemCode()), SMTLogger::INFO );
				$alarm->setClearStatus( TRUE );
			}
			
			$alarm->setSpecificProblemText( SMTAlarmSpecificProblemCode::decodeSystemAlarmSpecificProblem( $otuAlarm->getSpecificProblemCode(), $alarm->isCleared() ) );
		}
		else
		{
			SMTLogger::getInstance()->trace("Alarm not processed by SmartOtu: Invalid otu alarm: ".print_r( $otuAlarm->getJsonData(), TRUE ), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
			throw new SMTAlarmException( SMTAlarmException::ERROR_ALARM_RETRIEVAL_FAILED, "Alarm not processed by SmartOtu: invalid alarm".$otuAlarm->getCode()." file: ".$otuAlarm->getAcknowledgeString() );
		}
    
    	return $alarm;
    }    

    /**
     * Only one discriminator in the case of SmartOTU monitoring limited to attenuation faults.
     * Try to build the optical alarm discriminator from infos given in alarm. If more info is needed, try to retrieve alarms from database on same test.
     * 
     * @param SMTOtuOpticalAlarmDto $otuAlarm
     * @param \SQLite3 $dbConnection
     * 
     * @return array
     */
    public static function buildOpticalAlarmDiscriminators( SMTOtuOpticalAlarmDto $otuAlarm, \SQLite3 $dbConnection  )
    {
        $alarmDiscriminators = array();
        
        //To simplify, don't search if otu alarm is an attenuation alarm: for alarms limited to attenuation alarms:
//         $alarmDiscriminator = sprintf( ATTENUATION_ALARM_DISCRIMINATOR_FORMAT, $otuAlarm->getTestId() );
//         array_push( $alarmDiscriminators, $alarmDiscriminator );

//TODO: uncomment to search if otu alarm is an attenuation alarm for alarms not limited to attenuation alarms        
        //process attenuation alarm: change of severity or persistent failure for an attenuation
        if ( ($otuAlarm->getAttenuation()!= null) && SMTOtuAlarmSeverity::hasSeverityTransition( $otuAlarm->getAttenuation()->getOtuSeverity() ) ||  
        		($otuAlarm->getAttenuation()!= null) && ($otuAlarm->getAttenuation()->getSpecificProblemCode() == SMTAlarmSpecificProblemCode::OTU_OPT_ALARM_PERSISTENT_FAILURE_CODE )  )
        {
        	$alarmDiscriminator = sprintf( ATTENUATION_ALARM_DISCRIMINATOR_FORMAT, $otuAlarm->getTestId() );     
        	array_push( $alarmDiscriminators, $alarmDiscriminator);
        }
        //if no discriminator could be computed, search if an attenuation alarm exists on the test referenced by the otu alarm
        else
        {
        	$alarmDiscriminator = sprintf( ATTENUATION_ALARM_DISCRIMINATOR_FORMAT, $otuAlarm->getTestId() );
        	//search if attenuation optical alarm with that discriminator exists
        	$alarm = SMTAttenuationOpticalAlarmDto::fetch($dbConnection, $alarmDiscriminator);
            if ( $alarm != NULL )
            {
            	//alarm could be found on that test; add the discriminator to the list:
            	array_push( $alarmDiscriminators, $alarmDiscriminator );
            }
        }
        
        if ( ($otuAlarm->getPeaks()!= null) && ( count($otuAlarm->getPeaks()) > 0 ) && SMTOtuAlarmSeverity::hasSeverityTransition( $otuAlarm->getOtuSeverity() ) )
        {
        	foreach ($otuAlarm->getPeaks() as $peak)
        	{
        		if ( ($peak != null) && SMTOtuAlarmSeverity::hasSeverityTransition( $peak->getOtuSeverity()) )
        		{
        			$alarmDiscriminator = sprintf( PEAK_ALARM_DISCRIMINATOR_FORMAT, $otuAlarm->getTestId(), $peak->getReferenceTopDistance());
        			array_push( $alarmDiscriminators, $alarmDiscriminator );
        			break;
        		}
        	}
        }
        
//         //process ORL:
//         //NOT implemented in SmartOTU
//         //process fiber length extension:
//         //NOT implemented in SmartOTU
        
        return $alarmDiscriminators;
    }
    
    /**
     * Whether the alarm with the given discriminator is an attenuation alarm
     * @param string $alarmDiscriminator
     * 
     * @return boolean
     */
    public static function isAttenuationAlarm( $alarmDiscriminator )
    {
        return ( strpos( $alarmDiscriminator, SMTAlarmSpecificProblemDiscriminatorCode::ATTENUATION_ALARM ) !== FALSE );
    }
    
    /**
     * Whether the alarm with the given discriminator is a peak alarm
     * @param string $alarmDiscriminator
     *
     * @return boolean
     */
    public static function isPeakAlarm( $alarmDiscriminator )
    {
    	return ( strpos( $alarmDiscriminator, SMTAlarmSpecificProblemDiscriminatorCode::PEAK_ALARM ) !== FALSE );
    }
    
    /**
     * Build System alarm additional Information from OTU alarm additional info
     * 
     * @param SMTOtuSystemAlarmDto $otuAlarm
     * 
     * @return string system alarm additional info built from combination of specific problem and additional info.
     */
    private static function buildSystemAlarmAdditionalInfo( SMTOtuSystemAlarmDto $otuAlarm  )
    {
    	$additionalInfo = $otuAlarm->getAdditionalInfo();
    
    	// on indique un eventuel sous composant correlant :
    	switch ( $otuAlarm->getSpecificProblemCode() )
    	{
    		case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_MISSING_OR_CORRUPTED_FILE_CODE:
    			//$additionalInfo = "file=".$otuAlarm->getAdditionalInfo();
    			$additionalInfo = $otuAlarm->getAdditionalInfo();//"otu";
    			break;
    		case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_LOCAL_MODE_CODE:
    			$additionalInfo = $otuAlarm->getAdditionalInfo();//"otu";
    			break;
    		case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_INNER_APPLICATION_COMMUNICATION_PROBLEM_CODE:
    			$additionalInfo = $otuAlarm->getAdditionalInfo();//"otu";
    			break;
    		case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_REBOOT_CODE:
    			
    			$additionalInfo = $otuAlarm->getAdditionalInfo();
    			break;
    		case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_HARDDISK_SPACE_CODE:
    			$additionalInfo =$otuAlarm->getAdditionalInfo();//"otu";
    			break;
    		case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_TEMPERATURE_CODE:
    			if ( SMTSystemAlarmAdditionalInfoCode::INFO_ALARM_MODULE_1 == $otuAlarm->getAdditionalInfo() || SMTSystemAlarmAdditionalInfoCode::INFO_ALARM_MODULE_2 == $otuAlarm->getAdditionalInfo())
    			{
    				$additionalInfo ="module=".$otuAlarm->getAdditionalInfo();
    			}
    			else if ( SMTSystemAlarmAdditionalInfoCode::INFO_ALARM_CPU == $otuAlarm->getAdditionalInfo() )
    			{
    				$additionalInfo ="cpu ".$otuAlarm->getAdditionalInfo();
    			}
    			break;
    		case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_SWITCH_PROBLEM_CODE:
    			$parts = explode(',', $otuAlarm->getAdditionalInfo() );
    			if (isset($parts[0]))
    			{
    				$temp1 = $parts[0];
    				if (isset($parts[1]))
    				{
    					$temp0 = $parts[1];
    				}
    			}
    
    			$parts = explode(" ",$temp0);
    			if (isset($parts[0]))
    			{
    				$temp2 = $parts[0];
    				if (isset($parts[1]))
    				{
    					$temp3 = $parts[1];
    				}
    			}
    			 
    			if ( strcmp($temp2, SMTMacroSwitch::INTERNAL) == 0 )
    			{
    				$additionalInfo = sprintf( "switch=%s", $temp1 );
    			}
    			else if ( strcmp($temp2, SMTMacroSwitch::OSU) == 0 )
    			{
    				$additionalInfo = sprintf( "osx=%s", $temp1 );
    			}
    			else if ( strcmp($temp2, SMTMacroSwitch::ROTAU) == 0 )
    			{
    				$additionalInfo = sprintf( "rotau=%s", $temp1 );
    			}
    			break;
    		case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_MODULE_PROBLEM_CODE:
    			//MOD = 0 or 1 => +1
    			$additionalInfo ="module=".($otuAlarm->getAdditionalInfo());
    			break;
    		case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_MODULE_COMPATIBILITY_CODE:
    			//MOD = 0 or 1 => +1
    			$additionalInfo ="module=".($otuAlarm->getAdditionalInfo());
    			break;
    		case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_SWITCH_COMPATIBILITY_CODE:
    			if ( strcmp ( $otuAlarm->getAdditionalInfo(), "0") == 0 )
    			{
    		        $additionalInfo ="switch=".($otuAlarm->getAdditionalInfo());
    			}
    			else
    			{
    			    $additionalInfo ="rotau=".($otuAlarm->getAdditionalInfo());
    			}
    			
    		    break;
    		case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_LINKS_BACK_TO_AVAILABLE_CODE:
    			$additionalInfo =$otuAlarm->getAdditionalInfo();
    			break;
    			//
    		case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_MISSING_REFERENCE_TRACE_CODE:
    			$parts = explode(',', $otuAlarm->getAdditionalInfo() );
    			if (isset($parts[0]))
    			{
        			$temp1 = $parts[0];
        			if (isset($parts[1]))
        			{
        			    $temp2 = $parts[1];
        			}
    			}
    			$additionalInfo = sprintf("test=%s", $temp1 );
    			break;
			case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_SUCCESSFUL_CALLBACK_CODE:
    			$additionalInfo ="media=".($otuAlarm->getAdditionalInfo());
    			break;
			case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_TEST_DRIFT_CODE:
				$parts = explode(',', $otuAlarm->getAdditionalInfo() );
				if (isset($parts[0]))
				{
    				$temp1 = str_replace('"', "", $parts[0] );
    				if (isset($parts[1]))
    				{
    				    $temp2 = $parts[1];
    	            }
            	        }
            	$additionalInfo = sprintf("test=%s", $temp1 );
            	break;
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_COMMUNICATION_TEST_CODE:
        	    $additionalInfo ="media=".($otuAlarm->getAdditionalInfo());//."/interface=ETH0";
        	    break;
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_HEARTBEAT_CODE:
        		$additionalInfo = $otuAlarm->getAdditionalInfo();//"otu";
    			break;
			case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_HARDWARE_PROBLEM_CODE:
    			$additionalInfo =$otuAlarm->getAdditionalInfo();
    			break;
			case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_SOFTWARE_PROBLEM_CODE:
    			//$additionalInfo ="component=".($otuAlarm->getAdditionalInfo());
				$additionalInfo = $otuAlarm->getAdditionalInfo();//"otu";
    			break;
			case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_MEASUREMENT_CYCLE_CODE:
				$additionalInfo = $otuAlarm->getAdditionalInfo();
    			break;
			case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_ALARM_OVERFLOW_CODE:
				$additionalInfo = $otuAlarm->getAdditionalInfo();
    			break;
			case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_GENERIC_ALARM_CODE:
				$additionalInfo = $otuAlarm->getAdditionalInfo();//
    			break;
			case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_REBUILD_CLEAR_CODE:
				$additionalInfo = $otuAlarm->getAdditionalInfo();
    			break;
			case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_HDD_FAILED_CODE:
				$additionalInfo = $otuAlarm->getAdditionalInfo();
				break;
			case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_HDD_FAILED_USE_BACKUP_CODE:
				$additionalInfo = $otuAlarm->getAdditionalInfo();
				break;
			case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_POWER_FAILURE_CODE:
				$additionalInfo = $otuAlarm->getAdditionalInfo();
				break;
    		default:
    		    break;
    	}
    	
    	return $additionalInfo;
    	}
    
    
    /**
     * Build System Alarm Discriminator
     * 
     * @param SMTOtuSystemAlarmDto $otuAlarm
     */
    public static function buildSystemAlarmDiscriminator( SMTOtuSystemAlarmDto $otuAlarm  )
    {
        $alarmDiscriminator = "";        
        
        // on indique un eventuel sous composant correlant :
        switch ( $otuAlarm->getSpecificProblemCode() )
        {
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_MISSING_OR_CORRUPTED_FILE_CODE:
        	    $alarmDiscriminator ="file=".$otuAlarm->getAdditionalInfo().SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR;
                break;
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_LOCAL_MODE_CODE:
                break;
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_INNER_APPLICATION_COMMUNICATION_PROBLEM_CODE:
                break;
            case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_REBOOT_CODE:
                break;
            case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_HARDDISK_SPACE_CODE:
                break;
            case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_TEMPERATURE_CODE:
        	    if ( SMTSystemAlarmAdditionalInfoCode::INFO_ALARM_MODULE_1 == $otuAlarm->getAdditionalInfo() || SMTSystemAlarmAdditionalInfoCode::INFO_ALARM_MODULE_2 == $otuAlarm->getAdditionalInfo())
        	    {
        	        $alarmDiscriminator ="module".$otuAlarm->getAdditionalInfo().SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR;
        	    }
        	    else if ( SMTSystemAlarmAdditionalInfoCode::INFO_ALARM_CPU == $otuAlarm->getAdditionalInfo() )
        	    {
        	        $alarmDiscriminator ="cpu".SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR;
        	    }
        	    break;
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_SWITCH_PROBLEM_CODE:        	    
        	    $parts = explode(',', $otuAlarm->getAdditionalInfo() );
        	    if (isset($parts[0])) 
        	    {
        	    	$temp1 = $parts[0];
        	    	if (isset($parts[1])) 
        	    	{
        	    		$temp0 = $parts[1];
        	    	}
        	    }

        	    $parts = explode(" ",$temp0);
        	    if (isset($parts[0]))
        	    {
        	    	$temp2 = $parts[0];
        	    	if (isset($parts[1]))
        	    	{
        	    		$temp3 = $parts[1];
        	    	}
        	    }        	    
        	    
    			if ( strcmp($temp2, SMTMacroSwitch::INTERNAL) == 0 )
    			{
    				$alarmDiscriminator = sprintf( "switch=0/osu=%s/otau=%s", $temp1, $temp3 ).SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR;
    			}
    			else if ( strcmp($temp2, SMTMacroSwitch::OSU) == 0 )
    			{
    				$alarmDiscriminator = sprintf( "switch=0/osu=%s/otau=%s", $temp1, $temp2 ).SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR;
    			}
    			else if ( strcmp($temp2, SMTMacroSwitch::ROTAU) == 0 )
    			{
    				$alarmDiscriminator = sprintf( "switch=%s", $temp1 ).SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR;
    			}
    			break;
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_MODULE_PROBLEM_CODE:
        	    //MOD = 0 or 1 => +1
        	    $alarmDiscriminator ="module=".($otuAlarm->getAdditionalInfo()+1).SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR;
        	    break;
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_MODULE_COMPATIBILITY_CODE:
        	    //MOD = 0 or 1 => +1
        	    $alarmDiscriminator ="module=".($otuAlarm->getAdditionalInfo()+1).SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR;
        	    break;
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_SWITCH_COMPATIBILITY_CODE:
        	    $alarmDiscriminator ="switch=".($otuAlarm->getAdditionalInfo()).SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR;
        	    break;        	    
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_LINKS_BACK_TO_AVAILABLE_CODE:
        	    //MOD = 0 or 1 => +1
        	    $alarmDiscriminator ="module=".$otuAlarm->getAdditionalInfo().SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR;
        	    break;
        	//
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_MISSING_REFERENCE_TRACE_CODE:
        	    $parts = explode(',', $otuAlarm->getAdditionalInfo() );
        	    if (isset($parts[0]))
        	    {
        	    	$temp1 = $parts[0];
        	    	if (isset($parts[1]))
        	    	{
        	    		$temp2 = $parts[1];
        	    	}
        	    }        	     
       	    	$alarmDiscriminator = sprintf("mode=%s/test=%s", $temp2, $temp1 ).SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR;
        	    break;        	    
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_SUCCESSFUL_CALLBACK_CODE:
        	    $alarmDiscriminator ="media=".($otuAlarm->getAdditionalInfo()).SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR;
        	    break;        	    
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_TEST_DRIFT_CODE:
        	    $parts = explode(',', $otuAlarm->getAdditionalInfo() );
        	    if (isset($parts[0]))
        	    {
        	    	$temp1 = str_replace('"', "", $parts[0] );
        	    	if (isset($parts[1]))
        	    	{
        	    		$temp2 = $parts[1];
        	    	}
        	    }
        	    $alarmDiscriminator = sprintf("mode=%s/test=%s", $temp2, $temp1 ).SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR;
                break;
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_COMMUNICATION_TEST_CODE:
        	    $alarmDiscriminator ="media=".($otuAlarm->getAdditionalInfo())."/interface=ETH0".SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR;
        	    break;
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_HEARTBEAT_CODE:
        	    break;
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_HARDWARE_PROBLEM_CODE:
        	    $alarmDiscriminator ="component=".($otuAlarm->getAdditionalInfo()).SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR;
        	    break;
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_SOFTWARE_PROBLEM_CODE:
        	    $alarmDiscriminator ="component=".($otuAlarm->getAdditionalInfo()).SMTAlarmSpecificProblemDiscriminatorCode::ALARM_DISCRIMINATOR_SEPARATOR;
        	    break;
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_MEASUREMENT_CYCLE_CODE:
        	    break;
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_ALARM_OVERFLOW_CODE:
        	    break;
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_GENERIC_ALARM_CODE:
        	    break;
        	case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_REBUILD_CLEAR_CODE:
        	    break;
    	    case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_HDD_FAILED_CODE:
    	    	break;
    	    case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_HDD_FAILED_USE_BACKUP_CODE:
    	    	break;
    	    case SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_POWER_FAILURE_CODE:
    	    	break;
        	default:
                SMTLogger::getInstance()->trace( sprintf("Unknown OTU specific problem code : %s", $otuAlarm->getSpecificProblemCode()), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        		break;

        }
        return $alarmDiscriminator.$otuAlarm->getSpecificProblemCode();
    }

}
?>