<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\monitoring;

use app\serviceshelper\otdr\SMTOtdrTraceFileUtil;

use app\util\SMTInvalidValueException;
use app\util\SMTUtil;
use app\util\SMTRwDto;
use app\serviceshelper\otdr\SMTOtdrTraceType;
use app\settings\SMTSmartOtuThresholds;
use app\serviceshelper\monitoring\SMTPeakShiftMode;

/**
 * Monitoring test data: otdr trace name, markers, thresholds
 *
 * @author Sylvain Desplat
 */
class SMTTestDetailDto extends SMTRwDto
{   
    /**
     * @var number
     */
    protected $testId;    
    /**
     * OTDR trace name with its extesion but without its path
     * @var string
     */
    protected $otdrReferenceTraceName = NULL;
        
    /**
     * @var float
     */    
    protected $firstMarkerDistanceM = 0.0;
    /**
     * @var float
     */    
    protected $lastMarkerDistanceM = 0.0;
    /**
     * First marker attenuation thresholds
     * @SMTAttributeInfosAnnotation(classname='app\services\monitoring\SMTOtuMonitoringTestThresholdDto')
     * @var SMTOtuMonitoringTestThresholdDto
     */    
    protected $firstMarkerThresholdsDb = NULL; 
    
    /**
     * Variation of attenuation between first and last marker thresholds
     * @SMTAttributeInfosAnnotation(classname='app\services\monitoring\SMTOtuMonitoringTestThresholdDto')
     * @var SMTOtuMonitoringTestThresholdDto
     */
    protected $deltaFirstLastMarkersThresholdsDb = NULL;
    
    /**
     * The Monitoring Acceptable Noise Margin (in Db).
     * Used to display that min noise level in trace viewer and to force user to set markers outside noise level.
     * @var float
     */
    protected $monitoringAcceptableNoiseMarginDbDefault = NULL;
    
    /**
     * The attenuation critcal threshold (in Db).
     * Used to display that min noise level in trace viewer and to force user to set markers outside noise level.
     * @var float
     */
    protected $attenuationCriticalThresholdDbDefault = NULL;
    
    /**
     * The delta between TO and BACK TO thresholds for attenuation
     * @var float
     */
    protected $attenuationToBtoDeltaDbDefault = NULL;
    
    // MONITORING PEAK
    // MONITORING PEAK
    // MONITORING PEAK
    
    /**
     * Whether peak monitoring is on
     * @var BOOLEAN
     */
    protected $existingPeaksMonitored = FALSE;
    
    /**
     *
     * @var boolean FALSE by default
     */
    protected $peakMonitoringOptionEnabled = FALSE;
    
    /**
     * Peak shoft mode: REFERENCE_TRACE, TOLERANCE
     * @var string
     */
    protected $peakShiftMode = SMTPeakShiftMode::REFERENCE_TRACE;
    
    /**
     * Default Peak thresholds
     * @SMTAttributeInfosAnnotation(classname='app\services\monitoring\SMTOtuMonitoringTestThresholdDto')
     * @var SMTOtuMonitoringTestThresholdDto
     */
    protected $defaultPeakThresholdsDb = NULL;
    
    /**
     * @SMTAttributeInfosAnnotation(classname='app\services\monitoring\SMTMonitoringPeakDto',islist='true')
     * @var array app\services\monitoring\SMTMonitoringPeakDto
     */
    protected $monitoredPeaks= array();
    
    /**
     * 
     * @var int
     */
    protected $maxNumberOfMonitorablePeaks = SMTSmartOtuThresholds::DEFAULT_MAX_NUMBER_MONITORABLE_PEAKS;
    
    /**
     * The delta between TO and BACK TO thresholds for existing peaks
     * @var float
     */
    protected $peakToBtoDelta;
    
    /**
     * Missing peak shift gap:
     * @var int 
     */
    protected $missingPeakThresholdPt;
    
    /**
     * Min peak threshold
     * 
     * @var float
     */
    protected $monitorablePeakThresholdDb;
    
    /**
     * Whether the OTDR trace name is a reference trace, a new acquisition trace stored in SmartOTU directory or the last monitoring trace acquisition. 
     * 
     * @param string SMTOtdrTraceType::OTU_REFERENCE_TRACE, SMTOtdrTraceType::SMARTOTU_ACQUISITION_TRACE, SMTOtdrTraceType::LAST_TEST_ACQUISITION_TRACE
     */
    protected $otdrTraceType = SMTOtdrTraceType::OTU_REFERENCE_TRACE;
    
    /**
     * Minimum attenuation alarm threshold or 0 if none is set
     * @var float
     */
    protected $minAttenuationAlarmThreshold= 0;
    
                
    function setTestId( $id )
    {
        $this->testId = SMTUtil::convertToInt( $id );
    }
    function getTestId()
    {
    	return $this->testId;
    }

    /**
     * Set OTDR trace name (with its extension)
     * @param string $otdrReferenceTraceName
     */
    function setOtdrTraceName( $otdrReferenceTraceName )
    {
    	$this->otdrReferenceTraceName = $otdrReferenceTraceName;
    }
    /**
     * Get OTDR trace name (with its extension)
     * @return string
     */
    function getOtdrTraceName()
    {
    	return $this->otdrReferenceTraceName;
    }
    
    function getOtdrTraceNameWithoutExtension()
    {
    	return basename( $this->otdrReferenceTraceName, SMTOtdrTraceFileUtil::OTDR_FILE_SOR_EXTENSION );
    }        
    
    function setOtdrTraceType( $otdrTraceType )
    {
    	$this->otdrTraceType = $otdrTraceType;
    }
    function getOtdrTraceType()
    {
    	return $this->otdrTraceType;
    }    

    /**
     * @param float $firstMarkerDistanceM
     * @throws SMTInvalidValueException
     */
    function setFirstMarkerDistanceM( $firstMarkerDistanceM )
    {
        if ( !isset($firstMarkerDistanceM) || $firstMarkerDistanceM === NULL || !is_numeric( $firstMarkerDistanceM ) )
        {
        	throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "firstMarkerDistanceM", $firstMarkerDistanceM );
        }        
    	$this->firstMarkerDistanceM = floatval($firstMarkerDistanceM);
    }
    /** 
     * @return float
     */
    function getFirstMarkerDistanceM()
    {
    	return $this->firstMarkerDistanceM;
    }

    /**
     * 
     * @param float $lastMarkerDistanceM
     * @throws SMTInvalidValueException
     */
    function setLastMarkerDistanceM( $lastMarkerDistanceM )
    {
        if ( !isset($lastMarkerDistanceM) || $lastMarkerDistanceM === NULL || !is_numeric( $lastMarkerDistanceM ) )
        {
        	throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "lastMarkerDistanceM", $lastMarkerDistanceM );
        }        
    	$this->lastMarkerDistanceM = floatval($lastMarkerDistanceM);
    }
    /**
     * @return float
     */
    function getLastMarkerDistanceM()
    {
    	return $this->lastMarkerDistanceM;
    }

    /**
     * 
     * @param SMTOtuMonitoringTestThresholdDto $firstMarkerThresholdsDb
     */
    function setFirstMarkerThresholdsDb( $firstMarkerThresholdsDb )
    {     
    	$this->firstMarkerThresholdsDb = $firstMarkerThresholdsDb;
    }
    /**
     * 
     * @return \app\services\monitoring\SMTOtuMonitoringTestThresholdDto
     */
    function getFirstMarkerThresholdsDb()
    {
    	return $this->firstMarkerThresholdsDb;
    }  

    /**
     * @param SMTOtuMonitoringTestThresholdDto $deltaFirstLastMarkersThresholdsDb
     */
    function setDeltaFirstLastMarkersThresholdsDb( $deltaFirstLastMarkersThresholdsDb )
    {
    	$this->deltaFirstLastMarkersThresholdsDb = $deltaFirstLastMarkersThresholdsDb;
    }
    /**
     * 
     * @return \app\services\monitoring\SMTOtuMonitoringTestThresholdDto
     */
    function getDeltaFirstLastMarkersThresholdsDb()
    {
    	return $this->deltaFirstLastMarkersThresholdsDb;
    }        
    
    /**
     * Get the Monitoring Acceptable Noise Margin (in Db).
     * Used to display that min noise level in trace viewer and to force user to set markers outside noise level.
     *
     * @return float Monitoring Acceptable Noise Margin (in Db)
     */
    function getMonitoringAcceptableNoiseMarginDbDefault()
    {
    	return $this->monitoringAcceptableNoiseMarginDbDefault;
    }
    
    /**
     * Set the Monitoring Acceptable Noise Margin (in Db).
     * Used to display that min noise level in trace viewer and to force user to set markers outside noise level.
     *
     * @param float $monitoringAcceptableNoiseMarginDbDefault Monitoring Acceptable Noise Margin (in Db)
     */
    function setMonitoringAcceptableNoiseMarginDbDefault( $monitoringAcceptableNoiseMarginDbDefault )
    {
        if ( !isset($monitoringAcceptableNoiseMarginDbDefault) || $monitoringAcceptableNoiseMarginDbDefault === NULL || !is_numeric( $monitoringAcceptableNoiseMarginDbDefault ) )
        {
        	throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "monitoringAcceptableNoiseMarginDbDefault", $monitoringAcceptableNoiseMarginDbDefault );
        }        
    	$this->monitoringAcceptableNoiseMarginDbDefault = floatval($monitoringAcceptableNoiseMarginDbDefault);
    }    
    
    /**
     * Get the attenuation critical threshold (in Db).
     *
     * @return float attenuation critical threshold (in Db)
     */
    public function getAttenuationCriticalThresholdDbDefault()
    {
        return $this->attenuationCriticalThresholdDbDefault;
    }    

    /**
     * Register the attenuation critical threshold in Db
     *
     * @param float $attenuationCriticalThresholdDbDefault The attenuation critical threshold.
     */
    public function setAttenuationCriticalThresholdDbDefault( $attenuationCriticalThresholdDbDefault  )
    {
    	if ( !isset($attenuationCriticalThresholdDbDefault) || $attenuationCriticalThresholdDbDefault === NULL || !is_numeric( $attenuationCriticalThresholdDbDefault ) )
    	{
    		throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "attenuationCriticalThresholdDbDefault", $attenuationCriticalThresholdDbDefault );
    	}
    	$this->attenuationCriticalThresholdDbDefault = floatval($attenuationCriticalThresholdDbDefault);
    }    
        
    /**
     * Get the delta between TO and BACK TO thresholds for attenuation (in Db).
     *
     * @return float The delta between TO and BACK TO thresholds for attenuation (in Db)
     */
    function getAttenuationToBtoDeltaDbDefault()
    {
    	return $this->attenuationToBtoDeltaDbDefault;
    }
    
    /**
     * Set the delta between TO and BACK TO thresholds for attenuation (in Db).
     *
     * @param float $attenuationToBtoDeltaDbDefault The delta between TO and BACK TO thresholds for attenuation (in Db).
     */
    function setAttenuationToBtoDeltaDbDefault( $attenuationToBtoDeltaDbDefault )
    {
    	if ( !isset($attenuationToBtoDeltaDbDefault) || $attenuationToBtoDeltaDbDefault === NULL || !is_numeric( $attenuationToBtoDeltaDbDefault ) )
    	{
    		throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "attenuationToBtoDeltaDbDefault", $attenuationToBtoDeltaDbDefault );
    	}
    	$this->attenuationToBtoDeltaDbDefault = floatval($attenuationToBtoDeltaDbDefault);
    }
    
    //PEAK MONITORING
    //PEAK MONITORING
    //PEAK MONITORING
    //PEAK MONITORING
    //PEAK MONITORING
    public function setExistingPeaksMonitored( $existingPeaksMonitored)
    {
    	$this->existingPeaksMonitored  = filter_var($existingPeaksMonitored, FILTER_VALIDATE_BOOLEAN);
    }
    
    public function getExistingPeaksMonitored()
    {
    	return $this->existingPeaksMonitored;
    }  
    
    public function setPeakMonitoringOptionEnabled( $peakMonitoringOptionEnabled)
    {
    	$this->peakMonitoringOptionEnabled= filter_var($peakMonitoringOptionEnabled, FILTER_VALIDATE_BOOLEAN);
    }
    
    
    public function getPeakMonitoringOptionEnabled()
    {
    	return $this->peakMonitoringOptionEnabled;
    } 
    
    /**
     *
     * @param SMTOtuMonitoringTestThresholdDto $defaultPeakThresholdsDb
     */
    function setDefaultPeakThresholdsDb( SMTOtuMonitoringTestThresholdDto $defaultPeakThresholdsDb)
    {
    	$this->defaultPeakThresholdsDb = $defaultPeakThresholdsDb;
    }
    /**
     *
     * @return \app\services\monitoring\SMTOtuMonitoringTestThresholdDto
     */
    function getDefaultPeakThresholdsDb()
    {
    	return $this->defaultPeakThresholdsDb;
    } 
    
    /**
     * @param array
     */
    public function setMonitoredPeaks( $monitoredPeaks)
    {
    	$this->monitoredPeaks = $monitoredPeaks;
    }
    
    /**
     * @return array
     */
    public function getMonitoredPeaks()
    {
    	return $this->monitoredPeaks;
    }
    
    /**
     * @param SMTMonitoringPeakDto $peak
     */
    public function addMonitoredPeak( SMTMonitoringPeakDto $peak )
    {
    	array_push( $this->monitoredPeaks, $peak);
    } 
    
    function setMaxNumberOfMonitorablePeaks( $maxNumberOfMonitorablePeaks)
    {
    	$this->maxNumberOfMonitorablePeaks = SMTUtil::convertToInt( $maxNumberOfMonitorablePeaks);
    }
    
    function getMaxNumberOfMonitorablePeaks()
    {
    	return $this->maxNumberOfMonitorablePeaks;
    }
    
    /**
     *
     * @param float $peakLevelDb
     */
    function setPeakToBtoDelta( $peakToBtoDelta)
    {
    	$this->peakToBtoDelta= $peakToBtoDelta;
    }
    /**
     *
     * @return float
     */
    function getPeakToBtoDelta()
    {
    	return $this->peakToBtoDelta;
    } 
    
    /**
     *
     * @param float $monitorablePeakThresholdDb
     */
    function setMonitorablePeakThresholdDb( $monitorablePeakThresholdDb)
    {
    	$this->monitorablePeakThresholdDb= $monitorablePeakThresholdDb;
    }
    /**
     *
     * @return float
     */
    function getMonitorablePeakThresholdDb()
    {
    	return $this->monitorablePeakThresholdDb;
    } 
    
    
    function setMissingPeakThresholdPt( $missingPeakThresholdPt)
    {
    	$this->missingPeakThresholdPt= SMTUtil::convertToInt( $missingPeakThresholdPt);
    }
    
    function getMissingPeakThresholdPt()
    {
    	return $this->missingPeakThresholdPt;
    }

    function setPeakShiftMode( $peakShiftMode )
    {
    	$this->peakShiftMode= $peakShiftMode;
    }
    
    function getPeakShiftMode()
    {
    	return $this->peakShiftMode;
    }
    
    function setMinAttenuationAlarmThreshold( $minAttenuationAlarmThreshold)
    {
    	return $this->minAttenuationAlarmThreshold= $minAttenuationAlarmThreshold;
    }
    
    function getMinAttenuationAlarmThreshold()
    {
    	return $this->minAttenuationAlarmThreshold;
    }
    
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }
    
    /**
     * Whether test detail DTO has been modified on client side.
     * 
     * @return boolean Whether test detail DTO has been modified on client side. 
     */
    public function hasChanged()
    {       
    	$modified = FALSE;
    	if ( count ($this->getMonitoredPeaks()) > 0 )
    	{
    		foreach ($this->getMonitoredPeaks() as $peak)
    		{
    			if ($peak != NULL)
    			{
	    			$modified = ($peak->getCheckSum() != $peak->computeObjectDtoCheckSum()) || 
	    			            ($peak->getPeakThresholds()->getCheckSum() != $peak->getPeakThresholds()->computeObjectDtoCheckSum() );
    			}
    		}
    	}
    	
        return ($this->getCheckSum() != $this->computeObjectDtoCheckSum() 
                || $this->getFirstMarkerThresholdsDb()->getCheckSum() != $this->getFirstMarkerThresholdsDb()->computeObjectDtoCheckSum()
                || $this->getDeltaFirstLastMarkersThresholdsDb()->getCheckSum() != $this->getDeltaFirstLastMarkersThresholdsDb()->computeObjectDtoCheckSum()
        		|| $modified
                );
    }
    
    
    /**
     * Create the Dto from a Json Dto in an array
     *
     * @param array $json_array json data in an array
     *
     * @return \app\services\monitoring\SMTTestDetailDto
     */
    static function getInstance( array &$json_array )
    {
    	return self::forge( get_class(), $json_array );
    }    
            
}

?>