<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\monitoring;

use app\util\SMTUtil;

use app\util\SMTRwDto;

use app\sharedmemory\SMTMemoryManager;

use app\sharedmemory\SMTIMemorySupport;
/**
 * Test detail parameters strings formated by the OTU and converted by SmartOTU:
 * Only used for display
 * 
 * @author Sylvain Desplat
 */
class SMTTestAcquisitionParametersDto extends SMTRwDto implements SMTIMemorySupport
{
    /**
     * @var string
     */
    protected $testId;
    
    /**
     * @var string
     */
    protected $linkId;
    
    /**
     * @var int
     */
    protected $acquisitionTimeSec;
    /**
     * @var int
     */
    protected $wavelengthNm;
    /**
     * @var int
     */
    protected $pulseNs;
    /**
     * @var float
     */
    protected $rangeKm;
    /**
     * @var int
     */
    protected $resolutionCm;
    
    /**
     * Micro second constant for non SI OTU function
     * @var string
     */
    const Nano_Sec ="ns";
    /**
     * Micro second constant for non SI OTU function
     * @var string
     */
    const Micro_Sec ="µs";
    /**
     * Micro second constant for non SI OTU function
     * @var string
     */    
    const Sec ="s";
    /**
     * Centimeter constant for non SI OTU function
     * @var string
     */    
    const CM ="cm";
    /**
     * Meter constant for non SI OTU function
     * @var string
     */    
    const M ="m";
    /**
     * KiloMeter constant for non SI OTU function
     * @var string
     */
    const KM ="km";
            
    /**
     *  WARNING NOT SI: 10 s ...
     *
     * @param string $acquisitionTime
     */
    public function setAcquisitionTime( $acquisitionTime )
    {
    	$this->setAcquisitionTimeSec( substr($acquisitionTime, 0, -2) );
    }
    
    /**
     * 
     * @param int $acquisitionTimeSec
     */
    public function setAcquisitionTimeSec( $acquisitionTimeSec )
    {
    	$this->acquisitionTimeSec = intval( $acquisitionTimeSec );
    }
    /**
     * 
     * @return int
     */
    public function getAcquisitionTimeSec()
    {
    	return $this->acquisitionTimeSec;
    }
    
    /**
     *  WARNING NOT SI: 1625 nm ...
     *      
     * @param string $wavelength
     */
    public function setWavelength( $wavelength )
    {
    	$this->setWavelengthNm( substr($wavelength, 0, -3) );
    }
    
    /**
     * 
     * @param int $wavelengthNm
     */
    public function setWavelengthNm( $wavelengthNm )
    {
    	$this->wavelengthNm = intval( $wavelengthNm );
    }

    /**
     * 
     * @return int
     */
    public function getWavelengthNm()
    {
    	return $this->wavelengthNm;
    }
    
    /**
     * @param int $pulseNs
     */
    public function setPulseNs( $pulseNs )
    {    
    	$this->pulseNs = intval( $pulseNs );
    }
    
    /**
     * WARNING NOT SI: 5 µs or 10 ns ...
     * @param string $pulse
     */
    public function setPulse( $pulse )
    {
        if ( strpos($pulse, self::Nano_Sec) !== FALSE )
        {
            $this->setPulseNs( substr( $pulse, 0, -3)  );
        }
        else if ( strpos($pulse, self::Micro_Sec) !== FALSE )
        {
        	$this->setPulseNs( intval( substr( $pulse, 0, -3) ) * 1000.0 );
        }        
        else if ( strpos($pulse, self::Sec) !== FALSE )
        {
        	$this->setPulseNs( intval( substr( $pulse, 0, -2) ) * 1000000.0 );
        }
    }    

    /**
     * @return int
     */
    public function getPulseNs()
    {
    	return $this->pulseNs;
    }
    

    /**
     * WARNING NOT SI: 10000 m 100 km ...
     * @param string $pulse
     */
    public function setRange( $range )
    {
        if ( strpos($range, self::KM) !== FALSE )
        {
        	$this->setRangeKm( floatval( substr( $range, 0, -3) ) );
        }
        else if ( strpos($range, self::M) !== FALSE )
        {
        	$this->setRangeKm( floatval( substr( $range, 0, -2) ) / 1000.0 );
        }
    }
    
    /**
     * 
     * @param float $rangeKm
     */
    public function setRangeKm( $rangeKm )
    {                
    	$this->rangeKm = floatval( $rangeKm );
    }
    
    /**
     * 
     * @return float 
     */
    public function getRangeKm()
    {
    	return $this->rangeKm;
    }
    
    /**
     * WARNING NOT SI: 5cm or 1m ...
     * @param string $resolution
     */
    public function setResolution( $resolution )
    {
    	if ( strpos($resolution, self::CM) !== FALSE )
    	{
    		$this->setResolutionCm( substr( $resolution, 0, -2)  );
    	}
    	else if ( strpos($resolution, self::M) !== FALSE )
    	{
    		$this->setResolutionCm( substr( $resolution, 0, -1) * 100 );
    	}
    }
        
    /**
     * 
     * @param int $resolutionCm
     */
    public function setResolutionCm( $resolutionCm )
    {
    	$this->resolutionCm = intval( $resolutionCm );
    }
    /**
     * @return int
     */
    public function getResolutionCm()
    {
    	return $this->resolutionCm;
    }
        
    public function setTestId( $testId )
    {
    	$this->testId = SMTUtil::convertToInt($testId);
    }
    
    public function getTestId()
    {
    	return $this->testId;
    }
    
    public function setLinkId( $linkId )
    {
    	$this->linkId = SMTUtil::convertToInt($linkId);
    }
    
    public function getLinkId()
    {
    	return $this->linkId;
    }
    
    /**
     * Test identifier
     *
     * @return string
     */
    public function getId()
    {
    	return $this->testId;
    }
    
    /**
     * Save Dto in shared memory
     *
     * @throws SMTAPCMemoryException
     */
    public function save()
    {
    	SMTMemoryManager::saveDto( $this );
    }
    
    /**
     * Delete Dto from shared memory
     *
     * @throws SMTAPCMemoryException
     */
    public function delete()
    {
    	SMTMemoryManager::deleteDto( $this );
    }
        
    /**
     * Returns the dto class name.
     *
     * @return the dto class name.
     */
    public static function getClass()
    {
    	return __CLASS__;
    }
    
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }    
    
    /**
     * Otu OTDR module cache is never obsolete: it must not be purged
     *
     * @return FALSE
     */
    public function isObsolete()
    {
    	return FALSE;
    }
    
    /**
     * No obsolescence for those data: returns current time.
     *
     * @see isObsolete()
     * @return number Timestamp of the last APC memory access (used to handle obsolescence of dtos in cache)
     */
    public function getLastAccess()
    {
    	return microtime(TRUE);
    }
    
    /**
     * No obsolescence for those data: do nothing
     *
     * @see isObsolete()
     *
     */
    public function touchAccess()
    {
    }    
}
?>