<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\alarm;

use app\util\SMTUnit;

use app\database\SMTIExecuteInTransaction;

use app\database\SMTSmartOtuDB;

use app\serviceshelper\alarm\SMTAlarmSpecificProblemCode;

/**
 * Attenuation optical alarm Dto
 * 
 * @author Sylvain Desplat
 * @SMTClassTableNameAnnotation(tablename='optical_alarm', tablekey='id') 
 */
class SMTAttenuationOpticalAlarmDto extends SMTOpticalAlarmDto
{    
    /**
     * @SMTAttributeInfosAnnotation(classname='app\services\alarm\SMTAttenuationOpticalAlarmEventDto', id='id', tablename='attenuation_optical_alarm_event')
     * @var SMTAttenuationOpticalAlarmEventDto
     */
    protected $currentAlarmEvent;
        
    const FETCH_ATTENUATION_OPTICAL_ALARM = "SELECT optical_alarm.* FROM optical_alarm, attenuation_optical_alarm_event WHERE optical_alarm.id=:id AND optical_alarm.current_alarm_event_id=attenuation_optical_alarm_event.id";
    
    const FETCH_ATTENUATION_OPTICAL_ALARM_BY_DESCRIMINATOR = "SELECT optical_alarm.* FROM optical_alarm, attenuation_optical_alarm_event WHERE discriminator=:alarmDiscriminator AND optical_alarm.current_alarm_event_id=attenuation_optical_alarm_event.id";
    
    const FETCH_ATTENUATION_OPTICAL_ALARM_BY_TESTID = "SELECT optical_alarm.* FROM optical_alarm, attenuation_optical_alarm_event WHERE test_id=:testId AND optical_alarm.current_alarm_event_id=attenuation_optical_alarm_event.id";
    
    const FETCH_ALL_ATTENUATION_OPTICAL_ALARM = "SELECT optical_alarm.* FROM optical_alarm, attenuation_optical_alarm_event WHERE optical_alarm.current_alarm_event_id=attenuation_optical_alarm_event.id";
    
    const DELETE_ALL_ATTENUATION_OPTICAL_ALARM = "DELETE FROM optical_alarm";
    
    /**
     * Fetch the attenuation optical alarm for the given alarm descriptor
     *
     * @param \SQLite3 $dbConnection
     * @param string $alarmDiscriminator
     * @param boolean $joinFetch FALSE by default: TRUE to force the retrieval of joined properties (optical alarm last event)
     *
     * @return SMTAttenuationOpticalAlarmDto
     */
    public static function fetch( \SQLite3 $dbConnection, $alarmDiscriminator, $joinFetch = FALSE )
    {
        $query=self::FETCH_ATTENUATION_OPTICAL_ALARM_BY_DESCRIMINATOR;
        $statement = $dbConnection->prepare( $query);
        
        self::handleStatementError( $dbConnection, $statement, "Prepare select", $query, NULL, __FILE__,__METHOD__,__LINE__ );
        
        $bindParameters = ':alarmDiscriminator=>'.$alarmDiscriminator;
        $statement->bindValue(':alarmDiscriminator', $alarmDiscriminator, SQLITE3_TEXT );
        $resultSet = $statement->execute();
        
        self::handleStatementError( $dbConnection, $resultSet, "Select", $query, $bindParameters, __FILE__,__METHOD__,__LINE__ );
        
    	$opticalAlarm = parent::getSingleInstanceFromResultSet($resultSet, get_class() );
    	
    	//translates alarm specific problem
    	if ( $opticalAlarm != NULL )
    	{
    	    $opticalAlarm->setSpecificProblemText( SMTAlarmSpecificProblemCode::decodeOpticalAlarmSpecificProblem( $opticalAlarm->getSpecificProblemCode(), $opticalAlarm->isCleared() ) );
    	}
    	
    	if ( $opticalAlarm != NULL && $opticalAlarm instanceof SMTAttenuationOpticalAlarmDto)
    	{
    	    self::decodeSpecificProblemAndJoinFetchAlarmEvent($dbConnection, $opticalAlarm, $joinFetch );
    	}
    	
    	return $opticalAlarm;
    }       
    
    /**
     * Fetch the attenuation optical alarm for the given testId
     *
     * @param \SQLite3 $dbConnection
     * @param string $testId
     * @param boolean $joinFetch FALSE by default: TRUE to force the retrieval of joined properties (optical alarm last event)
     *
     * @return SMTAttenuationOpticalAlarmDto or NULL if no alarm found
     */
    public static function fetchFromTest( \SQLite3 $dbConnection, $testId, $joinFetch = FALSE )
    {
    	$query=self::FETCH_ATTENUATION_OPTICAL_ALARM_BY_TESTID;
    	$statement = $dbConnection->prepare( $query);
    
    	self::handleStatementError( $dbConnection, $statement, "Prepare select", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    
    	$bindParameters = ':testId=>'.$testId;
    	$statement->bindValue(':testId', $testId, SQLITE3_TEXT );
    	$resultSet = $statement->execute();
    
    	self::handleStatementError( $dbConnection, $resultSet, "Select", $query, $bindParameters, __FILE__,__METHOD__,__LINE__ );
    
    	$opticalAlarm = parent::getSingleInstanceFromResultSet($resultSet, get_class() );
    	 
    	//translates alarm specific problem
    	if ( $opticalAlarm != NULL )
    	{
    	    self::decodeSpecificProblemAndJoinFetchAlarmEvent($dbConnection, $opticalAlarm, $joinFetch );
    	}
    	 
    	return $opticalAlarm;
    }    
    
    /**
     * Fetch the attenuation optical alarm with its event for the given alarm id
     *
     * @param \SQLite3 $dbConnection
     * @param int $id alarm id
     *
     * @return SMTAttenuationOpticalAlarmDto
     */
    public static function joinFetch( \SQLite3 $dbConnection, $id )
    {
    	$query=self::FETCH_ATTENUATION_OPTICAL_ALARM;
    	$statement = $dbConnection->prepare( $query);
    
    	self::handleStatementError( $dbConnection, $statement, "Prepare select", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    
    	$bindParameters = ':id=>'.$id;
    	$statement->bindValue(':id', $id, SQLITE3_TEXT );
    	$resultSet = $statement->execute();
    
    	self::handleStatementError( $dbConnection, $resultSet, "Select", $query, $bindParameters, __FILE__,__METHOD__,__LINE__ );
    
    	$opticalAlarm = parent::getSingleInstanceFromResultSet($resultSet, get_class() );
    	 
    	if ( $opticalAlarm != NULL )
    	{
    	    self::decodeSpecificProblemAndJoinFetchAlarmEvent($dbConnection, $opticalAlarm, TRUE );
    	}
    	 
    	return $opticalAlarm;
    }    
    
    /**
     * Fetch all the attenuation optical alarms
     *
     * @param \SQLite3 $dbConnection
     * @param boolean $joinFetch FALSE by default: TRUE to force the retrieval of joined properties (optical alarm last event)
     *
     * @return array SMTAttenuationOpticalAlarmDto
     */
    public static function fetchAll( \SQLite3 $dbConnection, $joinFetch = FALSE )
    {
        $opticalAlarms = array();
        
    	$query=self::FETCH_ALL_ATTENUATION_OPTICAL_ALARM;
    	$statement = $dbConnection->prepare( $query);
    
    	self::handleStatementError( $dbConnection, $statement, "Prepare select", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    
    	$resultSet = $statement->execute();
    
    	self::handleStatementError( $dbConnection, $resultSet, "Select", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    
    	$opticalAlarms = parent::getListFromResultSet($resultSet, get_class() );
    	
    	foreach ( $opticalAlarms as &$alarm )
    	{
    	    //translates alarm specific problem
    	    if ( $alarm != NULL )
    	    {
                self::decodeSpecificProblemAndJoinFetchAlarmEvent($dbConnection, $alarm, $joinFetch );
    	    }
    	}
    	 
    	return $opticalAlarms;
    }    
    
    /**
     * 
     * @param \SQLite3 $dbConnection
     * @param SMTAttenuationOpticalAlarmDto $opticalAlarm
     * @param string $joinFetch FALSE by default: TRUE to force the retrieval of joined properties (optical alarm last event)
     */
    private static function decodeSpecificProblemAndJoinFetchAlarmEvent( \SQLite3 $dbConnection, SMTAttenuationOpticalAlarmDto $opticalAlarm, $fetchAlarmEvent = FALSE)
    {
        $opticalAlarmEvent = NULL;
        //if the alarm has a valid distance, add it to the specific problem:
        if ( $fetchAlarmEvent != TRUE && !$opticalAlarm->isCleared() && SMTAlarmSpecificProblemCode::hasSpecificProblemWithDistance( $opticalAlarm->getSpecificProblemCode() ) )
        {
        	$fetchAlarmEvent = TRUE;
        }
        
        //Join fetch attenuation optical alarm event
        if ( $fetchAlarmEvent )
        {
        	$opticalAlarmEvent = SMTAttenuationOpticalAlarmEventDto::fetch($dbConnection, $opticalAlarm->getId());
        	$opticalAlarm->setCurrentAlarmEvent( $opticalAlarmEvent );        	
        }
        
        $opticalAlarm->updateSpecificProblemText();
    }
    
    /**
     * Update specific problem text translation and append the fault distance if it is valid.
     * 
     */
    function updateSpecificProblemText()
    {
        $opticalAlarmEvent = $this->getCurrentAlarmEvent();
        $this->setSpecificProblemText( SMTAlarmSpecificProblemCode::decodeOpticalAlarmSpecificProblem( $this->getSpecificProblemCode(), $this->isCleared() ) );
        if ( $opticalAlarmEvent != NULL && $opticalAlarmEvent->isDistanceValid() )
        {
        	$text = $this->getSpecificProblemText()." (".SMTUnit::formatToUnitWithBestRange( $opticalAlarmEvent->getDistanceM() ).")";
        	$this->setSpecificProblemText( $text );
        }
    } 
    
    /**
     * Delete all attenuation optical alarms
     * 
     * @param SMTSmartOtuDB $dbConnection
     */
    static function performDeleteAll( SMTSmartOtuDB $dbConnection )
    {
		$query = self::DELETE_ALL_ATTENUATION_OPTICAL_ALARM;
		$statement = $dbConnection->prepare( $query);

		self::handleStatementError( $dbConnection, $statement, "Prepare delete", $query, NULL, __FILE__,__METHOD__,__LINE__ );

		$result = $statement->execute();

		self::handleStatementError( $dbConnection, $result, "Delete", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    }    
    
    /**
     * Save dto in database and save joined dtos (alarm event)
     *
     * @param SMTSmartOtuDB $dbConnection
     */
    public function save( SMTSmartOtuDB $dbConnection)
    {
    	$dbConnection->runInTransaction( new SMTSaveAttenuationAlarm( $this ) );
    }
    
    /**
     * Save dto in database and save joined dtos (alarm event)
     * 
     * @param SMTSmartOtuDB $dbConnection
     * 
     */
    function performSaveAttenuationAlarm( SMTSmartOtuDB $dbConnection )
    {
        parent::save($dbConnection );
        //update alarm event if it exists
        if ( $this->getCurrentAlarmEvent() != NULL )
        {
            $this->getCurrentAlarmEvent()->setAlarmId( $this->id );
            $this->getCurrentAlarmEvent()->save($dbConnection);
            
            //update current alarm event in alarm
            $this->updateSingleProperty( $dbConnection, "currentAlarmEvent" );                    
        }
    }
    
    /**
     *
     * @param SMTAttenuationOpticalAlarmEventDto $currentAlarmEvent
     */
    public function setCurrentAlarmEvent( SMTAttenuationOpticalAlarmEventDto $currentAlarmEvent )
    {
    	$this->currentAlarmEvent = $currentAlarmEvent;
    }
    /**
     *
     * @return SMTAttenuationOpticalAlarmEventDto
     */
    public function getCurrentAlarmEvent()
    {
    	return $this->currentAlarmEvent;
    }   
    
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }
}

/**
 * Save attenuation alarm in a transaction
 *
 * @author Sylvain Desplat
 */
class SMTSaveAttenuationAlarm implements SMTIExecuteInTransaction
{
	/**
	 *
	 * @var SMTAttenuationOpticalAlarmDto
	 */
	var $alarm;

	/**
	 *
	 * @param SMTAlarmDto $alarm
	 */
	function __construct( SMTAttenuationOpticalAlarmDto $alarm )
	{
		$this->alarm = $alarm;
	}

	/**
	 *
	 */
	public function run( SMTSmartOtuDB $dbConnection )
	{
		$this->alarm->performSaveAttenuationAlarm($dbConnection);
	}

	/**
	 * @return SMTResultSetDto
	 */
	public function getResultSetDto()
	{
		return $this->alarm;
	}
}
?>