import socket, sys, time

class EsrError(Exception):
    strings = {\
    16:"Impossible to execute",\
    32:"Syntax error",\
    }
    
    def __init__(self, value):
        self.value = value
    
    def __str__(self):
        return "ESR("+str(self.value)+"):"+self.strings[self.value]
    
    def displayError(self):
        print(self.strings[self.value])
    

class ConnectionError(Exception):
    def __init__(self, value):
        self.value = value
    def __str__(self):
        return str(self.value)



class ScpiAccess(object):
    """ SCPI Socket access """
    __max_buffer_size = 128
    
    def __init__(self, host, port, timeout = 5):
        self.__host         = host
        self.__port         = int(port)
        self.__isConnected  = False
        self.__timeout      = timeout
        self.__socket       = None
        self.__buff         = []
    
    def connect(self):
        if not self.__isConnected:
            self.__socket = socket.socket(socket.AF_INET,socket.SOCK_STREAM)
            self.__socket.settimeout(self.__timeout)
            self.__socket.connect((self.__host,self.__port))  
            if (sys.version_info < (3,0,0)):
                self.__socket.send("*rem\n")
            else:
                self.__socket.send(bytes("*rem\n",'ascii'))                
            self.__isConnected = True
            self.verify()

    def __readline(self):
        if self.__isConnected:
            if len(self.__buff) == 0:
                if (sys.version_info < (3,0,0)):
                    self.__buff = [l.strip() for l in self.__socket.recv(self.__max_buffer_size).split("\n") if len(l) > 0]
                    self.__buff.reverse()
                else :
                    self.__buff = [l.strip() for l in str(self.__socket.recv(self.__max_buffer_size), 'ascii').split("\n") if len(l) > 0]
                    self.__buff.reverse()
            if len(self.__buff):
                return self.__buff.pop()
            else:
                return ""
        else:
            raise ConnectionError("Not connected")
    
    def __writeline(self,val):
        self.connect()
        if (sys.version_info < (3,0,0)):
            self.__socket.send(val.strip()+"\n")
        else:
            self.__socket.send(bytes(val.strip()+"\n",'ascii'))  
    
    def __purge(self):
        self.__readline()
        self.__buff = []

    
    def disconnect(self):
        if self.__isConnected:
            self.__socket.close()
            self.__socket = None
            self.__isConnected = False
    
    def verify(self):
        tempo = 0
        cont = 1
        count = 0
        while (cont == 1) and (count < 3):
            cont = 0
            self.__writeline("*esr?\n")
            time.sleep(tempo)
            try:
                r = self.__readline()
                res = int(r)
            except ValueError as err:
                print("Error in verify (%s)"%r)
                self.__purge()
                tempo = 0.2
                count += 1
                cont = 1
                res=-1
        if res != 0:
            raise EsrError(res)
    
    def SendCommand(self,command,timeout=0,verbose=0):
        """ Send a SCPI command and check the ESR register"""
        if verbose:
            print(command)
        self.__writeline(command+"\n")
        if(timeout != 0):
            oldtimeout = self.__timeout 
            self.__timeout = timeout
        self.verify()
        if(timeout != 0):
            self.__timeout = oldtimeout

        
    def SendCommandHalt(self,command,verbose=0):
        if verbose:
            print(command)
        self.__writeline(command+"\n")
    
    def SendAndReadCommand(self,command,timeout=0,verbose=0):
        """ Send a SCPI query, check the ESR egister and return the query result"""
        tempo = 0
        cont = 1
        count = 0
        pterr = None
        while (cont == 1) and (count < 5):
            cont = 0
            self.__writeline(command+"\n")
            if(timeout != 0):
                oldtimeout = self.__timeout 
                self.__timeout = timeout
            res = self.__readline()
            time.sleep(tempo)
            try :
                self.verify()
                if(timeout != 0):
                    self.__timeout = oldtimeout
                if verbose:
                    print(command + "\t\t : " + res)
            except ValueError as err:
                tempo += 0.1
                count += 1
                cont = 1
                pterr = err
                print("")
                print(pterr)
                print("")
            except EsrError as err:
                tempo += 0.1
                count += 1
                cont = 1
                pterr = err
                print("")
                print(pterr)
                print("")
        if (count >= 5) : 
            if verbose:
                for l in res:
                    print(l.strip())
            raise pterr
        else : 
            if verbose:
                print(command + "\t\t : " + res)
            return res.strip()

    def query(self, command, *args):
        if '?' in command.split(" ")[0]:
            return self.SendAndReadCommand(command, *args)
        else:
            return self.SendCommand(command, *args)

if __name__ == "__main__":
    u = ScpiAccess("10.33.17.84", "8000")
    u.connect()
    print(u.SendAndReadCommand("*idn?"))
