<?php
namespace app\util\upload;

//chdir to root dir
chdir('../../../');

/**
 * include autoloader
 */
require_once('lib/restler.php');

use app\serviceshelper\maintenance\SMTOtuBackupRestore;
use Luracast\Restler\Format\JsonFormat;
use app\util\SMTLogger;
use app\services\maintenance\SMTUploadStatusDto;
use app\util\upload\SMTUploadStatus;

/**
 * Check php upload errors
 * Handle copy in the target directory 
 * 
 * @param string $path
 * @param boolean $isBackup Whether the uploaded file is a backup
 *
 * @return SMTUploadStatusDto
 */
function upload_file( $path, $isBackup = FALSE )
{
    $uploadStatus = new SMTUploadStatusDto();
    SMTLogger::getInstance()->trace( "Temporary uploaded file: ".$_FILES["datafile"]["tmp_name"]);

    //retrieve errors for php upload in its temporary directory
	$err=$_FILES["datafile"]["error"];
	$fileName = $_FILES["datafile"]["name"];
	
	$uploadStatus->setFileName($fileName);
		
	//process php error status at the end of the upload
	if ($err > 0)
	{
	    $message='Upload failed! ';	    
	    $uploadStatus->setProgress( SMTUploadStatusDto::NO_UPLOAD );
	    
		switch($err)
		{
			case '1':
				$message.='php.ini max file size exceeded.';
				$uploadStatus->setUploadErrorStatus( SMTUploadStatusDto::MAX_FILE_SIZE_EXCEEDED );
				break;
			case '2':
				$message.='max file size exceeded.';
				$uploadStatus->setUploadErrorStatus( SMTUploadStatusDto::MAX_FILE_SIZE_EXCEEDED );
				break;
			case '3':
				$message.='file upload was only partial.';
				$uploadStatus->setUploadErrorStatus( SMTUploadStatusDto::PARTIAL_FILE_UPLOAD );
				break;
			case '4':
				$message.='no file was attached.';
				$uploadStatus->setUploadErrorStatus( SMTUploadStatusDto::NO_FILE_ATTACHED );
				break;
			case '7':
				$message.='Error: file permission denied.';
				$uploadStatus->setUploadErrorStatus( SMTUploadStatusDto::FILE_PERMISSION_DENIED );
				break;
			default :
				$message.='Unexpected error occured.';
				$uploadStatus->setUploadErrorStatus( SMTUploadStatusDto::UNEXPECTED_ERROR );
				break;
		}
		
		SMTLogger::getInstance()->trace($message, SMTLogger::ERROR );
	}
	//no php error at the end of the upload
	else
	{
	    $message = "";
	    
	    //we replace existing file
		if (file_exists($path.$_FILES["datafile"]["name"]))
		{
			$message='file already exist.';
		}
		
        if ( $isBackup )
        {
            //remove old backups
            array_map('unlink', glob( $path.SMTOtuBackupRestore::OTU_BACKUP_NAME_PREFIX."*".SMTOtuBackupRestore::OTU_BACKUP_NAME_SUFFIX ) );
        }
		
		//move file to its target destination
		$success = @move_uploaded_file($_FILES["datafile"]["tmp_name"], $path.$fileName);
		
		if ( $success )
		{
    		$uploadStatus->setUploadErrorStatus( SMTUploadStatusDto::NO_ERROR );
    		$uploadStatus->setProgress( SMTUploadStatusDto::UPLOAD_COMPLETE );
    		
    		//retrieve upload rate from cache
    		$cacheDto = SMTUploadStatus::getCurrentUploadInfoFromCache();
    		if ( $cacheDto != NULL )
    		{
        		$uploadRate = $cacheDto->getRateMBs();
        		$uploadStatus->setRateMBs( $uploadRate );
    		}
    		$message.=' File '.$fileName.' has been successfully uploaded.';
    		SMTLogger::getInstance()->trace($message);
		}
		//failed to copy file to its target destination
		else
		{
		    $uploadStatus->setUploadErrorStatus( SMTUploadStatusDto::COPY_FAILED );
		    $uploadStatus->setProgress( SMTUploadStatusDto::NO_UPLOAD );
		    
		    $message.=' Error: Failed to copy temporary file to destination file '.$path.$fileName;
		    SMTLogger::getInstance()->trace($message, SMTLogger::ERROR);
		}
	}
	
	//upload is finished, remove the php upload progress from APC
	SMTUploadStatus::cleanPhpUploadProgressFromApcCache();
	
	//save the upload status in APC cache
	$uploadStatus->save();
	
	return $uploadStatus;
}

/**
 * Build JSON response
 * 
 * @param SMTUploadStatusDto $uploadStatus
 * @return string
 */
function buildJsonResponse( SMTUploadStatusDto $uploadStatus )
{
    $uploadStatusJson = $uploadStatus->getJsonData();
    
    @header('Cache-Control: no-cache, must-revalidate');
    @header('Expires: 0');
    @header('Content-Type: text/html; charset=utf-8');
    
    $responseFormat = new JsonFormat();
    $responseFormat->setCharset('utf-8');
    
    return $responseFormat->encode( $uploadStatusJson, FALSE );
}

?>