<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\util;

use app\serviceshelper\system\SMTSystemConfigurationDto;

class SMTUtil
{   
    const PHP_CLIENT_EXECUTABLE = '/usr/bin/env php ';
    const SMARTOTU_TEMP_DIR = "/otu/smartOTU/tmp/";
    
    /**
     * Deep copy of an array with object cloning.
     * 
     * @param array $array the array to copy
     * 
     * @return $array
     */
    public static function arrayCopy( array $array ) 
    {
    	$result = array();
    	foreach( $array as $key => $val ) 
    	{
    		if( is_array( $val ) ) {
    			$result[$key] = self::arrayCopy( $val );
    		} 
    		elseif ( is_object( $val ) ) {
    			$result[$key] = clone $val;
    		} 
    		else {
    			$result[$key] = $val;
    		}
    	}
    	return $result;
    }
    
    /**
     * Split a string separated by comma (handles comma between double quotes), and trim the array elements and remove their double quotes
     * 
     * @param string $stringToSplit
     * 
     * @return array
     */
    public static function splitStringByComma( $stringToSplit )
    {
//         SMTLogger::getInstance()->trace($stringToSplit);
        preg_match_all("/(\"[^\"]+\"|[^,]+)/", $stringToSplit, $matches );
//         SMTLogger::getInstance()->trace(self::serialize_array( $matches[0] ));
        return self::removeQuotesInArray( $matches[0] );
    }
    
    /**
     * Remove quotes (and trim array elements) in the given array
     * 
     * @param array $array
     * @return array
     */
    public static function removeQuotesInArray( &$array )
    {
        if ( is_array( $array ) )
        {
        	foreach( $array as $key=>$value)
        	{
        		$array[$key] = trim( str_replace("\"","",$value) );
        	}
        }
    	return $array;
    }
    
    /**
     * Remove line feed in the given array
     *
     * @param array $array
     * @return array
     */    
    public static function removeLfInArray( &$array )
    {
    	if ( is_array( $array ) )
    	{
    		foreach( $array as $key=>$value)
    		{
    			$array[$key] = trim( str_replace('\n',"",$value) );
    		}
    	}
    	return $array;
    }
    
    /**
     * Test Whether the given value is an integer or a long (doestn't exist in php: converted in float)...
     */
    public static function isInt( $value )
    {
    	$value = preg_match('/^-?[0-9]{1,}$/', trim($value));
    
    	if ($value == 1)
    		return TRUE;
    	else
    		return FALSE;
    }    
    
    /**
     * Convert the given variable to an integer (or a float if value > PHP_INT_MAX) if it is a numeric value or returns NULL in all other cases
     * 
     * @param $value The value to convert
     * 
     * @return an integer (or a float if value > PHP_INT_MAX) if it is a numeric value or returns NULL in all other cases
     */
    public static function convertToInt( $value )
    {
        if ( isset($value) && $value !== NULL)
        {
        	$match = preg_match('/^-?[0-9]{1,}$/', trim($value));
        
        	if ( $match == 1 )
        	{
        	    if ( ($value + 0) <= PHP_INT_MAX ) 
        	        return intval( $value );
        	    else
        	        return floatval( $value );
        	}        		
        	else
        		return NULL;
        }
        return NULL;
    }
    
    /**
     * Convert the given variable to a boolean 
     *
     * @param var $value
     *
     * @return boolean
     */
    public static function convertToBoolean( $value )
    {
    	return filter_var( $value, FILTER_VALIDATE_BOOLEAN );
    }
    
     /**
      *  Recursively dump an array
      *  
      * @param array $array
      * 
      * @return string serialized array string
      */
    public static function serialize_array( $array )
    {
    	$serializedValue = $array;
    	
    	if ( is_array( $array ) )
    	{
    	    $serializedValue = "";
        	foreach($array as $key => $value)
        	{
        
        		if (is_array( $value ))
        		{
        			$serializedValue .= " @@ ".$key."=> { ".self::serialize_array( $value ) ." } ";
        		}
        		else
        		{
        			$serializedValue .= " @@ ".$key."=>".$value;
        		}
        	}
    	}

    	return $serializedValue;
    }
    
    /**
     * Convert OTU timestamp in seconds and in current timezone to timestamp in UTC
     * 
     * @param string $otuTimestamp OTU timestamp in seconds and in current timezone
     * @param \SQLite3 $dbConnection
     * @param integer $timezoneOffset the timesone offset (if not set, fetch it from database)
     *
     * @return integer/float timestamp in seconds and in UTC
     */
    public static function getTimestampUTC( $otuTimestamp, \SQLite3 $dbConnection, $timezoneOffset = NULL )
    {
    	$timestamp = ( isset( $otuTimestamp) && self::isInt( $otuTimestamp) )? self::convertToInt( $otuTimestamp ): NULL;
    
    	if ( $timestamp !== NULL )
    	{
        	//retrieve timezone offset from database settings if it is not given
        	if ( $timezoneOffset === NULL )
        	{
            	$dto = SMTSystemConfigurationDto::fetch($dbConnection);
            	$timezoneOffset = $dto->getTimezoneSec();
        	}
        	if ( isset($timezoneOffset) &&  $timezoneOffset != NULL )
        	{
        		return ( $timestamp - $timezoneOffset );
        	}
        	return $timestamp;
    	}
    	else
    	{
    	    return 0;
    	}
    }    

    /**
     * Convert UTC timestamp in seconds to timestamp in current timezone
     *
     * @param string $timestampUtc  UTC timestamp in seconds
     * @param \SQLite3 $dbConnection
     * @param integer $timezoneOffset the timesone offset (if not set, fetch it from database)
     *
     * @return integer/float timestamp in current timezone
     */
    public static function getTimestamp( $timestampUtc, \SQLite3 $dbConnection, $timezoneOffset = NULL )
    {
    	$timestamp = $timestampUtc;
    
    	if ( $timestamp !== NULL )
    	{
    		//retrieve timezone offset from database settings if it is not given
    		if ( $timezoneOffset === NULL )
    		{
    			$dto = SMTSystemConfigurationDto::fetch($dbConnection);
    			$timezoneOffset = $dto->getTimezoneSec();
    		}
    		if ( isset($timezoneOffset) &&  $timezoneOffset != NULL )
    		{
    			return ( $timestamp + $timezoneOffset );
    		}
    		return $timestamp;
    	}
    	else
    	{
    		return 0;
    	}
    }    
    
    /**
     * Whether the string starts with $needle
     * @param string $haystack The string to search in
     * @param string $needle The string to search
     * @return boolean
     */
    public static function startsWith($haystack, $needle)
    {
    	return !strncmp($haystack, $needle, strlen($needle));
    }
    
    /**
     * Whether the string ends with $needle
     * @param string $haystack The string to search in
     * @param string $needle The string to search
     * @return boolean
     */
    public static function endsWith($haystack, $needle)
    {
    	$length = strlen($needle);
    	if ($length == 0) 
    	{
    		return true;
    	}
    
    	return (substr($haystack, -$length) === $needle);
    }
    
    /**
     * PHP escapeshellarg with unicode/utf-8 support:
     * Escape a string to be used as a shell argument
     *
     * @return string
     */
    public static function escapeShellArgUtf8($arg)
    {
    	return ( ($arg !== NULL) && isset($arg) && ("\0" != $arg) )? "'".str_replace("'", "'\\''", $arg)."'" : "";
    }
}
?>