<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\settings;

use app\services\monitoring\SMTOtuMonitoringTestThresholdDto;

/**
 * SmartOTU thresholds:
 * - retrieve and edit otu thresholds from .ini file if they are not found in cache memory.
 * when otu thresholds are edited, they are saved in both the cache memory and the .ini file 
 * 
 * @author Sylvain Desplat
 */
class SMTSmartOtuThresholds
{
    const SMARTOTU_THRESHOLDS_FILE = "data/smartotuthresholds.ini";
        
    const FM_NEGATIVE_BACK_TO_MINOR_THRESHOLD = "FM_NEGATIVE_BACK_TO_MINOR_THRESHOLD";
    const FM_NEGATIVE_BACK_TO_NORMAL_THRESHOLD = "FM_NEGATIVE_BACK_TO_NORMAL_THRESHOLD";
    const FM_NEGATIVE_TO_MINOR_THRESHOLD = "FM_NEGATIVE_TO_MINOR_THRESHOLD";
    const FM_NEGATIVE_TO_MAJOR_THRESHOLD = "FM_NEGATIVE_TO_MAJOR_THRESHOLD";
    const FM_POSITIVE_BACK_TO_MINOR_THRESHOLD = "FM_POSITIVE_BACK_TO_MINOR_THRESHOLD";
    const FM_POSITIVE_BACK_TO_NORMAL_THRESHOLD = "FM_POSITIVE_BACK_TO_NORMAL_THRESHOLD";
    const FM_POSITIVE_TO_MINOR_THRESHOLD = "FM_POSITIVE_TO_MINOR_THRESHOLD";
    const FM_POSITIVE_TO_MAJOR_THRESHOLD = "FM_POSITIVE_TO_MAJOR_THRESHOLD";
    
    const BUDGET_VARIATION_NEGATIVE_BACK_TO_MINOR_THRESHOLD = "BUDGET_VARIATION_NEGATIVE_BACK_TO_MINOR_THRESHOLD";
    const BUDGET_VARIATION_NEGATIVE_BACK_TO_NORMAL_THRESHOLD = "BUDGET_VARIATION_NEGATIVE_BACK_TO_NORMAL_THRESHOLD";
    const BUDGET_VARIATION_NEGATIVE_TO_MINOR_THRESHOLD = "BUDGET_VARIATION_NEGATIVE_TO_MINOR_THRESHOLD";
    const BUDGET_VARIATION_NEGATIVE_TO_MAJOR_THRESHOLD = "BUDGET_VARIATION_NEGATIVE_TO_MAJOR_THRESHOLD";
    const BUDGET_VARIATION_POSITIVE_BACK_TO_MINOR_THRESHOLD = "BUDGET_VARIATION_POSITIVE_BACK_TO_MINOR_THRESHOLD";
    const BUDGET_VARIATION_POSITIVE_BACK_TO_NORMAL_THRESHOLD = "BUDGET_VARIATION_POSITIVE_BACK_TO_NORMAL_THRESHOLD";
    const BUDGET_VARIATION_POSITIVE_TO_MINOR_THRESHOLD = "BUDGET_VARIATION_POSITIVE_TO_MINOR_THRESHOLD";
    const BUDGET_VARIATION_POSITIVE_TO_MAJOR_THRESHOLD = "BUDGET_VARIATION_POSITIVE_TO_MAJOR_THRESHOLD";    
    
    const PEAK_NEGATIVE_BACK_TO_MINOR_THRESHOLD = "PEAK_NEGATIVE_BACK_TO_MINOR_THRESHOLD";
    const PEAK_NEGATIVE_BACK_TO_NORMAL_THRESHOLD = "PEAK_NEGATIVE_BACK_TO_NORMAL_THRESHOLD";
    const PEAK_NEGATIVE_TO_MINOR_THRESHOLD = "PEAK_NEGATIVE_TO_MINOR_THRESHOLD";
    const PEAK_NEGATIVE_TO_MAJOR_THRESHOLD = "PEAK_NEGATIVE_TO_MAJOR_THRESHOLD";
    const PEAK_POSITIVE_BACK_TO_MINOR_THRESHOLD = "PEAK_POSITIVE_BACK_TO_MINOR_THRESHOLD";
    const PEAK_POSITIVE_BACK_TO_NORMAL_THRESHOLD = "PEAK_POSITIVE_BACK_TO_NORMAL_THRESHOLD";
    const PEAK_POSITIVE_TO_MINOR_THRESHOLD = "PEAK_POSITIVE_TO_MINOR_THRESHOLD";
    const PEAK_POSITIVE_TO_MAJOR_THRESHOLD = "PEAK_POSITIVE_TO_MAJOR_THRESHOLD";
    
    const MONITORING_ACCEPTABLE_NOISE_MARGIN_DB_DEFAULT = "MONITORING_ACCEPTABLE_NOISE_MARGIN_DB_DEFAULT";
    const DEFAULT_MONITORING_ACCEPTABLE_NOISE_MARGIN_DB_DEFAULT = 6.0;
    
    const ATTENUATION_CRITICAL_THRESHOLD = "ATTENUATION_CRITICAL_THRESHOLD";
    const DEFAULT_ATTENUATION_CRITICAL_THRESHOLD = 6.0;
    
    const MONITORING_ATTENUATION_TO_AND_BACK_TO_DELTA_DB_DEFAULT = "MONITORING_ATTENUATION_TO_AND_BACK_TO_DELTA_DB_DEFAULT";
    const DEFAULT_MONITORING_ATTENUATION_TO_AND_BACK_TO_DELTA_DB_DEFAULT = 0.1;   
    
    //PEAKS monitoring
    const MISSING_PEAK_THRESHOLD_POINTS="MISSING_PEAK_THRESHOLD_POINTS";
    const DEFAULT_MISSING_PEAK_THRESHOLD_POINTS=5;
    const MONITORABLE_PEAK_THRESHOLD_DB="MONITORABLE_PEAK_THRESHOLD_DB";
    const DEFAULT_MONITORABLE_PEAK_THRESHOLD_DB=0.0; 
    const MONITORING_PEAK_TO_AND_BACK_TO_DELTA_DB_DEFAULT="MONITORING_PEAK_TO_AND_BACK_TO_DELTA_DB_DEFAULT";
    const DEFAULT_MONITORING_PEAK_TO_AND_BACK_TO_DELTA_DB_DEFAULT=0.2;
    /**
     * DEFAULT_MAX_NUMBER_MONITORABLE_PEAKS = 32 in ONMSi
     */
    const DEFAULT_MAX_NUMBER_MONITORABLE_PEAKS= 32;
    
    const DEFAULT_FM_NEGATIVE_BACK_TO_MINOR_THRESHOLD = -2.9;
    const DEFAULT_FM_NEGATIVE_BACK_TO_NORMAL_THRESHOLD = -0.9;
    const DEFAULT_FM_NEGATIVE_TO_MINOR_THRESHOLD = -1;
    const DEFAULT_FM_NEGATIVE_TO_MAJOR_THRESHOLD = -3;
    const DEFAULT_FM_POSITIVE_BACK_TO_MINOR_THRESHOLD = 2.9;
    const DEFAULT_FM_POSITIVE_BACK_TO_NORMAL_THRESHOLD = 0.9;
    const DEFAULT_FM_POSITIVE_TO_MINOR_THRESHOLD = 1;
    const DEFAULT_FM_POSITIVE_TO_MAJOR_THRESHOLD = 3;
    
    const DEFAULT_BUDGET_VARIATION_NEGATIVE_BACK_TO_MINOR_THRESHOLD = -2.9;
    const DEFAULT_BUDGET_VARIATION_NEGATIVE_BACK_TO_NORMAL_THRESHOLD = -0.9;
    const DEFAULT_BUDGET_VARIATION_NEGATIVE_TO_MINOR_THRESHOLD = -1;
    const DEFAULT_BUDGET_VARIATION_NEGATIVE_TO_MAJOR_THRESHOLD = -3;
    const DEFAULT_BUDGET_VARIATION_POSITIVE_BACK_TO_MINOR_THRESHOLD = 2.9;
    const DEFAULT_BUDGET_VARIATION_POSITIVE_BACK_TO_NORMAL_THRESHOLD = 0.9;
    const DEFAULT_BUDGET_VARIATION_POSITIVE_TO_MINOR_THRESHOLD = 1;
    const DEFAULT_BUDGET_VARIATION_POSITIVE_TO_MAJOR_THRESHOLD = 3;          
    
    const DEFAULT_PEAK_NEGATIVE_BACK_TO_MINOR_THRESHOLD = -3.8;
    const DEFAULT_PEAK_NEGATIVE_BACK_TO_NORMAL_THRESHOLD = -1.8;
    const DEFAULT_PEAK_NEGATIVE_TO_MINOR_THRESHOLD = -2;
    const DEFAULT_PEAK_NEGATIVE_TO_MAJOR_THRESHOLD = -4;
    const DEFAULT_PEAK_POSITIVE_BACK_TO_MINOR_THRESHOLD = 3.8;
    const DEFAULT_PEAK_POSITIVE_BACK_TO_NORMAL_THRESHOLD = 1.8;
    const DEFAULT_PEAK_POSITIVE_TO_MINOR_THRESHOLD = 2;
    const DEFAULT_PEAK_POSITIVE_TO_MAJOR_THRESHOLD = 4;
    
    /**
     * Register the otu First Marker attenuation thresholds
     *
     * @param SMTOtuMonitoringTestThresholdDto $firstMarkerThresholds The otu First Marker attenuation thresholds.
     */
    public static function setDefaultFirstMarkerAttenuationThresholds( SMTOtuMonitoringTestThresholdDto $firstMarkerThresholds )
    {
    	self::saveDefaultFirstMarkerAttenuationThresholdsInCache($firstMarkerThresholds);
    	
    	//save first marker attenuation thresholds in settings file
    	SMTSmartOtuSettings::writeSettings( array( 
    	self::FM_NEGATIVE_BACK_TO_MINOR_THRESHOLD => $firstMarkerThresholds->getNegativeBackToMinorThreshold(),
    	self::FM_NEGATIVE_BACK_TO_NORMAL_THRESHOLD => $firstMarkerThresholds->getNegativeBackToNormalThreshold(),
    	self::FM_NEGATIVE_TO_MINOR_THRESHOLD => $firstMarkerThresholds->getNegativeToMinorThreshold(),
    	self::FM_NEGATIVE_TO_MAJOR_THRESHOLD => $firstMarkerThresholds->getNegativeToMajorThreshold(),
    	self::FM_POSITIVE_BACK_TO_MINOR_THRESHOLD => $firstMarkerThresholds->getPositiveBackToMinorThreshold(),
    	self::FM_POSITIVE_BACK_TO_NORMAL_THRESHOLD => $firstMarkerThresholds->getPositiveBackToNormalThreshold(),
    	self::FM_POSITIVE_TO_MINOR_THRESHOLD => $firstMarkerThresholds->getPositiveToMinorThreshold(),
    	self::FM_POSITIVE_TO_MAJOR_THRESHOLD => $firstMarkerThresholds->getPositiveToMajorThreshold() ), self::SMARTOTU_THRESHOLDS_FILE );
    }   
    
    /**
     * Save the otu First Marker attenuation thresholds i ncache
     *
     * @param SMTOtuMonitoringTestThresholdDto $firstMarkerThresholds The otu First Marker attenuation thresholds.
     *
     */
    private static function saveDefaultFirstMarkerAttenuationThresholdsInCache( SMTOtuMonitoringTestThresholdDto $firstMarkerThresholds )
    {
    	if ( ( $settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory() ) == NULL )
    	{
    		$settings = new SMTSmartOtuSettingsCache();
    	}
    	$settings->setDefaultFirstMarkerAttenuationThresholds( $firstMarkerThresholds );
    	$settings->save();
    }    
    
    /**
     * Get the otu First Marker attenuation thresholds
     *
     * @return SMTOtuMonitoringTestThresholdDto The otu First Marker attenuation thresholds.
     *
     */
    public static function getDefaultFirstMarkerAttenuationThresholds()
    {
        $firstMarkerThresholds = NULL;
    	
    	$settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory();
    	if ( $settings != NULL )
    	{
    		$firstMarkerThresholds = $settings->getDefaultFirstMarkerAttenuationThresholds();
    	} 	    	
    	
    	//retrieve first marker thresholds from ini file if it was not found in cache
    	if ( $firstMarkerThresholds == NULL )
    	{
    	    $firstMarkerThresholds = new SMTOtuMonitoringTestThresholdDto();
    	    $conf_array = parse_ini_file( self::SMARTOTU_THRESHOLDS_FILE );
    	    
    	    if( array_key_exists( self::FM_NEGATIVE_BACK_TO_MINOR_THRESHOLD, $conf_array) )
    	    {
    	    	$firstMarkerThresholds->setNegativeBackToMinorThreshold( $conf_array[self::FM_NEGATIVE_BACK_TO_MINOR_THRESHOLD] );
    	    }
    	    else
    	    {
    	        $firstMarkerThresholds->setNegativeBackToMinorThreshold( self::DEFAULT_FM_NEGATIVE_BACK_TO_MINOR_THRESHOLD );
    	    }
    	    if( array_key_exists( self::FM_NEGATIVE_BACK_TO_NORMAL_THRESHOLD, $conf_array) )    	    
    	    {
    	    	$firstMarkerThresholds->setNegativeBackToNormalThreshold( $conf_array[self::FM_NEGATIVE_BACK_TO_NORMAL_THRESHOLD] );
    	    }
    	    else
    	    {
    	    	$firstMarkerThresholds->setNegativeBackToNormalThreshold( self::DEFAULT_FM_NEGATIVE_BACK_TO_NORMAL_THRESHOLD );
    	    }    	    
    	    if( array_key_exists( self::FM_NEGATIVE_TO_MINOR_THRESHOLD, $conf_array) )
    	    {
    	    	$firstMarkerThresholds->setNegativeToMinorThreshold( $conf_array[self::FM_NEGATIVE_TO_MINOR_THRESHOLD] );
    	    }
    	    else
    	    {
    	    	$firstMarkerThresholds->setNegativeToMinorThreshold( self::DEFAULT_FM_NEGATIVE_TO_MINOR_THRESHOLD );
    	    }    	    
    	    if( array_key_exists( self::FM_NEGATIVE_TO_MAJOR_THRESHOLD, $conf_array) )
    	    {
    	    	$firstMarkerThresholds->setNegativeToMajorThreshold( $conf_array[self::FM_NEGATIVE_TO_MAJOR_THRESHOLD] );
    	    }
    	    else
    	    {
    	    	$firstMarkerThresholds->setNegativeToMajorThreshold( self::DEFAULT_FM_NEGATIVE_TO_MAJOR_THRESHOLD );
    	    }    	    
    	    if( array_key_exists( self::FM_POSITIVE_BACK_TO_MINOR_THRESHOLD, $conf_array) )
    	    {
    	    	$firstMarkerThresholds->setPositiveBackToMinorThreshold( $conf_array[self::FM_POSITIVE_BACK_TO_MINOR_THRESHOLD] );
    	    }
    	    else
    	    {
    	    	$firstMarkerThresholds->setPositiveBackToMinorThreshold( self::DEFAULT_FM_POSITIVE_BACK_TO_MINOR_THRESHOLD );
    	    }
    	    if( array_key_exists( self::FM_POSITIVE_BACK_TO_NORMAL_THRESHOLD, $conf_array) )
    	    {
    	    	$firstMarkerThresholds->setPositiveBackToNormalThreshold( $conf_array[self::FM_POSITIVE_BACK_TO_NORMAL_THRESHOLD] );
    	    }
    	    else
    	    {
    	    	$firstMarkerThresholds->setPositiveBackToNormalThreshold( self::DEFAULT_FM_POSITIVE_BACK_TO_NORMAL_THRESHOLD );
    	    }    	    
    	    if( array_key_exists( self::FM_POSITIVE_TO_MINOR_THRESHOLD, $conf_array) )
    	    {
    	    	$firstMarkerThresholds->setPositiveToMinorThreshold( $conf_array[self::FM_POSITIVE_TO_MINOR_THRESHOLD] );
    	    }
    	    else
    	    {
    	    	$firstMarkerThresholds->setPositiveToMinorThreshold( self::DEFAULT_FM_POSITIVE_TO_MINOR_THRESHOLD );
    	    }    	    
    	    if( array_key_exists( self::FM_POSITIVE_TO_MAJOR_THRESHOLD, $conf_array) )
    	    {
    	    	$firstMarkerThresholds->setPositiveToMajorThreshold( $conf_array[self::FM_POSITIVE_TO_MAJOR_THRESHOLD] );
    	    }
    	    else
    	    {
    	    	$firstMarkerThresholds->setPositiveToMajorThreshold( self::DEFAULT_FM_POSITIVE_TO_MAJOR_THRESHOLD );
    	    }
    	    
    	    //minor threshold disabled by default
    	    $firstMarkerThresholds->setMinorThresholdEnabled(FALSE);
    	    
    	    //save retrieved data in cache:
    	    self::saveDefaultFirstMarkerAttenuationThresholdsInCache($firstMarkerThresholds);    	    
    	} 
    	return $firstMarkerThresholds;
    }
        
    
    /**
     * Register the otu budget (LastMarker - First Marker) attenuation thresholds
     *
     * @param SMTOtuMonitoringTestThresholdDto $budgetVariationMarkerThresholds The otu budget attenuation thresholds.
     *
     */
    public static function setDefaultBudgetVariationThresholds( SMTOtuMonitoringTestThresholdDto $budgetVariationThresholds )
    {

    	self::saveDefaultBudgetVariationThresholdsInCache( $budgetVariationThresholds );
    	//save budget variation thresholds in settings file
    	//save first marker attenuation thresholds in settings file
    	SMTSmartOtuSettings::writeSettings( array(
    			self::BUDGET_VARIATION_NEGATIVE_BACK_TO_MINOR_THRESHOLD => $budgetVariationThresholds->getNegativeBackToMinorThreshold(),
    			self::BUDGET_VARIATION_NEGATIVE_BACK_TO_NORMAL_THRESHOLD => $budgetVariationThresholds->getNegativeBackToNormalThreshold(),
    			self::BUDGET_VARIATION_NEGATIVE_TO_MINOR_THRESHOLD => $budgetVariationThresholds->getNegativeToMinorThreshold(),
    			self::BUDGET_VARIATION_NEGATIVE_TO_MAJOR_THRESHOLD => $budgetVariationThresholds->getNegativeToMajorThreshold(),
    			self::BUDGET_VARIATION_POSITIVE_BACK_TO_MINOR_THRESHOLD => $budgetVariationThresholds->getPositiveBackToMinorThreshold(),
    			self::BUDGET_VARIATION_POSITIVE_BACK_TO_NORMAL_THRESHOLD => $budgetVariationThresholds->getPositiveBackToNormalThreshold(),
    			self::BUDGET_VARIATION_POSITIVE_TO_MINOR_THRESHOLD => $budgetVariationThresholds->getPositiveToMinorThreshold(),
    			self::BUDGET_VARIATION_POSITIVE_TO_MAJOR_THRESHOLD => $budgetVariationThresholds->getPositiveToMajorThreshold() ), self::SMARTOTU_THRESHOLDS_FILE );    	
    }    
    
    /**
     * Save the otu budget (LastMarker - First Marker) attenuation thresholds in cache
     *
     * @param SMTOtuMonitoringTestThresholdDto $budgetVariationMarkerThresholds The otu budget attenuation thresholds.
     *
     */    
    private static function saveDefaultBudgetVariationThresholdsInCache( SMTOtuMonitoringTestThresholdDto $budgetVariationThresholds )
    {
        if ( ( $settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory() ) == NULL )
        {
        	$settings = new SMTSmartOtuSettingsCache();
        }
        $settings->setDefaultBudgetVariationThresholds( $budgetVariationThresholds );
        $settings->save();
    }
    
    /**
     * Get the otu budget (LastMarker - First Marker) attenuation thresholds
     *
     * @return SMTOtuMonitoringTestThresholdDto The otu budget attenuation thresholds.
     */
    public static function getDefaultBudgetVariationThresholds()
    {
        $budgetVariationThresholds = NULL;
    	$settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory();
    	if ( $settings != NULL )
    	{
    		$budgetVariationThresholds = $settings->getDefaultBudgetVariationThresholds();
    	}
    	
    	//retrieve budget thresholds from ini file if it was not found in cache
    	if ( $budgetVariationThresholds == NULL )
    	{
    	    $budgetVariationThresholds = new SMTOtuMonitoringTestThresholdDto();
	    	$conf_array = parse_ini_file(self::SMARTOTU_THRESHOLDS_FILE);
	    		
	    	if( array_key_exists( self::BUDGET_VARIATION_NEGATIVE_BACK_TO_MINOR_THRESHOLD, $conf_array) )
	    	{
	    		$budgetVariationThresholds->setNegativeBackToMinorThreshold( $conf_array[self::BUDGET_VARIATION_NEGATIVE_BACK_TO_MINOR_THRESHOLD] );
	    	}
	    	else
	    	{
	    		$budgetVariationThresholds->setNegativeBackToMinorThreshold( self::DEFAULT_BUDGET_VARIATION_NEGATIVE_BACK_TO_MINOR_THRESHOLD );
	    	}
	    	if( array_key_exists( self::BUDGET_VARIATION_NEGATIVE_BACK_TO_NORMAL_THRESHOLD, $conf_array) )
	    	{
	    		$budgetVariationThresholds->setNegativeBackToNormalThreshold( $conf_array[self::BUDGET_VARIATION_NEGATIVE_BACK_TO_NORMAL_THRESHOLD] );
	    	}
	    	else
	    	{
	    		$budgetVariationThresholds->setNegativeBackToNormalThreshold( self::DEFAULT_BUDGET_VARIATION_NEGATIVE_BACK_TO_NORMAL_THRESHOLD );
	    	}	    	
	    	if( array_key_exists( self::BUDGET_VARIATION_NEGATIVE_TO_MINOR_THRESHOLD, $conf_array) )
	    	{
	    		$budgetVariationThresholds->setNegativeToMinorThreshold( $conf_array[self::BUDGET_VARIATION_NEGATIVE_TO_MINOR_THRESHOLD] );
	    	}
	    	else
	    	{
	    		$budgetVariationThresholds->setNegativeToMinorThreshold( self::DEFAULT_BUDGET_VARIATION_NEGATIVE_TO_MINOR_THRESHOLD );
	    	}	    	
	    	if( array_key_exists( self::BUDGET_VARIATION_NEGATIVE_TO_MAJOR_THRESHOLD, $conf_array) )
	    	{
	    		$budgetVariationThresholds->setNegativeToMajorThreshold( $conf_array[self::BUDGET_VARIATION_NEGATIVE_TO_MAJOR_THRESHOLD] );
	    	}
	    	else
	    	{
	    		$budgetVariationThresholds->setNegativeToMajorThreshold( self::DEFAULT_BUDGET_VARIATION_NEGATIVE_TO_MAJOR_THRESHOLD );
	    	}	    	
	    	if( array_key_exists( self::BUDGET_VARIATION_POSITIVE_BACK_TO_MINOR_THRESHOLD, $conf_array) )
	    	{
	    		$budgetVariationThresholds->setPositiveBackToMinorThreshold( $conf_array[self::BUDGET_VARIATION_POSITIVE_BACK_TO_MINOR_THRESHOLD] );
	    	}
	    	else
	    	{
	    		$budgetVariationThresholds->setPositiveBackToMinorThreshold( self::DEFAULT_BUDGET_VARIATION_POSITIVE_BACK_TO_MINOR_THRESHOLD );
	    	}	    	
	    	if( array_key_exists( self::BUDGET_VARIATION_POSITIVE_BACK_TO_NORMAL_THRESHOLD, $conf_array) )
	    	{
	    		$budgetVariationThresholds->setPositiveBackToNormalThreshold( $conf_array[self::BUDGET_VARIATION_POSITIVE_BACK_TO_NORMAL_THRESHOLD] );
	    	}
	    	else
	    	{
	    		$budgetVariationThresholds->setPositiveBackToNormalThreshold( self::DEFAULT_BUDGET_VARIATION_POSITIVE_BACK_TO_NORMAL_THRESHOLD );
	    	}	    	
	    	if( array_key_exists( self::BUDGET_VARIATION_POSITIVE_TO_MINOR_THRESHOLD, $conf_array) )
	    	{
	    		$budgetVariationThresholds->setPositiveToMinorThreshold( $conf_array[self::BUDGET_VARIATION_POSITIVE_TO_MINOR_THRESHOLD] );
	    	}
	    	else
	    	{
	    		$budgetVariationThresholds->setPositiveToMinorThreshold( self::DEFAULT_BUDGET_VARIATION_POSITIVE_TO_MINOR_THRESHOLD );
	    	}	    	
	    	if( array_key_exists( self::BUDGET_VARIATION_POSITIVE_TO_MAJOR_THRESHOLD, $conf_array) )
	    	{
	    		$budgetVariationThresholds->setPositiveToMajorThreshold( $conf_array[self::BUDGET_VARIATION_POSITIVE_TO_MAJOR_THRESHOLD] );
	    	}
	    	else
	    	{
	    		$budgetVariationThresholds->setPositiveToMajorThreshold( self::DEFAULT_BUDGET_VARIATION_POSITIVE_TO_MAJOR_THRESHOLD );
	    	}

	    	//minor threshold disabled by default
	    	$budgetVariationThresholds->setMinorThresholdEnabled(FALSE);
	    	
	    	//save retrieved data in cache:
	    	self::saveDefaultBudgetVariationThresholdsInCache( $budgetVariationThresholds );
    	}    	
    	
    	return $budgetVariationThresholds;
    }            
    
    /**
     * Register the otu peak thresholds
     *
     * @param SMTOtuMonitoringTestThresholdDto $peakThresholds The otu peaks thresholds.
     */
    public static function setDefaultPeakThresholds( SMTOtuMonitoringTestThresholdDto $peakThresholds )
    {
    	self::saveDefaultPeakThresholdsInCache($peakThresholds);
    	
    	//save first marker attenuation thresholds in settings file
    	SMTSmartOtuSettings::writeSettings( array(
    			self::PEAK_NEGATIVE_BACK_TO_MINOR_THRESHOLD => $peakThresholds->getNegativeBackToMinorThreshold(),
    			self::PEAK_NEGATIVE_BACK_TO_NORMAL_THRESHOLD => $peakThresholds->getNegativeBackToNormalThreshold(),
    			self::PEAK_NEGATIVE_TO_MINOR_THRESHOLD => $peakThresholds->getNegativeToMinorThreshold(),
    			self::PEAK_NEGATIVE_TO_MAJOR_THRESHOLD => $peakThresholds->getNegativeToMajorThreshold(),
    			self::PEAK_POSITIVE_BACK_TO_MINOR_THRESHOLD => $peakThresholds->getPositiveBackToMinorThreshold(),
    			self::PEAK_POSITIVE_BACK_TO_NORMAL_THRESHOLD => $peakThresholds->getPositiveBackToNormalThreshold(),
    			self::PEAK_POSITIVE_TO_MINOR_THRESHOLD => $peakThresholds->getPositiveToMinorThreshold(),
    			self::PEAK_POSITIVE_TO_MAJOR_THRESHOLD => $peakThresholds->getPositiveToMajorThreshold() ), self::SMARTOTU_THRESHOLDS_FILE );
    }
    
    /**
     * Save the otu peak thresholds i ncache
     *
     * @param SMTOtuMonitoringTestThresholdDto $peakThresholds The otu peak thresholds.
     *
     */
    private static function saveDefaultPeakThresholdsInCache( SMTOtuMonitoringTestThresholdDto $peakThresholds )
    {
    	if ( ( $settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory() ) == NULL )
    	{
    		$settings = new SMTSmartOtuSettingsCache();
    	}
    	$settings->setDefaultPeakThresholdsDb( $peakThresholds );
    	$settings->save();
    }
    
    /**
     * Get the otu peak thresholds
     *
     * @return SMTOtuMonitoringTestThresholdDto The otu peak thresholds.
     *
     */
    public static function getDefaultPeakThresholds()
    {
    	$peakThresholds = NULL;
    	
    	$settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory();
    	if ( $settings != NULL )
    	{
    		$peakThresholds = $settings->getDefaultPeakThresholdsDb();
    	}
    	
    	//retrieve first marker thresholds from ini file if it was not found in cache
    	if ( $peakThresholds == NULL )
    	{
    		$peakThresholds = new SMTOtuMonitoringTestThresholdDto();
    		$conf_array = parse_ini_file( self::SMARTOTU_THRESHOLDS_FILE );
    		
    		if( array_key_exists( self::PEAK_NEGATIVE_BACK_TO_MINOR_THRESHOLD, $conf_array) )
    		{
    			$peakThresholds->setNegativeBackToMinorThreshold( $conf_array[self::PEAK_NEGATIVE_BACK_TO_MINOR_THRESHOLD] );
    		}
    		else
    		{
    			$peakThresholds->setNegativeBackToMinorThreshold( self::DEFAULT_PEAK_NEGATIVE_BACK_TO_MINOR_THRESHOLD );
    		}
    		if( array_key_exists( self::PEAK_NEGATIVE_BACK_TO_NORMAL_THRESHOLD, $conf_array) )
    		{
    			$peakThresholds->setNegativeBackToNormalThreshold( $conf_array[self::PEAK_NEGATIVE_BACK_TO_NORMAL_THRESHOLD] );
    		}
    		else
    		{
    			$peakThresholds->setNegativeBackToNormalThreshold( self::DEFAULT_PEAK_NEGATIVE_BACK_TO_NORMAL_THRESHOLD );
    		}
    		if( array_key_exists( self::PEAK_NEGATIVE_TO_MINOR_THRESHOLD, $conf_array) )
    		{
    			$peakThresholds->setNegativeToMinorThreshold( $conf_array[self::PEAK_NEGATIVE_TO_MINOR_THRESHOLD] );
    		}
    		else
    		{
    			$peakThresholds->setNegativeToMinorThreshold( self::DEFAULT_PEAK_NEGATIVE_TO_MINOR_THRESHOLD );
    		}
    		if( array_key_exists( self::PEAK_NEGATIVE_TO_MAJOR_THRESHOLD, $conf_array) )
    		{
    			$peakThresholds->setNegativeToMajorThreshold( $conf_array[self::PEAK_NEGATIVE_TO_MAJOR_THRESHOLD] );
    		}
    		else
    		{
    			$peakThresholds->setNegativeToMajorThreshold( self::DEFAULT_PEAK_NEGATIVE_TO_MAJOR_THRESHOLD );
    		}
    		if( array_key_exists( self::PEAK_POSITIVE_BACK_TO_MINOR_THRESHOLD, $conf_array) )
    		{
    			$peakThresholds->setPositiveBackToMinorThreshold( $conf_array[self::PEAK_POSITIVE_BACK_TO_MINOR_THRESHOLD] );
    		}
    		else
    		{
    			$peakThresholds->setPositiveBackToMinorThreshold( self::DEFAULT_PEAK_POSITIVE_BACK_TO_MINOR_THRESHOLD );
    		}
    		if( array_key_exists( self::PEAK_POSITIVE_BACK_TO_NORMAL_THRESHOLD, $conf_array) )
    		{
    			$peakThresholds->setPositiveBackToNormalThreshold( $conf_array[self::PEAK_POSITIVE_BACK_TO_NORMAL_THRESHOLD] );
    		}
    		else
    		{
    			$peakThresholds->setPositiveBackToNormalThreshold( self::DEFAULT_PEAK_POSITIVE_BACK_TO_NORMAL_THRESHOLD );
    		}
    		if( array_key_exists( self::PEAK_POSITIVE_TO_MINOR_THRESHOLD, $conf_array) )
    		{
    			$peakThresholds->setPositiveToMinorThreshold( $conf_array[self::PEAK_POSITIVE_TO_MINOR_THRESHOLD] );
    		}
    		else
    		{
    			$peakThresholds->setPositiveToMinorThreshold( self::DEFAULT_PEAK_POSITIVE_TO_MINOR_THRESHOLD );
    		}
    		if( array_key_exists( self::PEAK_POSITIVE_TO_MAJOR_THRESHOLD, $conf_array) )
    		{
    			$peakThresholds->setPositiveToMajorThreshold( $conf_array[self::PEAK_POSITIVE_TO_MAJOR_THRESHOLD] );
    		}
    		else
    		{
    			$peakThresholds->setPositiveToMajorThreshold( self::DEFAULT_PEAK_POSITIVE_TO_MAJOR_THRESHOLD );
    		}
    		
    		//save retrieved data in cache:
    		self::saveDefaultPeakThresholdsInCache($peakThresholds);
    	}
    	return $peakThresholds;
    }
    
    /**
     * Register Monitoring Acceptable Noise Margin (in Db)
     *
     * @param float $monitoringAcceptableNoiseMarginDbDefault The Monitoring Acceptable Noise Margin (in Db).
     */
    public static function setMonitoringAcceptableNoiseMarginDbDefault( $monitoringAcceptableNoiseMarginDbDefault  )
    {    
    	self::saveMonitoringAcceptableNoiseMarginDbDefaultInCache( $monitoringAcceptableNoiseMarginDbDefault );
    	//save Monitoring Acceptable Noise Margin in settings file
    	SMTSmartOtuSettings::writeSettings( array( self::MONITORING_ACCEPTABLE_NOISE_MARGIN_DB_DEFAULT=>$monitoringAcceptableNoiseMarginDbDefault ) );
    }
    
    /**
     * Save Monitoring Acceptable Noise Margin (in Db) in cache.
     * Used to display that min noise level in trace viewer and to force user to set markers in traceviewer outside noise level.
     *
     * @param float $monitoringAcceptableNoiseMarginDbDefault The Monitoring Acceptable Noise Margin in Db.
     */
    private static function saveMonitoringAcceptableNoiseMarginDbDefaultInCache( $monitoringAcceptableNoiseMarginDbDefault )
    {
    	if ( ( $settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory() ) == NULL )
    	{
    		$settings = new SMTSmartOtuSettingsCache();
    	}
    	$settings->setMonitoringAcceptableNoiseMarginDbDefault( $monitoringAcceptableNoiseMarginDbDefault );
    	$settings->save();
    }    
    
    /**
     * Get the Monitoring Acceptable Noise Margin (in Db) from cache or from ini file.
     * Used to display that min noise level in trace viewer and to force user to set markers in traceviewer outside noise level.
     *
     * @return float Monitoring Acceptable Noise Margin (in Db)
     */
    public static function getMonitoringAcceptableNoiseMarginDbDefault()
    {
    	$monitoringAcceptableNoiseMarginDbDefault = NULL;
    	$settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory();
    	if ( $settings != NULL )
    	{
    		$monitoringAcceptableNoiseMarginDbDefault = $settings->getMonitoringAcceptableNoiseMarginDbDefault();
    	}
    	 
    	//Retrieve Monitoring Acceptable Noise Margin (in Db) from ini file if it was not found in cache
    	if ( $monitoringAcceptableNoiseMarginDbDefault == NULL )
    	{
    	    $conf_array = parse_ini_file(self::SMARTOTU_THRESHOLDS_FILE);
    	     
    	    if( array_key_exists( self::MONITORING_ACCEPTABLE_NOISE_MARGIN_DB_DEFAULT, $conf_array) )
    	    {
    	    	$monitoringAcceptableNoiseMarginDbDefault = $conf_array[self::MONITORING_ACCEPTABLE_NOISE_MARGIN_DB_DEFAULT];
    	    }
    	    else
    	    {
    	    	$monitoringAcceptableNoiseMarginDbDefault = self::DEFAULT_MONITORING_ACCEPTABLE_NOISE_MARGIN_DB_DEFAULT;
    	    }
    	    
    	    //save retrieved data in cache:
    	    self::saveMonitoringAcceptableNoiseMarginDbDefaultInCache( $monitoringAcceptableNoiseMarginDbDefault );
    	}
    	
    	return floatval($monitoringAcceptableNoiseMarginDbDefault);
    }    
    
    /**
     * Register the attenuation critical threshold in Db
     *
     * @param float $attenuationCriticalThresholdDbDefault The attenuation critical threshold.
     */
    public static function setAttenuationCriticalThresholdDbDefault( $attenuationCriticalThresholdDbDefault  )
    {
    	self::saveAttenuationCriticalThresholdDbDefaultInCache( $attenuationCriticalThresholdDbDefault );
    	//save attenuation critical threshold in settings file
    	SMTSmartOtuSettings::writeSettings( array( self::ATTENUATION_CRITICAL_THRESHOLD=>$attenuationCriticalThresholdDbDefault ) );
    }
    
    /**
     * Save attenuation critical threshold (in Db) in cache.
     *
     * @param float $attenuationCriticalThresholdDbDefault The attenuation critical threshold in Db.
     *
     */
    private static function saveAttenuationCriticalThresholdDbDefaultInCache( $attenuationCriticalThresholdDbDefault )
    {
    	if ( ( $settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory() ) == NULL )
    	{
    		$settings = new SMTSmartOtuSettingsCache();
    	}
    	$settings->setAttenuationCriticalThresholdDbDefault( $attenuationCriticalThresholdDbDefault );
    	$settings->save();
    }
    
    /**
     * Get the attenuation critical threshold (in Db) from cache or from ini file.
     *
     * @return float attenuation critical threshold (in Db)
     */
    public static function getAttenuationCriticalThresholdDbDefault()
    {
    	$attenuationCriticalThresholdDbDefault = NULL;
    	$settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory();
    	if ( $settings != NULL )
    	{
    		$attenuationCriticalThresholdDbDefault = $settings->getAttenuationCriticalThresholdDbDefault();
    	}
    
    	//Retrieve attenuation critical threshold (in Db) from ini file if it was not found in cache
    	if ( $attenuationCriticalThresholdDbDefault == NULL )
    	{
    		$conf_array = parse_ini_file(self::SMARTOTU_THRESHOLDS_FILE);
    
    		if( array_key_exists( self::ATTENUATION_CRITICAL_THRESHOLD, $conf_array) )
    		{
    			$attenuationCriticalThresholdDbDefault = $conf_array[self::ATTENUATION_CRITICAL_THRESHOLD];
    		}
    		else
    		{
    			$attenuationCriticalThresholdDbDefault = self::DEFAULT_ATTENUATION_CRITICAL_THRESHOLD;
    		}
    			
    		//save retrieved data in cache:
    		self::saveAttenuationCriticalThresholdDbDefaultInCache( $attenuationCriticalThresholdDbDefault );
    	}
    	 
    	return floatval($attenuationCriticalThresholdDbDefault);
    }    
    
    /**
     * Register the delta between TO and BACK TO thresholds for attenuation (in Db) in cache and in settings
     *
     * @param SMTOtuMonitoringTestThresholdDto $peakToBtoDeltaDbDefault The delta between TO and BACK TO thresholds for attenuation (in Db)
     */
    public static function setAttenuationToBtoDeltaDbDefault( $attenuationToBtoDeltaDbDefault  )
    {
    	self::saveAttenuationToBtoDeltaDbDefaultInCache( $attenuationToBtoDeltaDbDefault );
    	//save budget variation thresholds in settings file
    	//save first marker attenuation thresholds in settings file
    	SMTSmartOtuSettings::writeSettings( array( self::MONITORING_ATTENUATION_TO_AND_BACK_TO_DELTA_DB_DEFAULT=>$attenuationToBtoDeltaDbDefault ) );
    }
    
    /**
     * Save the delta between TO and BACK TO thresholds for attenuation (in Db) in cache.
     *
     * @param int $attenuationToBtoDeltaDbDefault The delta between TO and BACK TO thresholds for attenuation (in Db)
     *
     */
    private static function saveAttenuationToBtoDeltaDbDefaultInCache( $attenuationToBtoDeltaDbDefault )
    {
    	if ( ( $settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory() ) == NULL )
    	{
    		$settings = new SMTSmartOtuSettingsCache();
    	}
    	$settings->setAttenuationToBtoDeltaDbDefault( $attenuationToBtoDeltaDbDefault );
    	$settings->save();
    }
    
    /**
     * Get the delta between TO and BACK TO thresholds for attenuation (in Db) from cache or from ini file.
     *
     * @return The delta between TO and BACK TO thresholds for attenuation (in Db)
     */
    public static function getAttenuationToBtoDeltaDbDefault()
    {
    	$attenuationToBtoDeltaDbDefault = NULL;
    	$settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory();
    	if ( $settings != NULL )
    	{
    		$attenuationToBtoDeltaDbDefault = $settings->getAttenuationToBtoDeltaDbDefault();
    	}
    	 
    	//Retrieve the delta between TO and BACK TO thresholds for attenuation (in Db) from ini file if it was not found in cache
    	if ( $attenuationToBtoDeltaDbDefault == NULL )
    	{
    	    $conf_array = parse_ini_file(self::SMARTOTU_THRESHOLDS_FILE);
    	     
    	    if( array_key_exists( self::MONITORING_ATTENUATION_TO_AND_BACK_TO_DELTA_DB_DEFAULT, $conf_array) )
    	    {
    	    	$attenuationToBtoDeltaDbDefault = $conf_array[self::MONITORING_ATTENUATION_TO_AND_BACK_TO_DELTA_DB_DEFAULT];
    	    }
    	    else
    	    {
    	    	$attenuationToBtoDeltaDbDefault = self::DEFAULT_MONITORING_ATTENUATION_TO_AND_BACK_TO_DELTA_DB_DEFAULT;
    	    }
    	    
    	    //save retrieved data in cache:
    	    self::saveAttenuationToBtoDeltaDbDefaultInCache( $attenuationToBtoDeltaDbDefault );
    	}
    	
    	return floatval($attenuationToBtoDeltaDbDefault);
    }
    
    
    /**
     * Register the delta between TO and BACK TO thresholds for peak (in Db) in cache and in settings
     *
     * @param float $peakToBtoDeltaDbDefault The delta between TO and BACK TO thresholds for peak (in Db)
     */
    public static function setPeakToBtoDeltaDbDefault( $peakToBtoDeltaDbDefault  )
    {
    	self::savePeakToBtoDeltaDbDefaultInCache( $peakToBtoDeltaDbDefault );
    	//save budget variation thresholds in settings file
    	//save first marker attenuation thresholds in settings file
    	SMTSmartOtuSettings::writeSettings( array( self::MONITORING_PEAK_TO_AND_BACK_TO_DELTA_DB_DEFAULT=>$peakToBtoDeltaDbDefault ) );
    }
    
    /**
     * Save the delta between TO and BACK TO thresholds for attenuation (in Db) in cache.
     *
     * @param float $peakToBtoDeltaDbDefault The delta between TO and BACK TO thresholds for attenuation (in Db)
     *
     */
    private static function savePeakToBtoDeltaDbDefaultInCache( $peakToBtoDeltaDbDefault )
    {
    	if ( ( $settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory() ) == NULL )
    	{
    		$settings = new SMTSmartOtuSettingsCache();
    	}
    	$settings->setPeakToBtoDeltaDefault( $peakToBtoDeltaDbDefault);
    	$settings->save();
    }
    
    /**
     * Get the delta between TO and BACK TO thresholds for peak (in Db) from cache or from ini file.
     *
     * @return The delta between TO and BACK TO thresholds for peak (in Db)
     */
    public static function getPeakToBtoDeltaDbDefault()
    {
    	$peakToBtoDeltaDbDefault = NULL;
    	$settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory();
    	if ( $settings != NULL )
    	{
    		$peakToBtoDeltaDbDefault = $settings->getPeakToBtoDeltaDefault();
    	}
    	
    	//Retrieve the delta between TO and BACK TO thresholds for attenuation (in Db) from ini file if it was not found in cache
    	if ( $peakToBtoDeltaDbDefault == NULL )
    	{
    		$conf_array = parse_ini_file(self::SMARTOTU_THRESHOLDS_FILE);
    		
    		if( array_key_exists( self::MONITORING_PEAK_TO_AND_BACK_TO_DELTA_DB_DEFAULT, $conf_array) )
    		{
    			$peakToBtoDeltaDbDefault = $conf_array[self::MONITORING_PEAK_TO_AND_BACK_TO_DELTA_DB_DEFAULT];
    		}
    		else
    		{
    			$peakToBtoDeltaDbDefault = self::DEFAULT_MONITORING_PEAK_TO_AND_BACK_TO_DELTA_DB_DEFAULT;
    		}
    		
    		//save retrieved data in cache:
    		self::savePeakToBtoDeltaDbDefaultInCache( $peakToBtoDeltaDbDefault );
    	}
    	
    	return floatval($peakToBtoDeltaDbDefault);
    }
    
    /**
     * Register the delta between TO and BACK TO thresholds for peak (in Db) in cache and in settings
     *
     * @param float $monitorablePeakThreshold 
     */
    public static function setMonitorablePeakThresholdDbDefault( $monitorablePeakThreshold  )
    {
    	self::saveMonitorablePeakThresholdDbDefaultInCache( $monitorablePeakThreshold );
    	//save budget variation thresholds in settings file
    	//save first marker attenuation thresholds in settings file
    	SMTSmartOtuSettings::writeSettings( array( self::MONITORABLE_PEAK_THRESHOLD_DB=>$monitorablePeakThreshold ) );
    }
    
    /**
     * Save the threshold to detect peaks in cache and in settings
     *
     * @param float $monitorablePeakThreshold The delta between TO and BACK TO thresholds for attenuation (in Db)
     *
     */
    private static function saveMonitorablePeakThresholdDbDefaultInCache( $monitorablePeakThreshold )
    {
    	if ( ( $settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory() ) == NULL )
    	{
    		$settings = new SMTSmartOtuSettingsCache();
    	}
    	$settings->setMonitorablePeakThresholdDb( $monitorablePeakThreshold );
    	$settings->save();
    }
    
    /**
     * Get the threshold to detect peaks from cache or from ini file.
     *
     * @return The delta between TO and BACK TO thresholds for attenuation (in Db)
     */
    public static function getMonitorablePeakThresholdDbDefault()
    {
    	$monitorablePeakThreshold = NULL;
    	$settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory();
    	if ( $settings != NULL )
    	{
    		$monitorablePeakThreshold = $settings->getMonitorablePeakThresholdDb();
    	}
    	
    	//Retrieve the delta between TO and BACK TO thresholds for attenuation (in Db) from ini file if it was not found in cache
    	if ( $monitorablePeakThreshold == NULL )
    	{
    		$conf_array = parse_ini_file(self::SMARTOTU_THRESHOLDS_FILE);
    		
    		if( array_key_exists( self::MONITORABLE_PEAK_THRESHOLD_DB, $conf_array) )
    		{
    			$monitorablePeakThreshold = $conf_array[self::MONITORABLE_PEAK_THRESHOLD_DB];
    		}
    		else
    		{
    			$monitorablePeakThreshold = self::DEFAULT_MONITORABLE_PEAK_THRESHOLD_DB;
    		}
    		
    		//save retrieved data in cache:
    		self::saveMonitorablePeakThresholdDbDefaultInCache( $monitorablePeakThreshold );
    	}
    	
    	return floatval($monitorablePeakThreshold);
    }
    
    /**
     * Register the d peaks shift limit in cache and in settings
     *
     * @param int $missingPeakThresholdPt The  peaks shift limit
     */
    public static function setMissingPeakThresholdPtDefault( $missingPeakThresholdPt  )
    {
    	self::saveMissingPeakThresholdPtDefaultInCache( $missingPeakThresholdPt );
    	//save budget variation thresholds in settings file
    	//save first marker attenuation thresholds in settings file
    	SMTSmartOtuSettings::writeSettings( array( self::MISSING_PEAK_THRESHOLD_POINTS=>$missingPeakThresholdPt ) );
    }
    
    /**
     * Save the  peaks shift limit in cache.
     *
     * @param int $missingPeakThresholdPt The  peaks shift limit
     *
     */
    private static function saveMissingPeakThresholdPtDefaultInCache( $missingPeakThresholdPt )
    {
    	if ( ( $settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory() ) == NULL )
    	{
    		$settings = new SMTSmartOtuSettingsCache();
    	}
    	$settings->setMissingPeakThresholdPtDefault($missingPeakThresholdPt );
    	$settings->save();
    }
    
    /**
     * Get the  peaks shift limit from cache or from ini file.
     *
     * @return The peaks shift limit (in Db)
     */
    public static function getMissingPeakThresholdPtDefault()
    {
    	$missingPeakThresholdPt = NULL;
    	$settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory();
    	if ( $settings != NULL )
    	{
    		$missingPeakThresholdPt = $settings->getMissingPeakThresholdPtDefault();
    	}
    	
    	//Retrieve the delta between TO and BACK TO thresholds for attenuation (in Db) from ini file if it was not found in cache
    	if ( $missingPeakThresholdPt == NULL )
    	{
    		$conf_array = parse_ini_file(self::SMARTOTU_THRESHOLDS_FILE);
    		
    		if( array_key_exists( self::MISSING_PEAK_THRESHOLD_POINTS, $conf_array) )
    		{
    			$missingPeakThresholdPt = $conf_array[self::MISSING_PEAK_THRESHOLD_POINTS];
    		}
    		else
    		{
    			$missingPeakThresholdPt = self::DEFAULT_MISSING_PEAK_THRESHOLD_POINTS;
    		}
    		
    		//save retrieved data in cache:
    		self::saveMissingPeakThresholdPtDefaultInCache( $missingPeakThresholdPt );
    	}
    	
    	return floatval($missingPeakThresholdPt);
    }
} 

?>