<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\settings;

use app\util\SMTLogger;

use app\util\SMTDto;
use app\settings\SMTSmartOTUStatusValues;
use app\sharedmemory\SMTIMemorySupport;
use app\sharedmemory\SMTMemoryManager;

/**
 * SmartOTU status to be stored in APC memory:
 * 
 * @author Sylvain Desplat
 */
class SMTOtuStatusCacheDto extends SMTDto implements SMTIMemorySupport
{   
    /**
     * Id of the status cache DTO = 0: Only one instance stored in memory support 
     */
    const ID = "0";
    
    /**
     * SmartOTU status among the SMTSmartOTUStatusValues possible values
     */
    protected $otuStatus = SMTSmartOTUStatusValues::OK;
    
    /**
     * Whether switch detection has been performed once
     * @var string
     */
    protected $switchDetected = FALSE;
    
    /**
     * Whether switch configuration = detection. NULL if not set
     * @var boolean
     */
    protected $switchConfigStatus = NULL;
    
    /**
     * Whether module configuration = detection. NULL if not set
     * @var boolean
     */
    protected $moduleConfigStatus = NULL;
    
    /**
     * Whether database and patch version has been checked.
     * @var boolean
     */
    protected $databaseVersionChecked = FALSE;
    
    /**
     * Whether the database cleared status has bean checked
     * Used in case of mode change between SmartOtu and ONMSi: ensure that when switching to SmartOtu mode, the database is empty.
     * Database is cleared when OTU contains no link and no test.
     *
     * @var boolean
     */
    protected $databaseClearedStatusChecked = FALSE;
    
    /**
     * Whether activity must be fully retrieved and patch version has been checked.
     * @var boolean
     */
    protected $forceFullActivityRetrievalStatus = FALSE;
    
    /**
     * Single instance of otu status cache DTO
     * 
     * @var SMTOtuStatusCacheDto
     */
    private static $otuStatusCache = NULL;
        
    /**
     * Returns the otu status instance: retrieve it from APC memory if needed
     * 
     * @param $createIfNotExistInAPC boolean Whether to force the creation of the instance if it doesn't exist in APC memory
     * @oaram $forceFetch boolean Force fetch of otu status cache from APC memory
     * 
     * @return \app\settings\SMTOtuStatusCacheDto
     */
    private static function getOtuStatusDto( $createIfNotExistInAPC = FALSE, $forceFetch = FALSE )
    {
        if ( self::$otuStatusCache == NULL || $forceFetch )
        {
        	if ( ( ( $status = SMTMemoryManager::fetch( SMTOtuStatusCacheDto::getClass(), self::ID ) ) == NULL ) && $createIfNotExistInAPC )
        	{
        		$status = new SMTOtuStatusCacheDto();
        	}
        	self::$otuStatusCache = $status;
        }
        else
        {
        	$status = self::$otuStatusCache;
        }
        return $status;
    }
    
    /**
     * Set smartOTU status among the SMTSmartOTUStatusValues possible values
     *
     * @param string $smartOTUStatus
     */
    public static function setSmartOTUStatus ( $smartOTUStatus )
    {
    	$status = self::getOtuStatusDto(TRUE);
    	$status->setOtuStatus( $smartOTUStatus );
    	$status->save();
    }
    
    /**
     * Returns the current smartOTUStatus or NULL if none is set. 
     * Can optionaly force the reload of OTU status from APC memory. 
     * 
     * @param $forceFetch boolean FALSE by default; if TRUE, force the retrieval of the OTU status from memory.
     *
     * @return number or NULL if not set
     */
    public static function getSmartOTUStatus( $forceFetch = FALSE )
    {
        $smartOTUStatus = NULL;
        $status = self::getOtuStatusDto( FALSE, $forceFetch );
        if ( $status != NULL )
        {
        	$smartOTUStatus = $status->getOtuStatus();
        }
        return $smartOTUStatus;        
    }        
    
    /**
     * Reload SmartOTU status from APC memory.
     */
    public static function reloadSmartOTUStatusFromMemory()
    {
        self::getSmartOTUStatus( TRUE);
    }
    
    /**
     * Set Module config or NULL if none is set
     *
     * @param boolean $moduleConfigOK
     */
    public static function setModuleConfigOK( $moduleConfigOK )
    {
        $status = self::getOtuStatusDto( TRUE );
        $status->setModuleConfigStatus( $moduleConfigOK );
        $status->save();
    }
    
    /**
     * Returns the module config or NULL if none is set
     *
     * @return boolean or NULL
     */
    public static function getModuleConfigOK()
    {
        $moduleConfigOK = NULL;
        $status = self::getOtuStatusDto();
        if ( $status != NULL )
        {
        	$moduleConfigOK = $status->getModuleConfigStatus();
        }
                
    	return $moduleConfigOK;
    }
    
    /**
     * Set whether database version has been checked
     *
     * @param boolean $databaseVersionChecked
     */
    public static function setDatabaseVersionChecked( $databaseVersionChecked )
    {
        $status = self::getOtuStatusDto( TRUE );
    	$status->setDatabaseVersionStatus( $databaseVersionChecked );
    	$status->save();
    }
    
    /**
     * Returns Whether the database version has been checked 
     * Used to ensure that database model version is only checked one time before first database access after an OTU reboot.
     *
     * @return boolean
     */
    public static function getDatabaseVersionChecked()
    {
    	$databaseVersionChecked = FALSE;
    	$status = self::getOtuStatusDto();
    	if ( $status != NULL )
    	{
    		$databaseVersionChecked = $status->getDatabaseVersionStatus();
//     		SMTLogger::getInstance()->trace( sprintf("Database version checked: %s.", $databaseVersionChecked ),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    	}
    
    	return $databaseVersionChecked;
    }
    
    /**
     * Set whether the database cleared status has bean checked in APC memory
     * Used in case of mode change between SmartOtu and ONMSi: ensure that when switching to SmartOtu mode, the database is empty.
     * Database is cleared when OTU contains no link and no test.
     *
     * @param boolean $databaseClearedStatusChecked
     */
    public static function setDatabaseClearedStatusChecked( $databaseClearedStatusChecked )
    {
        $status = self::getOtuStatusDto( TRUE );
        $status->setDatabaseClearedStatus( $databaseClearedStatusChecked );
        $status->save();
    }
        
    /**
     * Returns Whether the database cleared status has bean checked from APC memory
     * Used in case of mode change between SmartOtu and ONMSi: ensure that when switching to SmartOtu mode, the database is empty.
     * Database is cleared when OTU contains no link and no test. 
     *
     * @return boolean
     */
    public static function getDatabaseClearedStatusChecked()
    {
    	$databaseClearedStatusChecked = FALSE;
    	$status = self::getOtuStatusDto();
    	if ( $status != NULL )
    	{
    		$databaseClearedStatusChecked = $status->getDatabaseClearedStatus();
    		SMTLogger::getInstance()->trace( sprintf("Database clear status checked: %s.", $databaseClearedStatusChecked ),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    	}
    
    	return $databaseClearedStatusChecked;
    }
    
    /**
     * Set whether the full activity must be retrieved
     *
     * @param boolean $forceFullActivityRetrieval
     */
    public static function setForceFullActivityRetrieval( $forceFullActivityRetrieval )
    {
    	$status = self::getOtuStatusDto( TRUE );
    	$status->setForceFullActivityRetrievalStatus( $forceFullActivityRetrieval );
    	$status->save();
    }
    
    /**
     * Returns Whether the full activity must be retrieved
     *
     * @return boolean Whether the full activity must be retrieved
     */
    public static function isForceFullActivityRetrieval()
    {
    	$forceFullActivityRetrieval = FALSE;
    	$status = self::getOtuStatusDto();
    	if ( $status != NULL )
    	{
    		$forceFullActivityRetrieval = $status->isForceFullActivityRetrievalStatus();
    		//SMTLogger::getInstance()->trace( sprintf("forceFullActivityRetrieval: %s.", $forceFullActivityRetrieval ),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    	}
    
    	return $forceFullActivityRetrieval;
    }
    
    /**
     * Set the switch config or NULL if none is set
     *
     * @param boolean $switchConfigOK
     */
    public static function setSwitchConfigOK( $switchConfigOK )
    {
        $status = self::getOtuStatusDto( TRUE );
        $status->setSwitchConfigStatus( $switchConfigOK );
        $status->save();
    }
    
    /**
     * Returns the switch config or NULL if none is set
     *
     * @return boolean or NULL
     */
    public static function getSwitchConfigOK()
    {    
    	$switchConfigOK = NULL;
    	$status = self::getOtuStatusDto();
    	if ( $status != NULL )
    	{
    		$switchConfigOK = $status->getSwitchConfigStatus();
    	}
    	
    	return $switchConfigOK;    	
    }    
    
    /**
     * Set whether the switch configuration = detection or NULL if not set
     *
     * @param boolean $switchConfigOK or NULL
     */
    function setSwitchConfigStatus( $switchConfigOK )
    {
    	$this->switchConfigStatus = $switchConfigOK;
    }
    
    /**
     * Returns the current switch config or NULL if none is set
     *
     * @return boolean or NULL
     */
    function getSwitchConfigStatus()
    {
    	return $this->switchConfigStatus;
    }

    /**
     * Set whether switch has been detected once
     *
     * @param boolean $switchDetectedOnce
     */
    public static function setSwitchDetectedOnce( $switchDetectedOnce)
    {
    	$status = self::getOtuStatusDto( TRUE );
    	$status->setSwitchDetected( $switchDetectedOnce);
    	$status->save();
    }
    
    /**
     * Whether switch has been detected once
     *
     * @return boolean
     */
    public static function getSwitchDetectedOnce()
    {
    	$switchDetectedOnce = FALSE;
    	$status = self::getOtuStatusDto();
    	if ( $status != NULL )
    	{
    		$switchDetectedOnce= $status->getSwitchDetected();
    	}
    	
    	return $switchDetectedOnce;
    }
    
    /**
     * Set whether the switch configuration = detection or NULL if not set
     *
     * @param boolean $switchConfigOK or NULL
     */
    function setSwitchDetected( $switchDetectedOnce)
    {
    	$this->switchDetected= $switchDetectedOnce;
    }
    
    /**
     * Returns the current switch config or NULL if none is set
     *
     * @return boolean or NULL
     */
    function getSwitchDetected()
    {
    	return $this->switchDetected;
    }
    
    /**
     * Set whether the database version has been checked
     *
     * @param boolean $databaseVersionChecked
     */
    function setDatabaseVersionStatus( $databaseVersionChecked )
    {
    	$this->databaseVersionChecked = $databaseVersionChecked;
    }
       
    /**
     * Returns whether the database version has been checked
     *
     * @return boolean or NULL
     */
    function getDatabaseVersionStatus()
    {
    	return $this->databaseVersionChecked;
    }
    
    /**
     * Set whether the database cleared status has bean checked
     * Used in case of mode change between SmartOtu and ONMSi: ensure that when switching to SmartOtu mode, the database is empty.
     * Database is cleared when OTU contains no link and no test.
     *
     * @param boolean $databaseClearedStatusChecked
     */
    public function setDatabaseClearedStatus( $databaseClearedStatusChecked )
    {
    	$this->databaseClearedStatusChecked = $databaseClearedStatusChecked;
    }
    
    /**
     * Whether the database cleared status has bean checked
     * Used in case of mode change between SmartOtu and ONMSi: ensure that when switching to SmartOtu mode, the database is empty.
     * Database is cleared when OTU contains no link and no test.
     *
     * @param boolean $databaseClearedStatusChecked
     */
    public function getDatabaseClearedStatus()
    {
    	return $this->databaseClearedStatusChecked;
    }    
    
    /**
     * Set whether the full activity must be retrieved
     *
     * @param boolean $forceFullActivityRetrieval
     */
    function setForceFullActivityRetrievalStatus( $forceFullActivityRetrieval )
    {
    	$this->forceFullActivityRetrievalStatus = $forceFullActivityRetrieval;
    }
    
    /**
     * Returns whether the full activity must be retrieved
     *
     * @return boolean or NULL
     */
    function isForceFullActivityRetrievalStatus()
    {
    	return $this->forceFullActivityRetrievalStatus;
    }
    
    /**
     * Set whether the module configuration = detection or NULL if not set
     *
     * @param boolean $moduleConfigOK or NULL
     */
    function setModuleConfigStatus( $moduleConfigOK )
    {
    	$this->moduleConfigStatus = $moduleConfigOK;
    }
    
    /**
     * Returns the current module config or NULL if none is set
     *
     * @return boolean or NULL
     */
    function getModuleConfigStatus()
    {
    	return $this->moduleConfigStatus;
    }
    
    /**
     * Set smartOTU status among the SMTSmartOTUStatusValues possible values
     *
     * @param string $smartOTUStatus
     */
    function setOtuStatus ( $smartOTUStatus )
    {
    	$this->otuStatus = $smartOTUStatus;
    }
    
    /**
     * Returns the current smartOTUStatus or NULL if none is set
     *
     * @return number or NULL
     */
    function getOtuStatus()
    {
    	return $this->otuStatus;
    }
        
    /**
     * @return string
     */
    public function getId()
    {
    	return self::ID;
    }
    
    /**
     * Save Dto in shared memory
     *
     * @throws SMTAPCMemoryException
     */
    public function save()
    {
    	SMTMemoryManager::saveDto( $this );
    }
    
    /**
     * Delete Dto from shared memory
     *
     * @throws SMTAPCMemoryException
     */
    public function delete()
    {
    	SMTMemoryManager::deleteDto( $this );
    }
    
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }
    
    /**
     * Returns the dto class name.
     *
     * @return string the dto class name.
     */
    public static function getClass()
    {
    	return __CLASS__;
    }    
    
    /**
     * Otu status cache is never obsolete: it must not be purged 
     *
     * @return FALSE
     */
    public function isObsolete()
    {
    	return FALSE;
    }    
    
    /**
     * No obsolescence for those data: returns current time.
     *
     * @see isObsolete()
     * @return number Timestamp of the last APC memory access (used to handle obsolescence of dtos in cache)
     */
    public function getLastAccess()
    {
    	return microtime(TRUE);
    }
    
    /**
     * No obsolescence for those data: do nothing
     *
     * @see isObsolete()
     *
     */
    public function touchAccess()
    {
    }    
} 



?>