<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\ethernet;

use app\util\SMTLogger;

use app\parser\SMTOtuSocket;

use app\parser\SMTOtuApi;
use app\serviceshelper\SMTServiceHelper;
use app\services\setup\SMTIpv4Dto;
use app\services\setup\SMTIpv6Dto;

/**
 * 
 * @author Sylvain Desplat
 */
class SMTEthernet extends SMTServiceHelper
{
    const HOST_NAME_MAX_SIZE = 28;
    
    const LOCAL_MODE_ADDRESS = '192.168.1.1';    
    
    /**
     * Check host name: #charaters > 0  and #charaters > 80. Throw exception if the test fails.
     *
     * @param string $name host name
     *
     * @throws SMTEthernetException
     */
    static function checkHostName( $name )
    {
        $result = preg_match_all("/^(([a-zA-Z0-9]|[a-zA-Z0-9][a-zA-Z0-9\-]*[a-zA-Z0-9])\.)*([A-Za-z0-9]|[A-Za-z0-9][A-Za-z0-9\-]*[A-Za-z0-9])$/", $name, $matches );
                
    	if ( strlen( $name ) <= 0 || strlen( $name ) > self::HOST_NAME_MAX_SIZE || $result === FALSE || $result == 0 )
    	{
    		throw new SMTEthernetException( SMTEthernetException::ERROR_INVALID_HOST_NAME );
    	}
    }          
    
    /**
     * Test whether OTU is in local mode.
     * 
     * @param SMTOtuSocket $socket
     * 
     * @return boolean
     */
    static function isLocalMode( SMTOtuSocket $socket)
    {
        $localMode = FALSE;
        try
        {
        	$localMode = ( $socket->sendReceive( SMTOtuApi::CMD_system_local ) == SMTOtuApi::RES_YES);
        }
        catch ( \Exception $e )
        {
        	SMTLogger::getInstance()->traceException( $e );
        }
        return $localMode;        
    }
    
    /**
     * Test whether OTU is in local mode for the current logged-in user.
     *
     * @param SMTOtuSocket $socket
     *
     * @return whether the OTU is in local mode.
     */
    static function isLocalModeForCurrentSession( SMTOtuSocket $socket )
    {
    	$localMode = FALSE;
    	try 
    	{
    		$localMode = self::isLocalMode( $socket );
        	$localMode = $localMode && ( $_SERVER['SERVER_NAME'] == SMTEthernet::LOCAL_MODE_ADDRESS );
    	}
    	catch ( \Exception $e )
    	{
    		SMTLogger::getInstance()->traceException( $e );
    	}
    	return $localMode;
    }
    
    /**
     * Check if ip given is a valid ipv4 address
     *
     * @param SMTIpv4Dto $ipv4
     *
     * @throws SMTEthernetException
     */
    static function checkIpv4( $ipv4 )
    {
        $ip = $ipv4->getIp();
        $gateway = $ipv4->getGateway();
        $subnet = $ipv4->getSubnet();
        
    	if ( strlen( $ip ) <= 0 || !filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 ) )
    	{
    		throw new SMTEthernetException( SMTEthernetException::ERROR_INVALID_IPV4 );
    	}
    	if ( strlen( $gateway ) <= 0 || !filter_var( $gateway, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 ) )
    	{
    		throw new SMTEthernetException( SMTEthernetException::ERROR_INVALID_IPV4GATEWAY );
    	}
    	if ( strlen( $subnet ) <= 0 || !filter_var( $subnet, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 ) )
    	{
    		throw new SMTEthernetException( SMTEthernetException::ERROR_INVALID_IPV4SUBNET );
    	}        
    }    
    
    /**
     * Check if ip given is a valid ipv6 address
     *
     * @param SMTIpv6Dto $ipv6
     *
     * @throws SMTEthernetException
     */
    static function checkIpv6( $ipv6 )
    {
    	$ip = $ipv6->getIp();
    	$gateway = $ipv6->getGateway();
    	
    	if ( strlen( $ip ) <= 0 || !filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6) )
    	{
    		throw new SMTEthernetException( SMTEthernetException::ERROR_INVALID_IPV6 );
    	}
    	if ( strlen( $gateway ) <= 0 || !filter_var( $gateway, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6 ) )
    	{
    		throw new SMTEthernetException( SMTEthernetException::ERROR_INVALID_IPV6GATEWAY );
    	}    	
    }
    
    /**
     *
     * Get Ipv4 config
     * 
     * @return SMTIpv4Dto
     */
    function getIpv4Config()
    {
    	$ipv4 = new SMTIpv4Dto();
    
		$ipv4->setMode( $this->sendReceive( SMTOtuApi::CMD_get_otu_mode ) );
        try
        {
            $ipv4->setIp( $this->sendReceive( sprintf(SMTOtuApi::CMD_get_ip, 'ETH0') ) );
            $ipv4->setSubnet( $this->sendReceive( sprintf(SMTOtuApi::CMD_get_subnet, 'ETH0') ) );
            $ipv4->setGateway( $this->sendReceive( SMTOtuApi::CMD_get_gateway ) );
            $ipv4->setDns( $this->sendReceive( SMTOtuApi::CMD_get_dns ) );
            $ipv4->setDomain( $this->sendReceive( SMTOtuApi::CMD_get_domain ) );
        }
        //don't throw exception if one value cannot be retrieved because it is a normal behaviour 
        //when network interface is not connected
    	catch ( \Exception $e )
    	{
    		SMTLogger::getInstance()->traceException( $e );
    	}
        
    	return $ipv4;
    }    
    
    /**
     *
     * Get Ipv6 config
     *
     * @return SMTIpv4Dto
     */
    function getIpv6Config()
    {
    	$ipv6 = new SMTIpv6Dto();
    
		$ipv6->setMode( $this->sendReceive( SMTOtuApi::CMD_get_otu_mode_ipv6 ) );
        try
        {
            $ipv6->setIp( $this->sendReceive( sprintf(SMTOtuApi::CMD_get_ipv6, 'ETH0') ) );
            $ipv6->setGateway( $this->sendReceive( SMTOtuApi::CMD_get_gatewayv6 ) );
            $ipv6->setDns( $this->sendReceive( SMTOtuApi::CMD_get_dnsv6 ) );
            $ipv6->setSite( $this->sendReceive( SMTOtuApi::CMD_get_sitev6 ) );
            $ipv6->setLinkIp( $this->sendReceive( SMTOtuApi::CMD_get_linkipv6 ) );
        }
        //don't throw exception if one value cannot be retrieved because it is a normal behaviour 
        //when network interface is not connected
    	catch ( \Exception $e )
    	{
    		SMTLogger::getInstance()->traceException( $e );
    	}
    
    	return $ipv6;
    }    
    
    /**
     * Set Ipv4 config in OTU
     *
     * @param SMTIpv4Dto $ipv4
     * 
     * @throws SMTEthernetException
     */
    function setIpv4Config( $ipv4 )
    {
        //DHCP not active: check IpV4 settings
        if ( !$ipv4->isDHCP() )
        {
            self::checkIpv4( $ipv4 );
        }
        
    	$this->send( sprintf( SMTOtuApi::CMD_set_otu_mode, $ipv4->getMode() ) );
	    
	    //DHCP not active: update ip settings
	    if ( !$ipv4->isDHCP() )
	    {
	        $ip = explode('.', $ipv4->getIp() );
            $gateway = explode('.', $ipv4->getGateway() );
	        $subnet = explode('.', $ipv4->getSubnet() );
	        if ( !is_array( $dns = explode('.', $ipv4->getDns() ) ) ) { $dns = array(0,0,0,0); }
	        
	        $this->send( vsprintf( SMTOtuApi::CMD_set_ip, $ip ) );
	        $this->send( vsprintf( SMTOtuApi::CMD_set_subnet, $subnet ) );
	        $this->send( vsprintf( SMTOtuApi::CMD_set_gateway, $gateway  ) );
	        $this->send( vsprintf( SMTOtuApi::CMD_set_dns, $dns  ) );
	        $this->send( sprintf( SMTOtuApi::CMD_set_domain, $ipv4->getDomain() ) );
	    }   
    }        
    
    /**
     * Set Ipv6 config in OTU
     *
     * @param SMTIpv6Dto $ipv6
     * 
     * @throws SMTEthernetException
     */
    function setIpv6Config( $ipv6 )
    {   
        //IPV6 enabled and DHCP not active: check IPV6 settings before setting IPV6 mode
        if ( $ipv6->isIpv6Active() && !$ipv6->isDHCP() )
        {
        	self::checkIpv6( $ipv6 );
        }
        
		$this->send( sprintf( SMTOtuApi::CMD_set_otu_mode_ipv6, $ipv6->getMode() ) );

		//IPV6 enabled and DHCP not active: update ip settings
		if ( $ipv6->isIpv6Active() && !$ipv6->isDHCP() )
		{
            $dns = $ipv6->getDns();
            if ( strlen( $dns ) <= 0 ) { $dns = "::"; }
			$this->send( sprintf( SMTOtuApi::CMD_set_ipv6, $ipv6->getIp() ) );
			$this->send( sprintf( SMTOtuApi::CMD_set_gatewayv6, $ipv6->getGateway()  ) );
			$this->send( sprintf( SMTOtuApi::CMD_set_dnsv6, $dns ) );
			$this->send( sprintf( SMTOtuApi::CMD_set_sitev6, $ipv6->getSite() ) );
		}
    }    
}
?>