<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\monitoring;

use app\services\setup\SMTOtuCurrentPortDto;

use app\sharedmemory\SMTMemoryManager;

use app\util\SMTPaginationDto;

use app\settings\SMTSmartOtuSettings;

use app\events\operations\SMTOperationEventDto;
use app\events\SMTEventFactory;
use app\events\SMTEventMessageManager;
use app\parser\SMTOtuApi;
use app\services\alarm\SMTOpticalAlarmListDto;
use app\services\otdr\SMTStartOtdrMeasureOnDemandParametersDto;
use app\services\security\SMTAccessControl;
use app\services\SMTIService;
use app\services\SMTService;
use app\serviceshelper\activity\SMTActivity;
use app\serviceshelper\alarm\SMTAlarm;
use app\serviceshelper\monitoring\cache\SMTOperationManager;
use app\serviceshelper\monitoring\SMTGpsRoute;
use app\serviceshelper\monitoring\SMTLinkTest;
use app\serviceshelper\monitoring\SMTLinkTestException;
use app\serviceshelper\monitoring\SMTMeasureOnDemand;
use app\serviceshelper\monitoring\SMTTestOnDemand;
use app\util\SMTIOException;
use app\util\SMTLogger;
use app\util\SMTUtil;
use app\view\SMTFileDownloadPage;
use app\view\SMTTestRestApiPage;
/**
 * Service to setup and list link-tests.
 * For smartOTU: link = test but linkId != testId
 *
 * @author Sylvain Desplat
 */
class SMTLinkTestService extends SMTService implements SMTIService
{    
    /**
     * Full json request data are by default stored in an array in parameter Default::$fullRequestDataName='request_Data'
     * Add a link if it doesn't exist and its monitoring test
     *
     * @url POST 
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function addLinkTest( $request_data )
    {
        $linkId = -1;
        $linkHelper = new SMTLinkTest();
        $linkHelper->setContext( $this->getContext() );
        
        try
        {
            //check if the OTU hardware config is OK before creating a Link-test
            if ( $this->getContext()->getModuleConfigOK() && $this->getContext()->getSwitchConfigOK() )
            {
                $linkTestDto = SMTLinkTestDto::getInstance( $request_data );    
                $linkId = $linkTestDto->getId() ;
                
                //create the link if it doesn't already exist
                if ( !SMTUtil::isInt( $linkId ) || $linkId < 0 )
                {
                    //check that no link exists on the given port
                    //and update $linkTestDto with linkId found.
                    if ( !$linkHelper->checkLinkAndUpdateDto($linkTestDto) )
                    {                
            	        $linkId = $linkHelper->createLink( $linkTestDto );
                    }
                    else
                    {
                    	throw new SMTLinkTestException( SMTLinkTestException::ERROR_LINK_ALREADY_EXIST, sprintf("Link already exists on link %s!", $linkTestDto->getId() ) );
                    }                
                }
                
                //test if no test is already present on link
                if ( !$linkHelper->hasTest( $linkTestDto->getId() ) )
                {
            	    $testId = $linkHelper->createTest( $linkTestDto );
            	    
            	    //if monitoring and measure is allowed force sequencer on
            	    if ( $linkTestDto->isMonitoringAllowed() && $linkTestDto->isMeasurementAllowed() )
            	    {
            	    	$linkHelper->setSequensorOn();
            	    	$linkTestDto->setSequensorOn( TRUE );
            	    }
                }
                else
                {
                    throw new SMTLinkTestException( SMTLinkTestException::ERROR_TEST_ALREADY_EXIST, sprintf("Test already exists on link %s!", $linkTestDto->getId() ) );
                }
            }
            else
            {
            	throw new SMTLinkTestException( SMTLinkTestException::ERROR_LINK_HARDWARE_CONFIG, "Link cannot be added because the module or switch configuration is invalid!" );
            }            
    	}
    	catch ( \Exception $e )
    	{    	    
    	    $this->handleException($e);
    	}
    	
    	//notify link-test creation
    	SMTEventMessageManager::getInstance()->createAndSendLinkTestUpdateEvent( $linkTestDto );
    	
    	//if test creation succeeded returns the updated test DTO but don't fetch the new test because it could not yet have been applied on OTU.
    	return $linkTestDto->getJsonData();
    }      
    
    /**
     * Full json request data are by default stored in an array in parameter Default::$fullRequestDataName='request_Data'
     * Update a link and its monitoring test
     *
     * @url PUT
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function updateLinkTest( $request_data )
    {
        $linkHelper = new SMTLinkTest();
        $linkHelper->setContext( $this->getContext() );
        
    	try
    	{
    	    //check if the OTU hardware config is OK before creating a Link-test
    	    if ( $this->getContext()->getModuleConfigOK() && $this->getContext()->getSwitchConfigOK() )
    	    {
    	        $otdrTraceFileReplaced = FALSE;
            	$linkTestDto = SMTLinkTestDto::getInstance( $request_data );
            	$linkId = $linkTestDto->getId() ;       
            	
        		//check that the link exists on the given port:
        		//and update $linkTestDto with linkId found
        		if ( !$linkHelper->checkLinkAndUpdateDto($linkTestDto) )
        		{
        		    throw new SMTLinkTestException( SMTLinkTestException::ERROR_LINK_NOT_FOUND, sprintf("Link not found %s!", $linkId ) );
        		}
            	
            	//tests if the test is present on link
        		$testIdRetrieved = SMTLinkTest::parseLinkTestId( $this->sendReceive( SMTOtuApi::getTestOnLinkCommand( $linkId ) ) );
        		if ( $testIdRetrieved != $linkTestDto->getTestId() )
        		{
        			throw new SMTLinkTestException( SMTLinkTestException::ERROR_TEST_NOT_FOUND, sprintf("Test not found on link %s!", $linkId ) );
        		}        	
        		
            	//retrieve monitoring test current status
            	$isMonitoringAllowedOldValue = $linkHelper->isMonitoringAllowed( $linkTestDto->getTestId() );
            	
            	//update link name
            	$linkHelper->renameLinkTest($linkId, $linkTestDto->getName(), $linkTestDto->getPortNumber() );
            	
            	//update link and test availability
            	$measurementAllowed = $linkTestDto->isMeasurementAllowed();
            	$monitoringAllowed = $linkTestDto->isMonitoringAllowed();
            	$linkHelper->setLinkTestAvailable( $linkId, $measurementAllowed, $monitoringAllowed );        	
            	
                //update the test parameters on OTU if they have changed
                $testDetail =$linkTestDto->getTestDetail(); 
                if ( $testDetail != NULL && $testDetail->hasChanged() )
                {
                    SMTLogger::getInstance()->trace( "Test parameters have changed, update it.", SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
            	    $otdrTraceFileReplaced = $linkHelper->updateTestParameters( $linkTestDto );
                }
                
                //if monitoring and measure are allowed and monitoring was not allowed before, 
                //force sequencer on                       
                if ( ( $isMonitoringAllowedOldValue != $monitoringAllowed ) && $monitoringAllowed && $measurementAllowed )
                {
                    $linkHelper->setSequensorOn();
                    //update dto with sequencer status               
                    $linkTestDto->setSequensorOn( $linkHelper->isSequensorOn() );                
                }        	
                
                try 
                {
                	//start a test on demand to check the updated test
                	if ( $measurementAllowed && $monitoringAllowed )
                	{
                    	$testOnDemandHelper = new SMTTestOnDemand();
                    	$testOnDemandHelper->setContext( $this->getContext() );
                    	$testOnDemandHelper->startTest( $linkTestDto->getTestId(), $linkTestDto->getId(), FALSE );
                	}
                }
                //don't return an error if the test on demand cannot be launched
                catch(\Exception $e)
                {
                    $this->getContext()->getLogger()->traceException( $e );
                }
    
                //si l'on update la trace OTDR, il faut attendre qu'elle ait ete copiee par le sequenceur, pour ne pas risquer de recuperer l'ancienne courbe dans l'application cliente                
//                 if ( $otdrTraceFileReplaced && $measurementAllowed )
//                 {
//                     //TODO Pourra etre supprime si l'on modifie le client web pour ne plus redemander la courbe dans l'ecran de configuration du test apres la sauvegarde
//                 	sleep(5);
//                 }
                
            	//notify link-test update
            	SMTEventMessageManager::getInstance()->createAndSendLinkTestUpdateEvent( $linkTestDto );
        	}
        	else
        	{
        	    throw new SMTLinkTestException( SMTLinkTestException::ERROR_LINK_HARDWARE_CONFIG, "Link cannot be added because the module or switch configuration is invalid!" );
            }
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	
    	//if the test update succeeded, returns the updated test but don't fetch the new test because it could not yet have been applied on OTU.
    	return $linkTestDto->getJsonData();   	
    }        
    
    /**
     * id = link identifier
     *
     * @url DELETE /{id}
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function deleteLinkTest( $id )
    {
        $isSequensorOn = FALSE;
        //whether the test is the last remaining monitoring test on OTU
        $isLastRemainingTest = FALSE;
        //whether no monitoring test is running on OTU
        $isNoTestRunning = FALSE;
        
    	try
    	{
    	    SMTLinkTest::checkLinkTestId($id);
    	    
        	$linkHelper = new SMTLinkTest();
        	$linkHelper->setContext( $this->getContext() );
        	        	
        	//retrieve link info
        	$linkDto = $linkHelper->retrieveLinkInfo( $id );
        	
        	//check whether we are deleting the last remaining test
        	$testsIds = explode( ",", $this->sendReceive( SMTOtuApi::getMonitoringTestsListCommand() ) );
        	$isLastRemainingTest = is_array( $testsIds ) && ( count( $testsIds ) == 1 );
        	
        	$isNoTestRunning = $linkHelper->isNoTestRunning( $testsIds );
        	
        	//retrieve test identifier:
        	$testId = $linkDto->getTestId();
        	if ( $testId !== NULL && $testId >= 0 )
        	{
        	    //check that test exists before removing it
        	    $testIdRetrieved = SMTLinkTest::parseLinkTestId( $this->sendReceive( SMTOtuApi::getTestOnLinkCommand( $id ) ) );
        	    if ( $testIdRetrieved != $testId )
        	    {
        	        throw new SMTLinkTestException( SMTLinkTestException::ERROR_TEST_NOT_FOUND, sprintf("Test not found %s!", $testId ) );
        	    }
        	    
        	    //set test KO
        	    $this->send( SMTOtuApi::testAvailableCommand( $testId, SMTOtuApi::RES_KO ) );
        	    
        	    $isSequensorOn = $linkHelper->isSequensorOn();        	    
        	    
        	    //delete test        	
        	    $linkHelper->deleteTest( $testId );        	    
        	}
        	
        	//delete link
        	$linkHelper->deleteLink($id);
        	
        	//delete of the test will be performed at next sequencer switch, remove test info from link dto
        	$linkDto->removeTest();
        	
			if( $isSequensorOn && !$isNoTestRunning && !$isLastRemainingTest )
			{
			    //save in APC memory that a test is being deleted on OTU; end of addition will be sent synchronized with sequensor (through activity).
			    $event = SMTEventFactory::createLinkTestUpdateEvent( $linkDto, TRUE, FALSE );
			    $event->save();		     
			}
			else
			{
				//last monitoring test
				if ( $isLastRemainingTest && !$isNoTestRunning && $isSequensorOn )
				{
				    //sleep until the end of the current monitoring test cycle
				    SMTActivity::sleepForEndOfActivityWithTimout( 60 );				    
				}
				else
				{
				    //sleep a few seconds to let the OTU start deleting the test and the link (wait before sending the delete event to the client)
				    sleep( SMTLinkTest::SLEEP_TIME_FOR_TEST_REMOVAL_S );				    
				}
			    
            	//notify link-test deletion
            	SMTEventMessageManager::getInstance()->createAndSendLinkTestUpdateEvent( $linkDto, TRUE );            	           	
			}
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return SMTService::RESULT_OK;
    }            
    
    /**
     *
     * @url DELETE /test/{testid}
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function deleteTest( $testid )
    {
        $isSequensorOn = FALSE;
        //whether the test is the last remaining monitoring test on OTU
        $isLastRemainingTest = FALSE;
        //whether no monitoring test is running on OTU
        $isNoTestRunning = FALSE;
        
    	try
    	{
    		SMTLinkTest::checkTestId($testid);    			
    		$linkHelper = new SMTLinkTest();
    		$linkHelper->setContext( $this->getContext() );
    		$linkDto = $linkHelper->retrieveLinkInfoFromTest( $testid, TRUE );
    		
    		//check whether we are deleting the last remaining test
    		$testsIds = explode( ",", $this->sendReceive( SMTOtuApi::getMonitoringTestsListCommand() ) );
    		$isLastRemainingTest = is_array( $testsIds ) && count( $testsIds ) == 1;
    		 
    		$isNoTestRunning = $linkHelper->isNoTestRunning( $testsIds );
    		    		
			//set test KO
			$this->send( SMTOtuApi::testAvailableCommand( $testid, SMTOtuApi::RES_KO ) );									
			 
			$isSequensorOn = $linkHelper->isSequensorOn();
			
			//delete test
			$linkHelper->deleteTest( $testid );
			
			//delete of the test will be performed at next sequencer switch, remove test info from link dto		
			$linkDto->removeTest();			
			
			if( $isSequensorOn && !$isNoTestRunning && !$isLastRemainingTest )
			{
				//save in APC memory that a test is being deleted on OTU
				$event = SMTEventFactory::createLinkTestUpdateEvent( $linkDto );
				$event->save();
			}
			else
			{
			    //last monitoring test
				if ( $isLastRemainingTest && !$isNoTestRunning && $isSequensorOn )
				{
				    //sleep until the end of the current test monitoring cycle
				    SMTActivity::sleepForEndOfActivityWithTimout( 60 );				    
				}
				else
				{
				    //sleep a few seconds to let the OTU start deleting the test and the link (wait before sending the delete event to the client)
				    sleep( SMTLinkTest::SLEEP_TIME_FOR_TEST_REMOVAL_S );				    
				}
				 
				//notify test deletion
				SMTEventMessageManager::getInstance()->createAndSendLinkTestUpdateEvent( $linkDto );
			}
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	
    	//the link without its test
    	return $linkDto->getJsonData();
    }
    
    /**
     * Update link-test name
     * param id: link identifier
     * param newname: new link name 
     * 
     * @url PUT /{id}/name/{newname}
     * @access protected
     * @class SMTAccessControl {@requires user}
     */    
    function renameLinkTest( $id, $newname )
    {
        try
        {            
            $linkHelper = new SMTLinkTest();
            $linkHelper->setContext( $this->getContext() );
            $linkHelper->renameLinkTest($id, $newname );
            $linkDto = $linkHelper->retrieveLinkTest( $id );
        }
        catch ( \Exception $e )
        {
            $this->handleException($e);
        }  
        return $linkDto->getJsonData();     
    }

    /**
     * Set test measurement and monitoring status
     * id = LinkId
     * measurementAllowed = boolean
     * monitoringAllowed = boolean
     * 
     * @url PUT /{id}/available/{measureAllowed}/{monitoringAllowed}
     * 
     * @access protected
     * @class SMTAccessControl {@requires user}
     */    
    function setLinkTestAvailable( $id, $measureAllowed, $monitoringAllowed )
    {
        try
        {                                
            $linkHelper = new SMTLinkTest();
            $linkHelper->setContext( $this->getContext() );
            $measureAllowed = filter_var($measureAllowed, FILTER_VALIDATE_BOOLEAN);
            $monitoringAllowed = filter_var($monitoringAllowed, FILTER_VALIDATE_BOOLEAN);
            $linkHelper->setLinkTestAvailable( $id, $measureAllowed, $monitoringAllowed );
            
            //if monitoring and measure is allowed force sequencer on
            if ( $monitoringAllowed && $measureAllowed )
            {
            	$linkHelper->setSequensorOn();            
            }            
            $linkTestDto = $linkHelper->retrieveLinkTest( $id );
            
            //notify link-test update
            SMTEventMessageManager::getInstance()->createAndSendLinkTestUpdateEvent( $linkTestDto );
        }
        catch ( \Exception $e )
        {
        	$this->handleException($e);
        }
        return $linkTestDto->getJsonData();
    }
    
    /**
     *
     * @url POST /list
     * 
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchLinkTests($request_data)
    {
    	try
    	{
    	    $linkHelper = new SMTLinkTest();
    	    $linkHelper->setContext( $this->getContext() );
    	    
            //retrieve sequensor status    		    			
    	    $sequensorOn = $linkHelper->isSequensorOn();
    	    
    	    //retrieve the number of ports available:
    	    $portCount = $linkHelper->getPortsCount();

    	    //retrieve the module running number:
    	    $moduleRunningNumber = $linkHelper->getModuleRunningNumber();
    	    
    	    // virtual Switch Port Count
    	    $virtualSwitchPortCount = $linkHelper->getVirtualSwitchPortCount();    	    

    	    //handle pagination
    	    try
    	    {
    	    	if ($request_data != null)
    	    	{
    	    		$paginationDto = SMTPaginationDto::getInstance( $request_data );
    	    	}
    	    	else
    	    	{
    	    		$this->getContext()->getLogger()->trace( sprintf("Pagination data not available, set max port count %d", $portCount), SMTLogger::PINFO);
    	    		$paginationDto = new SMTPaginationDto();
    	    		$paginationDto->setRowPerPage( max($portCount, SMTPaginationDto::MAX_ROW_COUNT_PER_PAGE) );
    	    	}
    	    }
    	    catch ( \Exception $paginationException )
    	    {
    	    	//catch exception to retrieve links list even if an error occurs
    	    	$this->getContext()->getLogger()->traceException( $paginationException );
    	    	$paginationDto = new SMTPaginationDto();
    	    	$paginationDto->setRowPerPage( max($portCount, SMTPaginationDto::MAX_ROW_COUNT_PER_PAGE) );
    	    }    	    
    	    
    		$linkTestListDto = new SMTLinkTestListDto();
    		$linkTestListDto->setPortCount( $portCount );
    		$linkTestListDto->setSequensorOn( $sequensorOn );
    		$linkTestListDto->setVirtualSwitchPortCount( $virtualSwitchPortCount );
    		if ( $virtualSwitchPortCount > 0)
    		{
    		    $currentPortOtuDto = SMTMemoryManager::fetch(SMTOtuCurrentPortDto::getClass(), SMTOtuCurrentPortDto::CURRENT_PORT_DTO_ID);
    		    if ($currentPortOtuDto != NULL)
    		    {
    		        $linkTestListDto->setVirtualSwitchCurrentPort($currentPortOtuDto->getCurrentPort());
    		        
    		        //SMTLogger::getInstance()->trace("navigation: " .$paginationDto->isNavigationPagination() . "port : " .$currentPortOtuDto->getCurrentPort() );
    		            		        
    		        //si on n'a pas de navigation, on force la recherche de la page correspondant au port courant
    		        if (!$paginationDto->isNavigationPagination() && ($currentPortOtuDto->getCurrentPort() > 0))
    		        {
    		            $page = intval( (($currentPortOtuDto->getCurrentPort() -1 ) / $paginationDto->getRowPerPage()) ) + 1;
    		        	$paginationDto->setFirstPage( $page );
    		        	
    		        //	SMTLogger::getInstance()->trace("$page paginationDto page: ". $paginationDto->getFirstPage()."row per page:".$paginationDto->getRowPerPage() );
    		        }
    		    }
    		}
    		    		
    		try
    		{
    		    $linkTestListDto->setModuleRunningNumber($moduleRunningNumber);
    			$linkTestListDto->setModuleConfigOK( $this->getContext()->getModuleConfigOK() );
    		}
    		catch ( \Exception $invalidValueException )
    		{
    		    $linkTestListDto->setModuleConfigOK( FALSE );
    			//catch exception to retrieve links list even if an error occurs
    			$this->getContext()->getLogger()->traceException( $invalidValueException );
    		}
    		
    		try
    		{
    			$linkTestListDto->setSwitchConfigOK( $this->getContext()->getSwitchConfigOK() );
    		}
    		catch ( \Exception $ex )
    		{
    		    $linkTestListDto->setSwitchConfigOK( FALSE );
    			//catch exception to retrieve links list even if an error occurs
    			$this->getContext()->getLogger()->traceException( $ex );
    		}
    		
    		//retrieve links ids
    		$linksId = explode(",", $this->sendReceive( SMTOtuApi::getLinkListPerPortCommand($paginationDto->getFirstElementId(), min($paginationDto->getLastElementId(), $portCount) ) ) );
    		$linkTestDto = NULL;
    		foreach ( $linksId as $linkId)
    		{
    		    if ( SMTLinkTest::checkLinkTestId( $linkId, FALSE) )
    		    {
        		    $linkTestDto = $linkHelper->retrieveLinkTest( $linkId, $sequensorOn, $portCount );    		    
        		    if ( $linkTestDto != NULL )
        		    {
        		        $linkTestListDto->addLinkTest( $linkTestDto );
        		    }
    		    }    		    
    		}		
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	
    	try 
    	{
    	    $alarmHelper = new SMTAlarm();
    	    $alarmHelper->setContext( $this->getContext() );
    	    $alarmsDtoList = $alarmHelper->fetchOpticalAlarms();
    	    
    	    foreach ( $alarmsDtoList as $opticalAlarm)
    	    {
    	    	if ( $opticalAlarm != NULL )
    	    	{
    	    		$linkTestListDto->addOpticalAlarm( $opticalAlarm );
    	    	}
    	    }
    	    
    	    //force the fetch of events to retrieve the peak in alarms
    	    $alarmsDtoList = $alarmHelper->fetchPeakAlarms(TRUE);
    	    
    	    foreach ( $alarmsDtoList as $opticalAlarm)
    	    {
    	    	if ( $opticalAlarm != NULL )
    	    	{
     	    		$linkTestListDto->addOpticalAlarm( $opticalAlarm );
    	    	}
    	    }
    	}
    	catch( \Exception $e)
    	{
    	    //if optical alarms retrieval fails, allow to return the list of link-tests (don't throw exception to client).
    	    $this->getContext()->getLogger()->traceException( $e );
    	}
    	    
    	return $linkTestListDto->getJsonData();
    }           
    
    /**
     *
     * @url POST /test/{testid}/alarm/
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchOpticalAlarms( $testid )
    {
    	try
    	{
    		$alarmHelper = new SMTAlarm();
    		$alarmHelper->setContext( $this->getContext() );
    		$alarmDto = $alarmHelper->fetchOpticalAlarmFromTest( $testid );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return ( $alarmDto != NULL)? $alarmDto->getJsonData() : NULL;
    }
    
    /**
     *
     * @url POST /test/{testid}/allalarm/
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchAllAlarms( $testid )
    {
    	try
    	{
    		$alarmHelper = new SMTAlarm();
    		$alarmHelper->setContext( $this->getContext() );
    		$attenuationOpticalAlarm= $alarmHelper->fetchOpticalAlarmFromTest( $testid, TRUE );
    		
    		$alarmsOnTest = new SMTOpticalAlarmListDto();
    		if ($attenuationOpticalAlarm != null )
    		{
    			$alarmsOnTest->setAttenuationOpticalAlarm($attenuationOpticalAlarm);
    		}
    		
    		$peakAlarms = $alarmHelper->fetchPeakAlarmFromTest( $testid, TRUE );
    		$alarmsOnTest->setPeakOpticalAlarms($peakAlarms);
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	
    	return ( $alarmsOnTest!= NULL)? $alarmsOnTest->getJsonData() : NULL;
    }
    
    /**
     *
     * @url DELETE /test/{testid}/alarm/
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function deleteOpticalAlarm( $testId )
    {
    	try
    	{
    		$alarmHelper = new SMTAlarm();
    		$alarmHelper->setContext( $this->getContext() );
    		$alarmHelper->deleteOpticalAlarmOnOtu( $testId );
    		$alarmHelper->deleteAllPeakAlarmOnOtu( $testId );
    		//don't clear the alarm on smartOTU, wait for the clear comming from OTU
//     		$alarmHelper->deleteOpticalAlarmOnSmartOtu( $testId );
    		
    		//notify listeners that optical and system alarms should be refreshed
//     		SMTEventMessageManager::getInstance()->sendRefreshOpticalAlarms();
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return SMTService::RESULT_OK;
    }    
    
    /**
     * id = link identifier
     * monitor = Whether the test on demand process has to be monitored (False for SmartOTU)
     * @url PUT /{id}/testOnDemand
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function startTest( $id )
    {
    	return $this->startTestAndMonitor( $id, TRUE );
    }    
    
    /**
     * id = link identifier
     * monitor = Whether the test on demand process has to be monitored (False for SmartOTU)
     * @url PUT /{id}/testOnDemand/{monitor}
     * 
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function startTestAndMonitor( $id, $monitor )
    {
        $testStatusDto = NULL;
        try
        {
        	SMTLinkTest::checkLinkTestId($id);
        	$testOnDemandHelper = new SMTTestOnDemand();
        	$testOnDemandHelper->setContext( $this->getContext() );
        	
        	//retrieve test identifier:
        	$linkTestIds = $this->sendReceive( SMTOtuApi::getTestOnLinkCommand( $id ) );
        	$testId = SMTLinkTest::parseLinkTestId( $linkTestIds );
        	if ( $testId !== NULL && $testId >= 0 )
        	{        	        	
        	    $testStatusDto = $testOnDemandHelper->startTest( $testId, $id, ( $monitor === TRUE )? TRUE : FALSE );
        	}
        	else
        	{
        	    throw new SMTLinkTestException( SMTLinkTestException::ERROR_INVALID_TEST_ID );
        	}
        }
        catch ( \Exception $e )
        {
        	$this->handleException($e);
        }
        return $testStatusDto->getJsonData();
    }
    
    /**
     * WARNING: POLLING MUST BE DONE AS POST TO GUARANTEE THAT THE HTTP REQUEST WILL BE RECEIVED BY THE SERVER
     * 
     * id = link identifier
     * @url POST /{id}/testOnDemand/{operationId}/status
     *
     * Access granted to everybody (used by asynchronous scripts).
     * Control is made on operation identifier which is generated from the server.
     */
    function getTestStatus( $id, $operationId )
    {    
    	$measureStatusDto = SMTEventFactory::createTestOnDemandEvent(NULL, $operationId, SMTOperationEventDto::NO_PROGRAMMED );
    	$this->getContext()->getLogger()->trace( sprintf( "getTestStatus from PHP REST API for linkId %s and operation id: %s !", $id, $operationId ) );
    	     	
    	try
    	{
    		SMTLinkTest::checkLinkTestId( $id );
    		if ( SMTOperationManager::hasOperation( $operationId ) )
    		{
    		    //retrieve test identifier:
    		    $linkTestIds = $this->sendReceive( SMTOtuApi::getTestOnLinkCommand( $id ) );
    		    $testId = SMTLinkTest::parseLinkTestId( $linkTestIds );
    		    if ( $testId !== NULL && $testId >= 0 )
    		    {    		    
        			$testOnDemandHelper = new SMTTestOnDemand();
        			$testOnDemandHelper->setContext( $this->getContext() );
        			$measureStatusDto = $testOnDemandHelper->getTestStatus( $testId, $operationId  );
    		    }
    		}
    	}
    	 
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $measureStatusDto->getJsonData();
    }    
    
    /**
     * id = link identifier
     * @url PUT /{id}/measure
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function startMeasureOnLink( $id, $request_data )
    {
    	try
    	{
    		SMTLinkTest::checkLinkTestId($id);
    		$acquisitionParametersDto = SMTStartOtdrMeasureOnDemandParametersDto::getInstance( $request_data );
    		
    		$measureOnDemandHelper = new SMTMeasureOnDemand();
    		$measureOnDemandHelper->setContext( $this->getContext() );
    		$measurementStatusDto = $measureOnDemandHelper->startMeasureOnLink( $acquisitionParametersDto );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $measurementStatusDto->getJsonData();
    } 
    
    /**
     * WARNING: POLLING MUST BE DONE AS POST TO GUARANTEE THAT THE HTTP REQUEST WILL BE RECEIVED BY THE SERVER
     * 
     * id = link identifier
     * @url POST /{id}/measureOnDemand/{operationId}/status
     *
     * Access granted to everybody (used by asynchronous scripts).
     * Control is made on operation identifier which is generated from the server.
     */
    function getMeasurementStatus( $id, $operationId )
    {
    	$measureStatusDto = SMTEventFactory::createMeasureOnDemandEvent($id, $operationId, SMTOperationEventDto::NO_PROGRAMMED );
    
    	$this->getContext()->getLogger()->trace( sprintf( "getMeasurementStatus from PHP REST API for linkId %s and operation id: %s !", $id, $operationId ) );
    	 
    	try
    	{
    		SMTLinkTest::checkLinkTestId( $id );
    		if ( SMTOperationManager::hasOperation( $operationId ) )
    		{
    			$measureOnDemandHelper = new SMTMeasureOnDemand();
    			$measureOnDemandHelper->setContext( $this->getContext() );
    			$measureStatusDto = $measureOnDemandHelper->getMeasurementStatus( $operationId );
    		}
    	}
    	 
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $measureStatusDto->getJsonData();
    }    
    
    /**
     * id = link identifier
     * @url GET /{id}
     * 
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchLinkTest( $id )
    {
    	try
    	{    	
    	    SMTLinkTest::checkLinkTestId($id);
    	    
    	    $linkHelper = new SMTLinkTest();
    	    $linkHelper->setContext( $this->getContext() );
    	    $linkDto = $linkHelper->retrieveLinkTest( $id );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $linkDto->getJsonData();
    }
    
    /**
     * Fetch link test with its test detail info if it exists
     * id = link identifier
     * 
     * @url GET /{id}/detail
     * 
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchLinkTestDetail( $id )
    {
    	try
    	{
    	    SMTLinkTest::checkLinkTestId($id);
    	    
    		$linkHelper = new SMTLinkTest();
    		$linkHelper->setContext( $this->getContext() );
    		$linkDto = $linkHelper->retrieveLinkTest( $id );
    		
    		//don't crash if we don't have any test
    		if ( $linkDto->getTestId() !== NULL )
    		{
        		$linkTestDetailDto = $linkHelper->retrieveLinkTestDetail( $linkDto->getTestId() );
        		$linkDto->setTestDetail( $linkTestDetailDto );
    		}
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $linkDto->getJsonData();
    }    
    
    
    
    /**
     * Fetch test detail info
     * testid = test identifier
     *
     * @url GET /test/{testid}
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchTestDetail( $testid )
    {
    	try
    	{
    		SMTLinkTest::checkLinkTestId($testid);
    			
    		$linkHelper = new SMTLinkTest();
    		$linkHelper->setContext( $this->getContext() );
    	    $testDetailDto = $linkHelper->retrieveLinkTestDetail( $testid );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $testDetailDto->getJsonData();
    }
    
    /**
     * Fetch test detail info
     * portid = port identifier
     *
     * @url GET /port/{portid}
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchTestDetailByPort( $portid )
    {
    	try
    	{
    	    //retrieve links ids
            $linkId = $this->sendReceive( SMTOtuApi::getLinkListPerPortCommand($portid, $portid) );
            $testDetailDto = NULL;
        	if ( SMTLinkTest::checkLinkTestId( $linkId ) )
        	{
    		    //retrieve main test info
    		    $testId = SMTLinkTest::parseLinkTestId( $this->sendReceive( SMTOtuApi::getTestOnLinkCommand( $linkId ) ) );
        		SMTLinkTest::checkLinkTestId($testId);
        			
        		$linkHelper = new SMTLinkTest();
        		$linkHelper->setContext( $this->getContext() );
        		$testDetailDto = $linkHelper->retrieveLinkTestDetail( $testId );
        	}
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $testDetailDto->getJsonData();
    }    
    
    /**
     * Init a new link test DTO with its link name and test detail info initialized with default values.
     *
     * @url POST /{portNumber}/{moduleRunningNumber}/init
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function initLinkTest( $portNumber, $moduleRunningNumber )
    {
    	try
    	{    			
    		$linkHelper = new SMTLinkTest();
    		$linkHelper->setContext( $this->getContext() );
    		
    		$linkDto = $linkHelper->initLinkTest( NULL, $portNumber, $moduleRunningNumber);
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $linkDto->getJsonData();
    }
    
    /**
     * Init a test detail DTO in an existing link; its test detail info are initialized with default values.
     * id = link identifier
     *
     * @url POST /{id}/init
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function initTestDetail( $id )
    {
    	try
    	{
    		$linkHelper = new SMTLinkTest();
    		$linkHelper->setContext( $this->getContext() );
    
    		$linkDto = $linkHelper->initLinkTest( $id, NULL, NULL);
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $linkDto->getJsonData();
    }
    
    
    /**
     * Fetch a test detail or if it doesn't exist:
     *  - create a default test detail DTO in an existing link; its test detail info are initialized with default values.
     *  - create a default test detail DTO on a port; its test detail info are initialized with default values.
     * portid = port identifier
     *
     * @url POST /port/{portid}
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function initTestByPort($portid)
    {
        try
        {
        	$linkHelper = new SMTLinkTest();
        	$linkHelper->setContext( $this->getContext() );
        	
        	//test if a link exists:
        	$linkId = $this->sendReceive( SMTOtuApi::getLinkListPerPortCommand($portid, $portid) );
        	$testDetailDto = NULL;
        	if ( SMTLinkTest::checkLinkTestId( $linkId, FALSE ) )
        	{
        		//does a test exist?
        		$testId = SMTLinkTest::parseLinkTestId( $this->sendReceive( SMTOtuApi::getTestOnLinkCommand( $linkId ) ) );
        		if ( SMTLinkTest::checkLinkTestId($testId, FALSE) )
        		{    
        		    $linkDto = $linkHelper->retrieveLinkTest( $linkId );
        		    
        		    //don't crash if we don't have any test
        		    if ( $linkDto->getTestId() !== NULL )
        		    {
        		    	$linkTestDetailDto = $linkHelper->retrieveLinkTestDetail( $linkDto->getTestId() );
        		    	$linkDto->setTestDetail( $linkTestDetailDto );
        		    }        		    
        		}
        		else
        		{  		    
        		    $linkDto = $linkHelper->initLinkTest( $linkId, NULL, NULL);
        		}
        	}
        	else
        	{
        	    //retrieve the module running number:
        	    $moduleRunningNumber = $linkHelper->getModuleRunningNumber();
        	    $linkDto = $linkHelper->initLinkTest( NULL, $portid, $moduleRunningNumber);
        	}
        }
        catch ( \Exception $e )
        {
        	$this->handleException($e);
        }
        
        return $linkDto->getJsonData();
    }
    
    /**
     * Generate gps file from kml and associate it to OTU link or replace an existing one.
     * id = link identifier
     * filename = kml file
     *
     * @url POST /{id}/kml/{filename}
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function processKml( $id, $filename )
    {
    	try
    	{
    	    $kmlHelper = new SMTGpsRoute();
    	    $kmlHelper->setContext( $this->getContext() );
    	    	
    		$linkHelper = new SMTLinkTest();
    		$linkHelper->setContext( $this->getContext() );    		    		
    		$linkDto = $linkHelper->retrieveLinkTest( $id );
    		//remove old gps file already associated
    		$gpsFileName = $linkDto->getGpsFileName(); 
    		if ( !empty( $gpsFileName ) )
    		{
    		    try 
    		    {
    		        $kmlHelper->deleteKml($gpsFileName, $id);
    		    }
    		    catch( \Exception $ex)
    		    {
    		        $this->getContext()->getLogger()->traceException($ex);
    		    }
    		}
    		$kmlHelper->processKml($filename, $id);
    		    		
			$linkDto = $linkHelper->retrieveLinkTest( $id );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $linkDto->getJsonData();
    }    
    
    /**
     * Delete gps file and dissociate it from OTU link
     * id = link identifier
     *
     * @url DELETE /{id}/kml
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function deleteKml( $id )
    {
    	try
    	{
    	    $linkHelper = new SMTLinkTest();
    	    $linkHelper->setContext( $this->getContext() );
    	    $linkDto = $linkHelper->retrieveLinkTest( $id );
    	        	    	    	    
    		$kmlHelper = new SMTGpsRoute();
    		$kmlHelper->setContext( $this->getContext() );
    		$kmlHelper->deleteKml($linkDto->getGpsFileName(), $id);
    
    		$linkDto = $linkHelper->retrieveLinkTest( $id );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $linkDto->getJsonData();
    }

    /**
     * Request to download the given kml file...
     * Can be accessed outside SmartOTU: no security access
     *
     * @url GET /kml
     */
    function downloadKmlFile( $request_data )
    {
    	try
    	{
    		$fileName = $request_data['fileName'];
    		$kmlFileName = SMTGpsRoute::retrieveKmlFullPath($fileName);
    		 
    		if ( $kmlFileName != NULL )
    		{
    			$page = new SMTFileDownloadPage($kmlFileName);
    			$this->getContext()->getHttpResponse()->send( $page );
    		}
    		else
    		{
    			$e = new SMTIOException( SMTIOException::FILE_NOT_FOUND );
    			$this->handleException($e);
    		}
    	}
    	catch( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    }
        
    
    /**
     * Fetch test acquisition parameters
     * id = link identifier
     * 
     * @url GET /{id}/acquisitionParameters
     * 
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchTestAcquisitionParameters( $id )
    {
        $linkHelper = new SMTLinkTest();
        $linkHelper->setContext( $this->getContext() );
        
        try
    	{
    	    SMTLinkTest::checkLinkTestId($id);
    	    
    	    //retrieve test identifier:
    	    $linkTestIds = $this->sendReceive( SMTOtuApi::getTestOnLinkCommand( $id ) );
    	    $testId = SMTLinkTest::parseLinkTestId( $linkTestIds );
    	    
    	    SMTLinkTest::checkTestId( $testId );    	        	    
       	    $testDetailedDto = $linkHelper->fetchTestAcquisitionParameters( $testId, $id );

    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $testDetailedDto->getJsonData();
    }
    
    
    /**
     * Retrieve current port used by virtual switch
     *
     * @url GET /currentPort
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getCurrentPort()
    {
    	try
    	{
    	    $linkHelper = new SMTLinkTest();
    	    $linkHelper->setContext( $this->getContext() );
    	    $dto = new SMTCurrentPortDto();
    	    
    	    // virtual Switch Port Count
    	    $virtualSwitchPortCount = $linkHelper->getVirtualSwitchPortCount();

    	    if ( $virtualSwitchPortCount > 0)
    	    {
    	        $dto->setVirtualSwitchPortCount( $virtualSwitchPortCount );
    	        $currentPortOtuDto = SMTMemoryManager::fetch(SMTOtuCurrentPortDto::getClass(), SMTOtuCurrentPortDto::CURRENT_PORT_DTO_ID);
    	        if ($currentPortOtuDto != NULL)
    	        {
        	    	$dto->setCurrentPort( $currentPortOtuDto->getCurrentPort() );
        	    }    	    
    	    }
    
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $dto->getJsonData();
    }
    
    
    
    /**
     * Create all links
     *
     * @url GET /test/createAllLinks
     */
    function testCreateAllLinks()
    {
    	$linkId = -1;
    	$linkHelper = new SMTLinkTest();
    	$linkHelper->setContext( $this->getContext() );
    
    	try
    	{
    		$linkTestDto = SMTLinkTestDto::getInstance( $request_data );

			$linkId = $linkHelper->createLink( $linkTestDto );
			$linkTestDto->setId( $linkId );
    	}
    	catch ( \Exception $e )
    	{    
    		$this->handleException($e);
    	}
    }    
    
    /**
     * @url GET /test/addtest/{portNumber}
     */
    function testAddTest( $portNumber )
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$linkTestDto = SMTLinkTest::createTestData( $portNumber );         
        	$url = '/tests';
        	$json_string = json_encode( $linkTestDto->getJsonData() );
        	$page = new SMTTestRestApiPage($url, $json_string, 'POST');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }    
    
    /**
     * @url GET /test/updatetest/{portNumber}/{linkId}/{testId}
     */
    function testUpdateTest($portNumber, $linkId, $testId )
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$linkTestDto = SMTLinkTest::createTestData( $portNumber );
        	$linkTestDto->setId($linkId);
        	$linkTestDto->setTestId($testId);
        	$url = '/tests/';
        	$json_string = json_encode( $linkTestDto->getJsonData() );
        	$page = new SMTTestRestApiPage($url, $json_string, 'PUT');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }
    
    /**
     * @url GET /test/initlinktest/{portNumber}/{moduleRunningNumber}
     */
    function testInitLinkTest($portNumber, $moduleRunningNumber )
    {    	
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$url = '/tests/'.$portNumber.'/'.$moduleRunningNumber.'/init';
        	$page = new SMTTestRestApiPage($url, NULL, 'POST');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }  

    /**
     * @url GET /test/inittestdetail/{linkId}
     */
    function testInitTestDetail( $linkId )
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$url = '/tests/'.$linkId.'/init';
        	$page = new SMTTestRestApiPage($url, NULL, 'POST');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }    

    /**
     * @url GET /test/renametest/{linkId}
     */
    function testRenameTest($linkId)
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$url = '/tests/'.$linkId.'/rename/newTestName';
        	$page = new SMTTestRestApiPage($url, NULL, 'PUT');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }    
    
    /**
     * @url GET /test/deletetest/{linkId}
     */
    function testDeleteTest($linkId)
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$url = '/tests/'.$linkId;
        	$page = new SMTTestRestApiPage($url, "", 'DELETE');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }    
    
    /**
     * @url GET /test/availabletest/{linkId}
     */
    function testAvailableTest($linkId)
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$url = '/tests/'.$linkId.'/available/true/true';
        	$page = new SMTTestRestApiPage($url, NULL, 'PUT');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }    
    
    /**
     * @url GET /test/testondemand/{linkId}
     */
    function testStartTestOnDemand($linkId)
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$url = '/tests/'.$linkId.'/testOnDemand';
        	$page = new SMTTestRestApiPage($url, "", 'PUT');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }    
    
//     /**
//      * @url GET /teststatus
//      */
//     function testTestOnDemandStatus()
//     {
//     	$url = '/tests/1/testOnDemand/'.session_id().'/status';
//     	$page = new SMTTestRestApiPage($url, "", 'GET');
//     	$this->getContext()->getHttpResponse()->send( $page );
//     }    
    
    /**
     * @url GET /test/measureondemand/{linkId}
     */
    function testStartMeasureOnDemand($linkId)
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
            $measurementDto = SMTMeasureOnDemand::createMeasureOnDemandTestData();
        	$url = '/tests/'.$linkId.'/measure';
        	$json_string = json_encode( $measurementDto->getJsonData() );
        	$page = new SMTTestRestApiPage($url, $json_string, 'PUT');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }    
}