#!/bin/bash
#
#
export PATH=$PATH:/sbin:/usr/sbin

usage() {
  echo usage :
  echo "	$PRG_NAME --tar archive.tar --target <RESCUE or NORMAL> --index <index name or number>  [--fromscratch] [--pmulti] [--noreboot] [--nodet] [--nomd5] [--short] [--help]"
  echo "	--tar : the path to the installation tarball"
  echo "	--target : can be RESCUE or NORMAL. Indicates which filesystem will be updated/installed"
  echo "	--index : Specifies which index of the release_soft.index file must be used (normal or rescue are the common index for OTU). Numeric values are also accepted"
  echo "	--fromscratch : The flash will be erased and the filesystem completely reinstalled. Configurations files are saved and restored. Cannot be used to reinstall the filesystem we're running from."
  echo "	--pmulti : The install process uses a pmulti. Used to reinstall the normal filesystem from the normal filesystem. Configurations files are saved and restored."
  echo "	--noreboot : tells the script no to reboot after its done with the installation"
  echo "	--nomd5 : tells the script no to look for an md5 file along the tarball"
  echo "	--nodet : tells the script not to detach the console to get the output of the script in the current console"
  echo "	--short : use default values for the unspecified parameters instead of cowardly refusing to run."
  echo "	          Those default values are :  --nomd5 is set"
  echo "	                                      --noreboot is set"
  echo "	                                      --nodet is not set"
  echo "	                                      --fromscratch is NOT set"
  echo "	                                      --index is normal"
  echo "	                                      --target is NORMAL"
  echo "	                                      --tar is the tarball of the /otu/release/current with latest modification time"
  exit -1
  }
PRG_NAME=`basename $0`
#echo "script $0 $@"
# Attention: test de l'absence de parametre a n'executer que lors du premier appel; pas a la deuxieme invocation par nohup ou l'on n'a plus de parametre (perdus dans le parsing "parse params for this script")
if [ -z "${DETACHED}" ] ; then
   [ $# -eq 0 ] && usage
fi
# ce script ne fonctionne pas correctement: retourne toujours 0 et donc, n'appelle jamais: usage(). Non corrige, car pourrait etre bloquant si l'on ajoutait un parametre et que l'on oubliait de la rajouter ici.
set -- `getopt -n$0 -u --longoptions="tar: index: pmulti fromscratch target: noreboot nodet help nomd5 short upgradeversion" "h" "$@"` || usage

#parse params for this script
while [ $# -gt 0 ]
do
    case "$1" in
       --tar)   		export TARBALL=$2;shift;;
       --index)   		export INDEX=$2;shift;;
       --target) 		export TARGET=$(echo $2 | tr '[:upper:]' '[:lower:]') ;shift;;
       --fromscratch)   export FROMSCRATCH="--fromscratch";;
       --pmulti)   	    export PMULTI=1;;
       --noreboot)    	export NOREBOOT=1;;
       --nomd5)		    export NOMD5=1;;
       --upgradeversion) SHOW_VERSION=1;;
       --help)        	usage;;
       --short)  	    export USE_DEFAULT=1;;
       --nodet)     	NOTDETACHED=1;;
       -h)        	    usage;;
       --) 		        shift;break;;
       -*)        	    usage;;       
       *)         	    usage;;
    esac
    shift
done

# Display the version if asked and exit without doing anything else
if [ ! -z "${SHOW_VERSION}" ] ; then
     VERSION=$(tar --test-label -f ${TARBALL})
     echo ${VERSION} | cut -f 2 -d ':' | tr -d ' '
     exit 0
fi

if [ ! -z ${USE_DEFAULT} ] ; then
	# if the tarball hasn't been specified then use the tarball located in /otu/release/current with the mostrecent modification time
	[ -z ${TARBALL} ] && TARBALL=`ls -t /otu/release/current/*.tar* | head -n 1` && echo "no tarball specified. using default tarball : ${TARBALL}"
	TARGET=normal 
	INDEX=normal
	NOMD5=1
fi

# echo tarball $TARBALL
# echo target $TARGET
# echo fromscratch $FROMSCRATCH
# echo noreboot $NOREBOOT
# echo other arguments param [nb = $#] $*

#check the parameters
if [ -z ${TARBALL} ] ; then
  echo "missing the installation Tar file in command line"
  usage
  exit -1
fi

if [ ! -f ${TARBALL} ] ; then
  echo "${TARBALL} is not a valid file"
  usage
  exit -1
fi

# Warning DETACHED variable used only to skip this nohup code for the second invocation which really start the upgrade. Keep the name "DETACHED" for legacy code
if [ -z "${DETACHED}" ] ; then
   export DETACHED=1
   export LOGFILE=/var/log/otu_upgrade_$(date +%y-%m-%d-%H.%M.%S)
    
   #detached case   
   if [ -z "$NOTDETACHED" ] ; then
      if [ ! -z "${DEBUG}" ] ; then
         # $* ne sert a rien car les parametres sont perdus lors du premier parsing dans "parse params for this script"
         nohup $0 $* &
      else
         # $* ne sert a rien car les parametres sont perdus lors du premier parsing dans "parse params for this script"
         nohup $0 $* > /dev/null &
      fi
   # Not detached case
   else
      # $* ne sert a rien car les parametres sont perdus lors du premier parsing dans "parse params for this script"
      nohup $0 $* 
      cat ${LOGFILE} 
   fi
   sleep 1  
   exit
fi

#Magic trick to log the output on both the console and a file without invoking tee with a '|'
#create a named pipe that will be used for stdout and stderr. this pipe will be connected to 'tee'
PIPEFILE=$(mktemp -t)
rm ${PIPEFILE}
mkfifo ${PIPEFILE}
tee ${LOGFILE} < ${PIPEFILE} &
# capture tee's process ID for the wait command.
TEEPID=$!
# redirect the rest of the stderr and stdout to our named pipe.
exec > ${PIPEFILE} 2>&1

# Get partition information
source /etc/otu/otuparts.sh

STATUS_LOG_FILE=/otu/release/current/upgrade_status
if [ ! -e ${STATUS_LOG_FILE} ] && [ -d "/otu" ] ; then
   mkdir -p $(dirname ${STATUS_LOG_FILE})
fi

function log_msg() {
   echo $* >> ${STATUS_LOG_FILE}
   echo $*
}


#check if an upgrade is already running
LOCKFILE_FILE=/tmp/otu_upgrade_lock
if [ -f ${LOCKFILE_FILE} ] ; then
    echo "An upgrade is already in progress ($LOCKFILE_FILE exists). Exiting..."
    log_msg "CANCELLED : ALREADY RUNNING"
    exit -1
fi

if [ -f /var/volatile/upgrade_in_progress ] ; then
    echo "An upgrade is already in progress (/var/volatile/upgrade_in_progress exists). Exiting..."
    log_msg "CANCELLED : DUPLICATION RUNNING"
    exit -1
fi

TRAP_NO_REBOOT="exec 1>&- 2>&- ;  wait $TEEPID; rm -f $LOCKFILE_FILE $PIPEFILE"
TRAP_WITH_REBOOT="exec 1>&- 2>&- ;  wait $TEEPID; rm -f $LOCKFILE_FILE $PIPEFILE ; log_msg 'FAILURE : unexpected end of the upgrade script' ; do_reboot"

#no upgrade running. create a temporary lock file and register its removal when the script is terminated
trap  "${TRAP_NO_REBOOT}" EXIT
touch ${LOCKFILE_FILE}

MAX_USED_SPACE_ALLOWED=90
INSTALLATION_ROOT_FOLDER=/otu/release
INSTALLATION_TMP_FOLDER=/otu/release/tmp

RET=0

FW_PART=$(fw_printenv mmc_part)
if [ "${CURRENT_PART:(-1)}" != "${FW_PART:(-1)}" ] ; then
    echo "An upgrade has been done already, reboot required before starting another one."
    exit -1
fi

#check if fromscratch can be used
[ ! -z "${FROMSCRATCH}" ] && [ "${CURRENT_FS}" == "${TARGET}" ] && { echo "can't do an installation from scratch on the current filesystem!" ; exit -1 ; }
#check if pmulti can be used
[ ! -z "${PMULTI}" ] && [ "${CURRENT_FS}" == "rescue" ] && { echo "can't do an installation from pmulti from the rescue filesystem!" ; exit -1 ; }
[ ! -z "${PMULTI}" ] && [ "rescue" == "${TARGET}" ] && { echo "can't do an installation of the rescue filesystem from pmulti!" ; exit -1 ; }

function do_reboot() {
if [ -z "$NOREBOOT" ] ; then
   REBOOT_DELAY=15
   echo "no --noreboot option detected. rebooting in $REBOOT_DELAY secs ..."
   sleep ${REBOOT_DELAY}
   sync
   reboot
fi
}

echo $(date) > ${STATUS_LOG_FILE}

log_msg "installing : ${TARBALL}"

#installing a new trap
trap  "${TRAP_WITH_REBOOT}" EXIT

#untar the tarball
log_msg "Uncompressing the tarball."
if [ "$RET" == "0" ] ; then
      echo extracting ${TARBALL} ...
      rm -rf ${INSTALLATION_TMP_FOLDER}
      mkdir -p ${INSTALLATION_TMP_FOLDER} && tar xf ${TARBALL}  --no-same-owner --no-same-permissions -C ${INSTALLATION_TMP_FOLDER}
      RET=$? ; 
      [ "$RET" != "0" ] && log_msg "Error while extracting $TARBALL in ${INSTALLATION_TMP_FOLDER}"  || log_msg "done..."
      ! [ -e ${INSTALLATION_TMP_FOLDER}/release.xml ] && log_msg "Error no release.xml in tarball!" && RET=1
fi

if [ "$RET" == "0" ] ; then
	#start the upgrade
	log_msg "UPGRADING..."

	# get back log from syslog to the log file
	tail -n 0 -f /var/log/daemon.log &
	LOG_REDIRECT=$!

	# Check release-manager daemon is running
	RELEASE_MANAGER_PID=$(pgrep release-manager)
	[ -z ${RELEASE_MANAGER_PID} ] && /etc/init.d/release-manager restart && log_msg "Release-manager was not running! starting it"

	# Launch client for release_manager : starting the operation but not doing it.
	release-manager start_upgrade 0:${INSTALLATION_ROOT_FOLDER}/tmp/release.xml
	if [ "$?"  != "0" ]; then
		log_msg "Release-manager client has failed to start"
		RET=1
		kill ${LOG_REDIRECT}
	else
		sleep 1  		# lets the daemon start its job	
	fi
fi
if [ "${RET}" == "0" ] ; then
    fw_setenv upgrade_pending yes
    # Do not sync when duplicating
    /etc/init.d/otu_rfs_sync.sh stop
	# Wait for the upgrade to be completed
	OPERATION_COMPLETED="NO"

	while [ ${OPERATION_COMPLETED} == "NO" ] ; do
		sleep 1
		kill -0 ${RELEASE_MANAGER_PID}
		[ $? -ne 0 ] && log_msg "Release-manager has stopped (probably crash!), upgrade failed" && OPERATION_COMPLETED="YES"
		! [ -e /var/lib/release-manager/upgrade_status ] &&  log_msg "No status file for upgrade! upgrade failed" && OPERATION_COMPLETED="YES"
		CURRENT_STATUS=$(head -1 /var/lib/release-manager/upgrade_status)
		[ "$CURRENT_STATUS" != "Started" ] && log_msg "Upgrade operation finished with status $CURRENT_STATUS" && OPERATION_COMPLETED="YES"
	done

	RET=1
	[ $(head -1 /var/lib/release-manager/upgrade_status) == "Success" ] && RET=0
	kill ${LOG_REDIRECT}
fi



if [ "${RET}" == "0" ] ; then
	echo ${CURRENT_PART} > ${INSTALLATION_TMP_FOLDER}/first_boot_after_upgrade.tmp
fi


# Set the partition to be booted on
if [ "${RET}" == "0" ] ; then
    fw_setenv mmc_part ${ALT_PART:(-1)}
    # reset eotdrv2 flag, do not mess with uboot volatile boot
    fw_setenv mmc_part_volatile
fi

trap "${TRAP_NO_REBOOT}" EXIT

if [ ${RET} != "0" ] ; then
    log_msg "FAILURE : exiting with error code ${RET}"
else
    log_msg "SUCCESS"
fi

# Delete the duplication log file for avoid onmsi reading the file at startup before the duplication even started
rm -f ${INSTALLATION_ROOT_FOLDER}/current/duplication_status
otu_confirm_start.sh

do_reboot

echo "${RET}"
exit ${RET}
