<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of VIAVI
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from VIAVI
// Copyright VIAVI 2016. All rights reserved.
// *********************************************************
namespace app\serviceshelper\media;

use app\serviceshelper\alarm\SMTOtuAlarmSeverity;

use app\services\alarm\SMTOpticalAlarmDto;

use app\http\SMTHttpPostJson;

use app\services\alarm\SMTSystemAlarmDto;

use app\serviceshelper\alarm\SMTAlarmSpecificProblemCode;

use app\serviceshelper\monitoring\SMTLinkTest;

use app\services\alarm\SMTAlarmDto;

use app\serviceshelper\alarm\SMTAlarmSeverity;

use app\serviceshelper\system\SMTOtu;

use app\services\alarm\SMTAttenuationOpticalAlarmDto;

use app\services\alarm\webservice\SMTWsAlarmDto;

use app\services\alarm\webservice\SMTWsOpticalAlarmAttenuationDto;

use app\util\SMTLogger;
use app\serviceshelper\SMTServiceHelper;
use app\parser\SMTOtuApi;
use app\http\SMTContext;
use app\util\SMTUtil;


/**
 * Create the content of the alarms sent by web service
 *
 * @author Sylvain Desplat
 *
 */
class SMTWebService extends SMTServiceHelper
{        
    const NO_ALARM      = 0;
    const OPTICAL_ALARM = 1;
    const SYSTEM_ALARM  = 2;
    const OPTICAL_SYSTEM_ALARM = 3;
    
    const OPTIC_STRING  = 'optic';
    const SYSTEM_STRING = 'system';
    const PORT_STRING   = 'port';
	
    const MEDIA_INDEX = 0;
    const MEDIA_URL = 1;
    const MEDIA_ALARM_TYPE = 2;
    
    /**
     * url of the web service. By default local web server for tests
     * 
     * @var String
     */
    private $webServiceUrl = "http://127.0.0.1/ws/alarm";
    
    /**
     * Set the url of the web service
     *
     * @param String $webServiceUrl
     */
    public function setWebServiceUrl($webServiceUrl)
    {
    	$this->webServiceUrl = $webServiceUrl;
    }
    
    /**
     * Send a test alarm (optical or system) to the web service
     *
     * @param String $alarmType
	 * @param $otuSerialNumber
	 * @param $otuHostName
     */
    public function sendTestAlarm($alarmType, $otuSerialNumber, $otuHostName)
    {        
        //build the alarm
        $wsAlarm = new SMTWsAlarmDto();
        $this->buildBaseTestAlarm($wsAlarm, $alarmType, $otuSerialNumber, $otuHostName);
        
        if ( $alarmType === SMTOtuApi::RES_OPT )
        {            
            $wsAlarm->setAlarmSpecificProblem(SMTAlarmSpecificProblemCode::decodeOpticalAlarmSpecificProblem( SMTAlarmSpecificProblemCode::OTU_OPT_ALARM_LINK_BREAK_CODE, FALSE ) );
           	$wsAlarm->setAlarmResource(self::PORT_STRING.'=1');
            
            //attenuation
            $attenuation = new SMTWsOpticalAlarmAttenuationDto();
            $wsAlarm->setAttenuation($attenuation);
            $attenuation->setLinkName('Link test 1');
            //level invalid fiber cut
            //$attenuation->setLevelDb( $opticalAlarm->getCurrentAlarmEvent()->getLevelDb() );
           	$attenuation->setDistanceKm( 35.782 );
           	$attenuation->setGpsLong( -121.886327 );
           	$attenuation->setGpsLat( 37.338166 );
        }
        else
        {
            $wsAlarm->setAlarmSpecificProblem(SMTAlarmSpecificProblemCode::decodeSystemAlarmSpecificProblem( SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_MEASUREMENT_CYCLE_CODE, FALSE ) );
            $wsAlarm->setAlarmResource("otu");
        }
        
        //send the alarm
        $http = new SMTHttpPostJson();
        $http->setServerUrl($this->webServiceUrl);
        $http->postJsonData($wsAlarm);
    }
    
	/**
	 *
	 * Retrieve if webservice media is enabled and which type of alarm to send and the web service url.
	 *
	 * @return String which type of alarm to send
	 */
	private function isAlarmToSendbyWebService()
	{
	    $alarmToSendByWebservice = SMTOtuApi::RES_NONE;

	    $mediaConfig = $this->sendReceive( SMTOtuApi::getWebServiceCommand() );
	    list($webServiceUrl, $alarmType) = explode(",", $mediaConfig );
	    $webServiceUrl = trim($webServiceUrl);
	    $alarmType = trim($alarmType);
	    
	    if ( self::isWebServiceConfigValid( $webServiceUrl, $alarmType ) )
	    {
	        $alarmToSendByWebservice = $alarmType;
	        $this->setWebServiceUrl( $webServiceUrl);
	    }
	    else
	    {
	        $alarmToSendByWebservice = SMTOtuApi::RES_NONE;
	    }
	    return $alarmToSendByWebservice;
	}
	
	/**
	 * Build and post attenuation optical alarm
	 * 
	 * @param SMTAttenuationOpticalAlarmDto $opticalAlarm
	 */
	private function sendAttenuationOpticalAlarm( SMTAttenuationOpticalAlarmDto $opticalAlarm )
	{
	    $wsAlarm = $this->convertAttenuationOpticalAlarm( $opticalAlarm );
	    $http = new SMTHttpPostJson();
	    $http->setServerUrl($this->webServiceUrl);
	    $http->postJsonData($wsAlarm);
	}

	/**
	 * Build and post system alarm
	 * 
	 * @param SMTSystemAlarmDto $systemAlarm
	 */
	private function sendSystemAlarm( SMTSystemAlarmDto $systemAlarm )
	{
	    $wsAlarm = $this->convertSystemAlarm( $systemAlarm );
	    $http = new SMTHttpPostJson();
	    $http->setServerUrl($this->webServiceUrl);
	    $http->postJsonData($wsAlarm);	     
	}	
	
	/**
	 * Fill alarm common properties
	 * 
	 * @param $alarm     SMTAlarmDto
	 * @param $baseAlarm SMTWsAlarmDto 
	 */
	private function buildBaseAlarm( SMTAlarmDto $alarm, SMTWsAlarmDto $baseAlarm )
	{
	    $hostName = '';
	    $otuSerialNumber = '-1';
	     
	    try
	    {
	    	$otu = new SMTOtu();
	    	$otu->setContext( $this->getContext() );
	    	$otuSerialNumber = $otu->getOtuSerialNumber();
	    }
	    catch ( \Exception $e )
	    {
	    	SMTLogger::getInstance()->traceException($e);
	    }
	    try
	    {
	    	$hostName = $this->sendReceive( SMTOtuApi::CMD_get_host_name );
	    }
	    catch ( \Exception $e )
	    {
	    	$this->handleException($e);
	    }

	    $baseAlarm->setAlarmId($alarm->getId(), $otuSerialNumber);
	    $baseAlarm->setOtuSerialNumber("$otuSerialNumber");	    
	    if ( $alarm->isCleared() )
	    {
	        $baseAlarm->setAlarmSeverity( SMTOtuAlarmSeverity::OTU_SEVERITY_CLEAR );
	    }
	    else
	    {
	        $baseAlarm->setAlarmSeverity( SMTAlarmSeverity::fromSmartOtuSeverityToOtuSeverity( $alarm->getSeverity() ) );
	    }
	    $baseAlarm->setAlarmTimeStamp( $alarm->getLastUpdate() );
	    $baseAlarm->setOtuName($hostName);
	}
	
	
	/**
	 * Build attenuation optical alarm
	 * 
	 * @param $opticalAlarm SMTAttenuationOpticalAlarmDto
	 * @return SMTWsOpticalAlarmAttenuationDto
	 */
	private function convertAttenuationOpticalAlarm( SMTAttenuationOpticalAlarmDto $opticalAlarm )
	{	    
	    $linkName = '';
	    
	    $wsAlarm = new SMTWsAlarmDto();
	    $wsAlarm->setAlarmType(self::OPTIC_STRING);
        $this->buildBaseAlarm( $opticalAlarm, $wsAlarm );
        
        $wsAlarm->setAlarmSpecificProblem(SMTAlarmSpecificProblemCode::decodeOpticalAlarmSpecificProblem( $opticalAlarm->getSpecificProblemCode(), $opticalAlarm->isCleared() ) );
        $linkTestHelper = new SMTLinkTest();
        $linkTestHelper->setContext( $this->getContext() );
        try 
        {
            $linkTestDto = $linkTestHelper->fetchLinkInfoFromTestAndFromCache( $opticalAlarm->getTestId() );
            $alarmResource = self::PORT_STRING.'='.$linkTestDto->getPortNumber();
            $linkName = $linkTestDto->getName();
            $wsAlarm->setAlarmResource($alarmResource);
        }
        catch(\Exception $e)
        {
            SMTLogger::getInstance()->traceException($e);
        }
        
        //attenuation
        $attenuation = new SMTWsOpticalAlarmAttenuationDto();
        $wsAlarm->setAttenuation($attenuation);
        $attenuation->setLinkName($linkName);
        if ( $opticalAlarm->getCurrentAlarmEvent() != NULL )
        {
	        if ( $opticalAlarm->getCurrentAlarmEvent()->isLevelValid() )
	        {
	            $attenuation->setLevelDb( $opticalAlarm->getCurrentAlarmEvent()->getLevelDb() );
	        }
	        if ( $opticalAlarm->getCurrentAlarmEvent()->isDistanceValid() )
	        {
	            $distanceKmFormated = number_format( ($opticalAlarm->getCurrentAlarmEvent()->getDistanceM()) / 1000.0 ,3,'.','');
	        	$attenuation->setDistanceKm( $distanceKmFormated );
	        }
	        if ( $opticalAlarm->getCurrentAlarmEvent()->isGpsValid() )
	        {
	        	$attenuation->setGpsLong( $opticalAlarm->getCurrentAlarmEvent()->getGpsX() );
	        	$attenuation->setGpsLat( $opticalAlarm->getCurrentAlarmEvent()->getGpsY() );
	        }
        }
        
        return $wsAlarm;
	}
	
	/**
	 * Build system alarm
	 * 
	 * @param $systemAlarm SMTSystemAlarmDto
	 * @return SMTWsAlarmDto
	 */
	private function convertSystemAlarm( SMTSystemAlarmDto $systemAlarm )
	{
	    $wsAlarm = new SMTWsAlarmDto();
	    $wsAlarm->setAlarmType(self::SYSTEM_STRING);
	    $this->buildBaseAlarm( $systemAlarm, $wsAlarm);
	    $wsAlarm->setAlarmSpecificProblem(SMTAlarmSpecificProblemCode::decodeSystemAlarmSpecificProblem( $systemAlarm->getSpecificProblemCode(), $systemAlarm->isCleared() ) );
	    $wsAlarm->setAlarmResource($systemAlarm->getAdditionalText());	    
	    
	    return $wsAlarm;			
	}
	
	/**
	 * Send alarm by web service if webservice is enabled
	 * 
	 * @param SMTContext $context
	 * @param SMTAlarmDto $alarm
	 */
	public static function sendAlarmByWebService( SMTContext $context, SMTAlarmDto $alarm )
	{
	    $ws = new SMTWebService();
	    $ws->setContext($context);
	    $alarmToSend = $ws->isAlarmToSendbyWebService();
	    
	    if ( $alarmToSend !== SMTOtuApi::RES_NONE )
	    {
	    	if ( $alarm instanceof SMTAttenuationOpticalAlarmDto)
    	    {
    	        if ( ($alarmToSend === SMTOtuApi::RES_OPT) || ($alarmToSend === SMTOtuApi::RES_BOTH))
    	        {
   	        		$ws->sendAttenuationOpticalAlarm( $alarm );
    	        }    	         
    	    }
    	    else if ( $alarm instanceof SMTSystemAlarmDto )
    	    {
    	        if ( ($alarmToSend === SMTOtuApi::RES_SYST) || ($alarmToSend === SMTOtuApi::RES_BOTH))
    	        {
    	        	$ws->sendSystemAlarm( $alarm );
    	        }
    	    }
	    }
	}
	
	/**
	 * Test if web service url is valid
	 *
	 * @param array $webServiceArray the webservice configuration array to check
	 *
	 * returns true if the web service url is valid
	 */
	private static function isWebServiceConfigValid( $webServiceUrl, $alarmType )
	{	    	   
		$isWebServiceConfigValid = FALSE;
		if ( ($webServiceUrl != "") && 
		     ($alarmType != SMTOtuApi::RES_NONE) )
		{
		    if ( (substr( $webServiceUrl, 0, 7 ) === "http://") || (substr( $webServiceUrl, 0, 8 ) === "https://") )
		    {
			    $isWebServiceConfigValid = TRUE;
			    SMTLogger::getInstance()->trace( sprintf("web service url: %s, alarm type: %s", $webServiceUrl, $alarmType), SMTLogger::PINFO);
		    }
		    else 
		    {
		        SMTLogger::getInstance()->trace( sprintf("Invalid web service url: %s", $webServiceUrl), SMTLogger::ERROR);
		    }
		}
		return $isWebServiceConfigValid;
	}
	

	/**
	 * Fill test alarm common properties
	 *
	 * @param $baseAlarm SMTWsAlarmDto
	 * @param $alarmType string ( OPT / SYST )
	 * @param $otuSerialNumber
	 * @param $otuHostName
	 */
	private function buildBaseTestAlarm( SMTWsAlarmDto $baseAlarm, $alarmType, $otuSerialNumber, $otuHostName )
	{	
		if ( $alarmType == SMTOtuApi::RES_OPT )
	    {
	        $baseAlarm->setAlarmType(self::OPTIC_STRING);
		    $baseAlarm->setAlarmId(1, $otuSerialNumber);
	    }
	    else
	    {
	        $baseAlarm->setAlarmType(self::SYSTEM_STRING);
	        $baseAlarm->setAlarmId(2, $otuSerialNumber);
	    }
		$baseAlarm->setOtuSerialNumber("$otuSerialNumber");
		$baseAlarm->setAlarmSeverity( SMTOtuAlarmSeverity::OTU_SEVERITY_CRITICAL );

		try 
		{
		    //retrieve database connection
		    $dbConnection = $this->getContext()->getDatabase();
		    $timestamp = SMTUtil::getTimestampUTC( time(), $dbConnection );
		    $baseAlarm->setAlarmTimeStamp( $timestamp );
		    $dbConnection->close();
		} 
		catch (Exception $e) 
		{
		    SMTLogger::getInstance()->traceException($e);
		}

		$baseAlarm->setOtuName($otuHostName);
	}
}
?>