<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\serviceshelper\activity;

use app\services\monitoring\SMTTestAcquisitionParametersDto;

use app\events\linktest\SMTLinkTestUpdateEventDto;

use app\events\SMTEventMessageManager;

use app\sharedmemory\SMTMemoryManager;

use app\util\SMTIOException;

use app\services\activity\SMTActivityType;

use app\serviceshelper\monitoring\cache\SMTOperationManager;

use app\services\monitoring\SMTLinkTestDto;

use app\services\activity\SMTActivityState;

use app\serviceshelper\SMTServiceHelper;

use app\serviceshelper\monitoring\SMTLinkTest;

use app\services\activity\SMTActivityDto;

use app\services\activity\otu\SMTOtuActivityDto;

use app\util\SMTLogger;


/**
 * Create the activity DTO sent to listening clients
 *
 * @author Sylvain Desplat
*/
class SMTActivity extends SMTServiceHelper
{
    const LOCK_ACTIVITY_FILE = "/../../../tmp/activity.lock";
    
    /**
     * Temporary activity file name including its path. Used to store last activity
     * No need to worry about temporary activity file version because it expires after 60s 
     * and because it is stored in a directory deleted during upgrade
     * 
     * @var unknown
     */
    const SERIALIZED_ACTIVITY_FILE = "/../../../tmp/activity.tmp";
    
    /**
     * Temporary activity file timeout
     * 
     * @var number 120s
     */
    const ACTIVITY_FILE_EXPIRE_DURATION = 120;
    
    /**
     * Activity file exclusive lock
     *
     * @var string
     */
    private static $activityLockFile = NULL;
    
    /**
     * Create an activity DTO from the otu activity.
     * Critical section to process activities one by one
     *
     * @param SMTOtuActivityDto $otuActivity
     * @param boolean $forceFullActivityRetrieval (FALSE by default) Use to force the retrieval of test parameters 
     * @return \app\services\activity\SMTActivityDto The created activity
     */
    public function processActivity( SMTOtuActivityDto $otuActivity, $forceFullActivityRetrieval = FALSE )
    {    
        $activityDto = NULL;
    	if ( self::acquireActivityLock() )
    	{
    	    try 
    	    {
        	    $activityDto = $this->createActivityDto( $otuActivity, $forceFullActivityRetrieval );
        	    if ( $activityDto != NULL )
        	    {	
        	    	// If processing fails, catch the exception to send the activity event anyway.
        	    	try
        	    	{        	    	
        	    	    $this->processLinkTestUpdateEvent( $otuActivity );
        	    	}
    	    	    catch( \Exception $e )
    	    	    {
    	    	    	$this->getContext()->getLogger()->traceException($e);
    	    	    }        	    	   
        	    }
        	    self::releaseActivityLock();
    	    }
    	    catch( \Exception $e)
    	    {
    	    	self::releaseActivityLock();
    	    	throw $e;
    	    }        	    
    	}
    	else
    	{
    		$this->getContext()->getLogger()->trace("Couldn't acquire lock to process activity: ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
    	}
    	
    	return $activityDto;
    }
    
    /**
     * Process link-test update events stored in APC memory (link and test deferred delete or update on OTU) and send them to clients stations.
     * @param SMTOtuActivityDto $otuActivity
     */
    private function processLinkTestUpdateEvent( SMTOtuActivityDto $otuActivity )
    {
        //test if for that new sequencing, no operation on link were processed (delete operation) 
        if ( SMTActivityState::isInitializationState( $otuActivity->getState() ) )
        {
            $events = SMTMemoryManager::fetchAll( SMTLinkTestUpdateEventDto::getClass() );            
            
            if ( $events != NULL )
            {
                SMTMemoryManager::deleteAll( SMTLinkTestUpdateEventDto::getClass() );
                
                if ( is_object( $events ) && $events instanceof SMTLinkTestUpdateEventDto )
                {
                    //notify link-test event
                    SMTEventMessageManager::getInstance()->sendLinkTestUpdateEvent( $events );

                    //if Link-Test changes pending are applied, force the cleanup of the test acquisition parameters
                    SMTMemoryManager::delete( SMTTestAcquisitionParametersDto::getClass(), $events->getLinkTest()->getTestId() );                
                    SMTMemoryManager::delete( SMTLinkTestDto::getClass(), $events->getLinkTest()->getId() );
                }
                else if ( is_array( $events ) )
                {
                    foreach ( $events as $event)
                    {
                    	//notify link-test event
                    	SMTEventMessageManager::getInstance()->sendLinkTestUpdateEvent( $event );           

                    	//if Link-Test changes pending are applied, force the cleanup of the test acquisition parameters
                    	SMTMemoryManager::delete( SMTTestAcquisitionParametersDto::getClass(), $event->getLinkTest()->getTestId() );
                    	SMTMemoryManager::delete( SMTLinkTestDto::getClass(), $event->getLinkTest()->getId() );
                    }                    
                }
            }
        }
    }
    
    /**
     * Create the activity dto to send to client from the given otu activity
     * Retrieve monitoring test information only if we are at the initialization phase of the test.
     *  
     * @param SMTOtuActivityDto $otuActivity
     * @param boolean $forceFullActivityRetrieval (FALSE by default) Use to force the retrieval of test parameters
     * 
     * @return \app\services\activity\SMTActivityDto
     */
    public function createActivityDto( SMTOtuActivityDto $otuActivity, $forceFullActivityRetrieval = FALSE )
    {        
        $activity = new SMTActivityDto();
        $testId = ( $otuActivity->isTestIdValid() )? $otuActivity->getTestId() : NULL;
        $activity->setTestId( $testId );        
        $activity->setState( $otuActivity->getDecodedState() );
        $activity->setType( $otuActivity->getDecodedType() );                
        
        //Retrieve additional information for measurement and monitoring test activity
        if ( SMTActivityType::isMonitoringOrTestOnDemandOtuCode( $otuActivity->getType() ) ||
             SMTActivityType::isMeasurementActivityOtuCode( $otuActivity->getType() )    )
        {
            // If test parameters retrieval for activity fails, catch the exception to send the activity event anyway.
            try 
            {
                $this->handleMeasurementAndTestsActivityIfNeeded( $otuActivity, $testId, $activity, $forceFullActivityRetrieval );
            }
            catch( \Exception $e )
            {
            	$this->getContext()->getLogger()->traceException($e);
            }            
        } 
        
        return $activity;
    }
    
    /**
     * Retrieve additional information for measurement and monitoring test activity
     * 
     * @param SMTOtuActivityDto $otuActivity
     * @param integer $testId
     * @param SMTActivityDto $activity
     * @param boolean $forceFullActivityRetrieval (FALSE by default) Use to force the retrieval of test parameters
     * 
     * @return SMTActivityDto
     */
    private function handleMeasurementAndTestsActivityIfNeeded( SMTOtuActivityDto $otuActivity, $testId, SMTActivityDto &$activity, $forceFullActivityRetrieval = FALSE )
    {        
        //only retrieve monitoring test parameters and link info at initialization phase or when we force their retrieval
        if ( $testId !== NULL )
        {
            if ( SMTActivityState::isInitializationState( $otuActivity->getState() ) || $forceFullActivityRetrieval )
            {
                $linkTestHelper = new SMTLinkTest();
                $linkTestHelper->setContext( $this->getContext() );
                $testDetailedDto = $linkTestHelper->fetchTestAcquisitionParameters( $testId );
                $activity->setAcquisitionParameter( $testDetailedDto );
                
                //retrieve link info
                $linkId = $testDetailedDto->getLinkId();
                if ( $linkId !== NULL )
                {
                    $activity->setLinkId($linkId);                
                    $temporarylinkTestDto = $linkTestHelper->fetchLinkInfoFromCache( $linkId );                
                	$activity->setLinkName( $temporarylinkTestDto->getName() );
                	$activity->setPortNumber( $temporarylinkTestDto->getPortNumber() );
                }
//                 $otuActivity->save();
            }
//             //Cache activity in "acquisition" state => used to detect acquisition errors with switch failure
//             else if ( SMTActivityState::isAcquisitionState( $otuActivity->getState() ) )
//             {
//             	$otuActivity->save();
//             }
            //update last monitoring time stamp
            //(check the test when it is "finalizing"; never test "completed" because even a test not started has a COMPLETED state)
            else if ( SMTActivityState::isFinalizationState( $otuActivity->getState() ) )
            {
            	$activity->setLastMonitoringTimeStamp( $otuActivity->getTimestampUTC( $this->getContext()->getDatabase() ) );
            }            

//             else if ( SMTActivityState::isCompletedState( $otuActivity->getState() ) )
//             {
//                 //handles OTU acquisition failure (switch failure)                
//                 $lastOtuAcquisitionActivity = SMTMemoryManager::fetch( SMTOtuActivityDto::getClass(), SMTOtuActivityDto::OTU_LAST_ACTIVITY_ID );
//                 if ( $lastOtuAcquisitionActivity != NULL && !$lastOtuAcquisitionActivity->isObsolete() )
//                 {
//                     if ( ( $lastOtuAcquisitionActivity->getTestId() == $testId && !SMTActivityState::isAcquisitionState( $lastOtuAcquisitionActivity->getState() ) ) )
//                     {
//                      //OTU acquisition failure if no acquisition was done
//             	        $activity->setState( SMTActivityState::ERROR );
//                     }
//                 }            	
//             }            
        }
        else if ( $otuActivity->isMeasurement() )
        {
            $measurementInfo = SMTOperationManager::getCurrentMeasureInfo();
            if ( $measurementInfo != NULL )
            {
                //retrieve link info
                $linkId = $measurementInfo->getLinkId();
                if ( $linkId !== NULL )
                {
                    $activity->setLinkId($linkId);
                    //retrieve link info              
                    $linkTest = new SMTLinkTest();
                    $linkTest->setContext( $this->getContext() );
                    $temporarylinkTestDto = $linkTest->fetchLinkInfoFromCache( $linkId );
                    $activity->setLinkName( $temporarylinkTestDto->getName() );
                }
                $activity->setPortNumber( $measurementInfo->getPortId() );
            }
        }        
        
        return $activity;
    }
    
    /**
     * Whether the given test is in analysis status.
     * 
     * @param integer $testId
     * @return boolean
     */
    public static function isActivityStatusAnalysis( $testId )
    {
        try
        {
        	$activity_data = SMTActivity::unSerializeActivity();
        
        	if ( $activity_data != FALSE )
        	{
        		$activityOtuDto = SMTOtuActivityDto::getInstance( $activity_data );
        		$activityTestId = ( $activityOtuDto->isTestIdValid() )? $activityOtuDto->getTestId() : NULL;
        		 
        		return  ( $testId == $activityTestId &&
        				  SMTActivityType::isMonitoringOrTestOnDemandOtuCode( $activityOtuDto->getType() ) &&
        				  SMTActivityState::isAnalysisState( $activityOtuDto->getState() ) );
        	}
        }
        catch( \Exception $e )
        {
        	SMTLogger::getInstance()->traceException($e);
        }
        return FALSE;
    }
    
    /**
     * Sleep until $timeout untill the activity is stopped
     * 
     * @param boolean $timeout
     */
    public static function sleepForEndOfActivityWithTimout( $timeout )
    {
        $activity_data = self::unSerializeActivity();
        $maxWaitLoopCount =  $timeout / SMTLinkTest::SLEEP_TIME_FOR_TEST_REMOVAL_S;
        if ( $activity_data != FALSE )
        {
        	$count = 0;
        	$activityOtuDto = SMTOtuActivityDto::getInstance( $activity_data );
        	while ( $activityOtuDto != NULL && !$activityOtuDto->isNotRunning() && $count++ < $maxWaitLoopCount )
        	{
        		sleep( SMTLinkTest::SLEEP_TIME_FOR_TEST_REMOVAL_S );
        		$activity_data = self::unSerializeActivity();
        		if ( $activity_data != FALSE )
        		{
        			$activityOtuDto = SMTOtuActivityDto::getInstance( $activity_data );
        		}
        		else
        		{
        		    $activityOtuDto = NULL;
        		}
        	}
        }
    }    
    
    /**
     * Serialize the activity string into a temporary file to be used later when a new web client retrieve the activity for the first time
     * No need to worry about temporary activity file version because it expires after 120s 
     * and because it is stored in a directory deleted during upgrade
     * 
     * @param array $activityData array of string with activity data
     * 
     * return mixed FALSE if it failed or the number of bytes written
     */
    public static function serializeActivity( array $activityData )
    {
        return file_put_contents ( __DIR__.self::SERIALIZED_ACTIVITY_FILE, serialize( $activityData ), LOCK_EX);
    }
    
    /**
     * Read the temporary activity file if it exists and if it is not older than 120s
     * No need to worry about temporary activity file version because it expires after 120s 
     * and because it is stored in a directory deleted during upgrade
     * 
     * @return array of string The activity read or FALSE if it is obsolete or if an error occurs
     */
    public static function unSerializeActivity()
    {
        $contents = FALSE;
        $fileName = __DIR__.self::SERIALIZED_ACTIVITY_FILE;
        
        if ( file_exists($fileName) && ( ( time() - filectime($fileName) ) < self::ACTIVITY_FILE_EXPIRE_DURATION ) )
        {
            //don't use lock to ensure data integrity/writing because the information is not critical: if it fails, returns FALSE
            if ( ( $contents = file_get_contents( $fileName ) ) != FALSE )
            {
                $contents = unserialize( $contents );
            }
        }
        return $contents;
    }
    
    private static function getLockActivityFile()
    {
    	if ( self::$activityLockFile == NULL )
    	{
    		$handle = fopen( __DIR__.self::LOCK_ACTIVITY_FILE, "w+");
    		 
    		self::$activityLockFile = ( $handle != FALSE)? $handle : NULL;
    	}
    	return self::$activityLockFile;
    }
    
    /**
     * Acquire activity exclusive lock
     *
     * @return boolean
     */
    private static function acquireActivityLock()
    {
    	$retries = 0;
    	$max_retries = 24;
    
    	//SMTLogger::getInstance()->trace( "acquireActivityLock" );
    
    	// keep trying to get a lock as long as possible ( max about 10s n(n+1)(2n+1)/6))
    	do
    	{
    		if ($retries > 0)
    		{
    			usleep( 2000 * ( $retries*$retries ) );
    		}
    		$retries += 1;
    	} while ( ( !flock(self::getLockActivityFile(), LOCK_EX, $eWouldBlock) || $eWouldBlock ) && $retries <= $max_retries);
    
    	return ($retries < $max_retries);
    }
    
    /**
     * Releases activity exclusive lock
     *
     * @return boolean
     */
    private static function releaseActivityLock()
    {
    	//SMTLogger::getInstance()->trace( "releaseAlarmLock" );
    	return isset(self::$activityLockFile) && self::$activityLockFile != NULL? flock( self::$activityLockFile, LOCK_UN ) : TRUE;
    }
}
?>