<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\services\autotest;

use app\util\SMTResultSetWithAutoIndexDto;

use app\util\SMTUtil;

use app\util\SMTInvalidValueException;

/**
 * Autotest read only dto (record is not updated after insert in database: don't extend SMTRwDto)
 * 
 * @author Sylvain Desplat
 * @SMTClassTableNameAnnotation(tablename='autotest', tablekey='id') 
 */
class SMTAutotestDto extends SMTResultSetWithAutoIndexDto
{    
    const STATUS_OK = "OK";
    const STATUS_KO = "KO";
    const MAX_AUTOTEST_AVAILABLE = 10;
    
    /**
     * Timestamp of autotest start time in seconds
     * 
     * @var number
     */
    protected $startTimestamp;
    
    /**
     * Autotest duration in seconds
     * 
     * @var number
     */
    protected $durationSec;
         
    /**
     * Autotest status
     *
     * @var string OK / KO
     */
    protected $status;
    
    /**
     * Autotest summary file name
     *
     * @var string
     */
    protected $autotestSummaryFileName;
    
    /**
     * Whether autotest file is available
     * 
     * @var boolean
     */
    protected $autotestSummaryFileAvailable = FALSE;
    
    /**
     * last sequencer date
     *
     * @var number
     */
    protected $sequencerLastRunTimestamp;

    const FETCH_ALL_AUTOTESTS= "SELECT * FROM autotest ORDER BY start_timestamp DESC";
    
    const DELETE_AUTOTESTS_OLDER_THAN_ONE_YEAR= "DELETE FROM autotest WHERE (start_timestamp + 31536000)< CAST(strftime('%s', 'now') as integer)";
    
    /**
     * Fetch all autotests
     *
     * @param \SQLite3 $dbConnection
     *
     * @return array SMTAutotestDto
     */
    public static function fetchAll( \SQLite3 $dbConnection )
    {
    	$query=self::FETCH_ALL_AUTOTESTS;
    	$statement = $dbConnection->prepare( $query);
    
    	self::handleStatementError( $dbConnection, $statement, "Prepare select", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    	$resultSet = $statement->execute();
    
    	self::handleStatementError( $dbConnection, $resultSet, "Select", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    
    	$autotests = parent::getListFromResultSet($resultSet, get_class() );
    	
    	//autotests summary file available
    	$index = 0;
    	foreach ( $autotests as $autotest)
    	{
    		if ( $index < self::MAX_AUTOTEST_AVAILABLE )
    		{
    			if ( ($autotest->autotestSummaryFileName != NULL) && (strlen( $autotest->autotestSummaryFileName) > 0 ) )
    			{
    				$autotest->setAutotestSummaryFileAvailable( TRUE );
    			}
    		}
    		$index++;
    	}
    	 
    	return $autotests;
    }    
    
    /**
     * Delete autotests older than 1 year
     *
     * @param \SQLite3 $dbConnection
     */
    public static function purgeOldAutotests( \SQLite3 $dbConnection )
    {
        $query=self::DELETE_AUTOTESTS_OLDER_THAN_ONE_YEAR;
        $statement = $dbConnection->prepare( $query);
        
        self::handleStatementError( $dbConnection, $statement, "Prepare delete", $query, NULL, __FILE__,__METHOD__,__LINE__ );
        $statementResult = $statement->execute();
        
        self::handleStatementError( $dbConnection, $statementResult, "Delete", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    }
    
    /**
     * Set the sequencer start Timestamp
     *
     * @param number $startTimestamp
     */
    public function setStartTimestamp( $startTimestamp )
    {
    	$this->startTimestamp = $startTimestamp;
    }
    
    /**
     * Set the autotest duration in seconds
     *
     * @param number $durationSec
     */
    public function setDurationSec( $durationSec )
    {
    	$this->durationSec = $durationSec;
    	
    	if ( SMTUtil::isInt( $durationSec ) && $durationSec >=0 )
    	{
    		$this->durationSec = intval( $durationSec );
    	}
    	else
    	{
    		throw new SMTInvalidValueException(SMTInvalidValueException::INVALID_VALUE, NULL, "durationSec", $durationSec);
    	}
    }
    
    /**
     * Set whether autotest status is OK
     *
     * @param boolean $isStatusOk
     */
    public function setStatusOK( $isStatusOk )
    {        
    	$this->status = $isStatusOk? self::STATUS_OK : self::STATUS_KO;
    }
    
    /**
     * Set autotest status
     *
     * @param string $status [OK KO]
     */
    public function setStatus( $status )
    {
    	$this->status = $status;
    }
    
    /**
     * Set the Last Run Timestamp of the sequencer
     *
     * @param number $sequencerLastRunTimestamp
     */
    public function setSequencerLastRunTimestamp( $sequencerLastRunTimestamp )
    {
    	$this->sequencerLastRunTimestamp = $sequencerLastRunTimestamp;
    }
    
    /**
     * Set autotest summary file name
     *
     * @param string $autotestSummaryFileName
     */
    public function setAutotestSummaryFileName( $autotestSummaryFileName)
    {
    	$this->autotestSummaryFileName= $autotestSummaryFileName;
    }
    
    /**
     * Set whether autotest summary file is available
     *
     * @param string $autotestSummaryFileAvailable
     */
    public function setAutotestSummaryFileAvailable( $autotestSummaryFileAvailable)
    {
    	$this->autotestSummaryFileAvailable = $autotestSummaryFileAvailable;
    }
        
    /**
     * Serialize as Json data the object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json array.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    } 
}
?>