<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\alarm\otu;

use app\util\SMTUtil;

use app\serviceshelper\alarm\SMTAlarmSeverity;

/**
 * Base OTU optical alarm Dto
 * 
 * @author Sylvain Desplat
 */
class SMTOtuOpticalAlarmDto extends SMTOtuAlarmDto
{    
    /**
     * Test identifier
     * @var integer
     */
    protected $test;

    /**
     * Detection file name 
     * @var string
     */
    protected $detection;
    
    
    /**
     * Localisation file name
     * @var string
     */
    protected $localisation1;

    /**
     * ShortAcq file name
     * @var string
     */
    protected $shortAcq;
    
    /**
     * Alarm code (cOtu_Alarm_Confirmee or cOtu_Alarm_Non_Confirmee)
     * @var boolean
     */
    protected $confirmed;
        
    /**
     * deviations for the optical alarm
     * @SMTAttributeInfosAnnotation(classname='app\services\alarm\otu\SMTOtuOpticalAlarmDeviationDto')
     * @var SMTOtuOpticalAlarmDeviationDto
     */
    protected $deviation = NULL;
    
    /**
     * deviations for the optical alarm
     * @SMTAttributeInfosAnnotation(classname='app\services\alarm\otu\SMTOtuOpticalAlarmAttenuationDto')
     * @var SMTOtuOpticalAlarmAttenuationDto
     */        
    protected $attenuation = NULL;
    
    /**
     * @SMTAttributeInfosAnnotation(classname='app\services\alarm\otu\SMTOtuOpticalAlarmPeakDto',islist='true')
     * @var array app\services\alarm\otu\SMTOtuOpticalAlarmPeakDto
     */
    protected $peaks = array();
    
    /**
     * Test on demand OTU operation id 
     * @integer
     */
    protected $tSDOperationId;
    
    /**
     * Test identifier
     * 
     * @return integer
     */
    public function getTestId()
    {
    	return SMTUtil::convertToInt( $this->test );
    }    
   
    /**
     * Test identifier
     *
     * @return integer
     */
    public function getTestOnDemandOperationId()
    {
    	return ( isset( $this->tSDOperationId) && SMTUtil::isInt( $this->tSDOperationId ) )? $this->tSDOperationId : -1;
    }
    
    /**
     * Get detection file name
     * @return string
     */
    public function getDetectionFileName()
    {
    	return $this->detection;
    }    
    
    /**
     * Get localisation file name
     * @return string
     */
    public function getLocalisationFileName()
    {
    	return $this->localisation1;
    }    
    
    /**
     * Get ShortAcq file name
     * @return string
     */
    public function getShortAcqFileName()
    {
    	return $this->shortAcq;
    }
    
    /**
     * Is optical alarm confirmed
     * @return boolean
     */
    public function isConfirmed()
    {
        return filter_var( $this->confirmed, FILTER_VALIDATE_BOOLEAN );
    }           
    
    /**
     * Get optical alarm deviations
     * @return SMTOtuOpticalAlarmDeviationDto
     */
    public function getDeviation()
    {
    	return $this->deviation;
    }
    
    /**
     * Get optical alarm deviations
     * @return SMTOtuOpticalAlarmAttenuationDto
     */
    public function getAttenuation()
    {
    	return $this->attenuation;
    }
    
    /**
     * @return array
     */
    public function getPeaks()
    {
    	return $this->peaks;
    }

    /**
     * Whether the Otu attenuation severity is valid
     * If the severity given in attenuation is different than alarm severity and 
     * if the severity in attenuation is cleared or info or unknown,
     * attenuation severity is not valid
     *
     * @return boolean
     */
    public function isValidAttenuationSeverity()
    {
        $valid = TRUE;
        if ($this->getAttenuation() != null )
        {
	    	// if the severity given in attenuation is different than alarm severity and
	    	// if the severity in attenuation is cleared or info or unknown, attenuation severity is not valid
	    	$attenuationSeverity = $this->getAttenuation()->getSeverityConvertedToSmartOtu();
	    	$alarmSeverity = SMTAlarmSeverity::fromOtuSeverityToSmartOtuSeverity( $this->severity );
	    	if ( $alarmSeverity != $attenuationSeverity &&
	    			!SMTAlarmSeverity::isAlarmSeverity( $attenuationSeverity ) )
	    	{
	    		$valid = FALSE;
	    	}
        }
        else
        {
        	$valid = FALSE;
        }
    
    	return $valid;
    }
    
    /**
     * Whether the Otu peak severity is valid
     * if the severity in peak is info or unknown,
     * peak severity is not valid
     *
     * @return boolean
     */
    public function isValidPeakSeverity(SMTOtuOpticalAlarmPeakDto $peakAlarmDto)
    {
    	$valid = TRUE;
    	// if the severity in peak is info or unknown, peak severity is not valid
    	$peakSeverity = $peakAlarmDto->getSeverityConvertedToSmartOtu();
    	if (($peakSeverity==SMTAlarmSeverity::SEVERITY_INFO)||($peakSeverity==SMTAlarmSeverity::SEVERITY_UNKNOWN))
    	{
    		$valid=FALSE;
    	}
    	return $valid;
    }
    
    /**
     * Otu severity
     * 
     * If the severity given in attenuation is different than alarm severity and 
     * if the severity in attenuation is cleared or info or unknown,
     * use alarm severity given in alarm header instead of the attenuation severity
     * 
     * Tip: not usefull, we could simply return: $this->severity
     * 
     * @return string
     */
    public function getOtuSeverity()
    {
        // if the severity given in attenuation is different than alarm severity and 
        // if the severity in attenuation is cleared or info or unknown,
        // use alarm severity given in alarm header instead of the attenuation severity
    	if ( $this->isValidAttenuationSeverity() && $this->getAttenuation() != null)
        {
            //not usefull, we could return: $this->severity
            $severity = $this->getAttenuation()->getOtuSeverity();
        }
        else
        {
            $severity = $this->severity;
        }        
        
    	return $severity;
    }
    
    /**
     * SmartOtu Severity
     *
     * @return integer
     */
    public function getSeverityConvertedToSmartOtu()
    {
        $severity = $this->getOtuSeverity();
        
    	return SMTAlarmSeverity::fromOtuSeverityToSmartOtuSeverity( $severity );
    }    
    
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }
    
    /**
     * Create the Dto from a Json Dto in an array
     *
     * @param array $json_array json data in an array
     *
     * @return \app\services\alarm\otu\SMTOtuOpticalAlarmDto
     */
    static function getInstance( array &$json_array )
    {
    	return self::forge( get_class(), $json_array, FALSE );
    }
}
?>