<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\admin;

use app\serviceshelper\maintenance\SMTLicence;

use app\serviceshelper\ethernet\SMTEthernet;

use app\util\SMTLogger;
use app\util\SMTUtil;
use app\http\SMTContext;


class SMTUserUtil
{   
    const INSTALL_ETH_PASSWD_SHA_256 = "bf8000ed022ff815da9fa6733840fbe70be3b338bc5a1ae2154a59cc4d128925";
    const SYS_USER = "cm9vdA==";
    const INSTALL_IP_USER = "aW5zdGFsbGV0aA==";
    const DEFAULT_INSTALL_USER = "YWRtaW4=";
//     const DEFAULT_PASSWD = "app/admin/.passwd";
    const DEFAULT_INSTALL_PASSWORD_SHA_256 = "5e884898da28047151d0e56f8dc6292773603d0d6aabbdd62a11ef721d1542d8";
    const PASSWD = "/otu/smartOTU/.passwd";
    const USER_ALIASES_FILE = "/etc/aliases";
    const SMARTOTU_USER = "smartotu";
    const LOGIN_TOKEN_FILE = "/otu/smartOTU/login_tokens";
    
    /**
     * Authenticate the user from a password file and save it in the context
     *
     * @param app\http\SMTContext $context
     * @param string $login
     * @param string $password_sha_256 password hashed in SHA_256
     * @param string $password_encoded pasword encoded base64 + fake characters
     *
     * @throws SMTAuthenticationException it user not authenticated.
     * @return SMTUserDto or nul if not found
     */
    public static function authenticate( SMTContext $context, $login, $password_sha_256, $password_encoded )
    { 
        $user = NULL;
        
        $password = self::decodePassword($password_encoded);
        if ( $password === FALSE )
        {
            SMTLogger::getInstance()->trace( sprintf( "Invalid password encoded: %s", $password_encoded ), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        }
        else
        {
            //SMTLogger::getInstance()->trace( sprintf( "password_encoded %s, password decoded: %s", $password_encoded, $password ) );
        }
        
        $userRoleName = self::checkLogin($context, $login, $password_sha_256, $password);
        if ( $userRoleName != NULL )
        {
    	    $user = SMTUserDto::forgeUser( $login, $userRoleName );
        }
        
        $context->setUser($user);
        
        if ( $user == NULL )
        {
            throw new SMTAuthenticationException();
        }
    	
    	return $user;
    }
    
    /**
     * Returns the lines of password file in an array.
     * 
     * @return array of lines of password file
     */
    private static function readOldPasswordFile()
    {
//         if ( !file_exists(self::PASSWD) )
//         {
//             if ( !copy(self::DEFAULT_PASSWD, self::PASSWD) )
//             {
//                 SMTLogger::getInstance()->trace( sprintf( "Failed to copy default password file to: %s ", self::DEFAULT_PASSWD ), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
//                 //force use of default password file to allow user to login
//                 $lines = file(self::DEFAULT_PASSWD);
//             }
//             else
//             {
//             	$lines = file(self::PASSWD);
//             }
//         }
//         else
//         {
            $lines = file(self::PASSWD);
//         }
        return $lines;
    }
    
    /**
     * Returns the user role name if the user is found or null if the user is not found.
     * If the OTU is in local mode, allow to login without checking credentials.
     * In local mode, use normal user credentials, and promote user with install_ip role
     *
     * @param SMTContext $context
     * @param string $login Login is passed by reference: it can be changed for a login as super user or a login in local mode.
     * @param string $password_sha_256 password hashed in SHA_256
     * @param string password
     *
     * @return string the user role name
     */
    private static function checkLogin(SMTContext $context, &$login, $password_sha_256, $password)
    {
    	$userRole = NULL;
    	$highSecurityMode = self::isEnhancedSecurityOption( $context );
    	$localMode = self::isLocalModeForCurrentSession( $context );

    	// in a new install; smartotu user doesn't exist
    	self::checkSmartOtuUnixUserExist();
    	
		//trac #2922 super user, allowed in local mode or when high security is disabled
		if ( ( ($highSecurityMode == FALSE) || $localMode) &&  
		     (strcmp( base64_encode( $login ), self::SYS_USER ) == 0) && self::checkRootUnixPassword($password) )
		{
			$userRole = SMTRoleDto::SUPER_USER;
			SMTLogger::getInstance()->trace("checkLogin >> Login Super User", SMTLogger::PINFO );
			
			//with super user, always use normal user read from SmartOTU password file (we can change normal user login/password even if we have lost them)
			$login = self::isOldPasswordFile()? self::getOldInstallUserFromFile() : self::getSmartOtuUnixLogin();
			SMTLogger::getInstance()->trace("checkLogin >> Super user login with User: $login; Role: $userRole", SMTLogger::PINFO );
		}
		
		//trac #2922 installeth user, allowed when high security is disabled
		else if ( ($highSecurityMode == FALSE) && (self::INSTALL_ETH_PASSWD_SHA_256 === $password_sha_256) && 
		          (strcmp( base64_encode( $login ), self::INSTALL_IP_USER ) == 0)  )
		{
			$userRole = SMTRoleDto::DEBUG;
			SMTLogger::getInstance()->trace("checkLogin >> Login Install eth", SMTLogger::PINFO );
			
			//with installeth user, always use normal user read from SmartOTU password file (we can change normal user login/password even if we have lost them)
			$login = self::isOldPasswordFile()? self::getOldInstallUserFromFile() : self::getSmartOtuUnixLogin();
			SMTLogger::getInstance()->trace("checkLogin >> Installeth user login with User: $login; Role: $userRole", SMTLogger::PINFO );
		}
		
		//normal user login
		else
		{
		    // if old password file, migration needed
		    if ( self::isOldPasswordFile() )
		    {
		    	$success = self::checkOldLogin($login, $password, $password_sha_256, $userRole);
		    
		    	if ($success)
		    	{
		    		//truncate and delete file to avoid a new migration
		    		$fp = fopen(self::PASSWD, "w");
		    		fclose($fp);
		    		unlink(self::PASSWD);
		    	}
		    }
		    
		    //check normal user login
		    if (self::checkSmartOtuUnixLogin($login))
		    {
		    	//check normal user password
		    	if ( self::checkSmartOtuUnixPassword($password) )
		    	{		    		
		    		$userRole = SMTRoleDto::INSTALL;
		    		SMTLogger::getInstance()->trace("checkLogin >> Login User: $login Role: $userRole", SMTLogger::PINFO);
		    	}
		    	// invalid password given
		    	else
		    	{
		    		SMTLogger::getInstance()->trace("checkLogin >> Invalid password for user: $login Role: $userRole", SMTLogger::ERROR);
		    		$userRole = NULL;
		    	}
		    }
		    else
		    {
		    	SMTLogger::getInstance()->trace("checkLogin >> Invalid login for user: $login Role: $userRole", SMTLogger::ERROR);
		    	$userRole = NULL;
		    }
		}
				
		// #trac 2922  En mode local, on doit toujours se logger comme en mode normal
		if ( $localMode && ($userRole != NULL) )
		{
			//in local mode, promote user with INSTALL_IP role to change ip settings
			if ( ($userRole != SMTRoleDto::SUPER_USER) && ($userRole != SMTRoleDto::DEBUG) )
			{
				$userRole = SMTRoleDto::INSTALL_IP;
			}
		
			//in local mode, always use normal user read from SmartOTU password file (in local mode, we can change normal user login/password even if we have lost them)
			$login = self::isOldPasswordFile()? self::getOldInstallUserFromFile() : self::getSmartOtuUnixLogin();
		
			SMTLogger::getInstance()->trace("checkLogin >> Local mode login with User: $login; Role: $userRole", SMTLogger::PINFO );
		}
		//in ONMSi mode or SmartOEM mode, promote user with INSTALL_IP role to change ip settings
		else if ( ($userRole != NULL) && (SMTOtuMode::isONMSi() || SMTOtuMode::isSmartOem()))
		{
			if ( ($userRole != SMTRoleDto::SUPER_USER) && ($userRole != SMTRoleDto::DEBUG) )
			{
				$userRole = SMTRoleDto::INSTALL_IP;
				SMTLogger::getInstance()->trace("checkLogin >> promote User $login with role $userRole", SMTLogger::INFO );
			}
		}

		return $userRole;
	}
    
	/**
	 * Check user login according to the old password file otu/smartotu/.passwd file
	 *
	 * @param string $remote_login
	 * @param string $remote_password
	 * @param string $remote_group
	 */
	private static function checkOldLogin($login, $password, $password_sha_256, &$userRole)
	{
		$success = FALSE;
		$userFound = FALSE;
	
		//la migration n'a pas ete faite
		if ( self::isOldPasswordFile() )
		{	
			$lines = file(self::PASSWD);
        	foreach ($lines as $line)
        	{
        	    $line = trim($line);
        		if ( $line != "" && (strpos( $line, ":" ) !== FALSE) ) //remove empty lines
        		{
        			list($oldfileLogin, $oldFilePassword, $fileRole) = explode(":",$line);        			
        			$oldfileLogin = trim($oldfileLogin);
        			$oldFilePassword = trim($oldFilePassword);
        			$fileRole = trim($fileRole);
	
				    //SMTLogger::getInstance()->trace("checkOldLogin >> oldfileLogin: $oldfileLogin, old_file_password: $oldFilePassword, password_sha_256: $password_sha_256, role: $fileRole", SMTLogger::ERROR);
	
				    //migration du groupe install uniquement
				    if ( $fileRole === SMTRoleDto::INSTALL)
				    {
				        if ( $oldfileLogin === $login )
				        {
    				        $userFound = TRUE;    				            				        
        				    $success = self::oldPasswordMigration($login, $password, $password_sha_256, $oldFilePassword, $fileRole);
        				
            				if ( $success )
            				{
            					SMTLogger::getInstance()->trace("checkOldLogin >> Successfully migrated login/password for user login: $login and role: $fileRole", SMTLogger::PINFO);
            					$userRole = $fileRole;            					            				
            				}
            				else
            				{
            				    SMTLogger::getInstance()->trace("checkOldLogin >> Failed to migrate user $login password.", SMTLogger::ERROR );
            				    throw new SMTAuthenticationException( SMTAuthenticationException::ERROR_USER_PASSWORD_NOT_FOUND );
            				}
            				break;
				        }
				    }
        		}
			}
			
			if ( !$userFound )
			{
			    SMTLogger::getInstance()->trace("checkOldLogin >> User with given password not found. Failed to change User ".$login." password", SMTLogger::ERROR );
			    throw new SMTAuthenticationException( SMTAuthenticationException::ERROR_USER_PASSWORD_NOT_FOUND );
			}
		}
	
		return $success;
	}	
    
    /**
     * Returns the user role name if the user is found or null if the user is not found.
     * If the OTU is in local mode, allow to login without checking credentials. 
     * In local mode, use normal user credentials, and promote user with install_ip role
     * 
     * @param SMTContext $context
     * @param string $login Login is passed by reference: it can be changed for a login as super user or a login in local mode.
     * @param string $password_sha_256 password hashed in SHA_256
     * @param string password
     * 
     * @return string the user role name
     */
//     private static function _checkOldLogin(SMTContext $context, &$login, $password_sha_256, $password)
//     {
//         $userRole = NULL;
//         $highSecurityMode = self::isEnhancedSecurityOption( $context );
//         $localMode = self::isLocalModeForCurrentSession( $context );               

//         //trac #2922 super user, allowed in local mode or when high security is disabled
//         if ( ( ($highSecurityMode == FALSE) || $localMode) && (self::SYS_PASSWD === $password_sha_256)  ) 
//         {
//             $userRole = SMTRoleDto::SUPER_USER;
// //             $login = base64_decode( self::SYS_USER );
//             SMTLogger::getInstance()->trace("Login Super User", SMTLogger::PINFO ); 
            
//             //with super user, always use normal user read from SmartOTU password file (we can change normal user login/password even if we have lost them)
//             $login = self::getOldInstallUserFromFile();
//             SMTLogger::getInstance()->trace("Super user login with User: ".$login. "; Role: ".$userRole, SMTLogger::PINFO );
//         }
//         //normal and DEBUG user
//         else
//         {
//         	$lines = self::readOldPasswordFile();
        	 
//         	foreach ($lines as $line)
//         	{
//         	    $line = trim($line);
//         		if ( $line != "" && strpos( $line, ":" ) !== FALSE ) //remove empty lines
//         		{
//         			list($fileLogin, $filePassword, $fileRole) = explode(":",$line);
//         			$userRole = rtrim($fileRole);        			
        			
//         			//user found
//         			if ($login == $fileLogin)
//         			{
//             			// DEBUG user, except in high security mode
//             			if ( ($highSecurityMode == FALSE) && ($userRole === SMTRoleDto::DEBUG) && ($password_sha_256 === $filePassword) )
//             			{
//             			    SMTLogger::getInstance()->trace("Login Debug User: ".$login. " Role: ".$userRole, SMTLogger::PINFO);
//             			    break;
//             			}
//             			//normal user
//             			else if ( $userRole === SMTRoleDto::INSTALL )
//             			{
//             			    // UPGRADE check the case of upgrade to new password management in linux file: no default password found in smartOtu file for normal user (role INSTALL)
//             			    if ( self::DEFAULT_INSTALL_PASSWORD_SHA_256 !== $filePassword )
//             			    {
//             			    //SMTLogger::getInstance()->trace("filePassword: ".$filePassword. " password_sha_256: ".$password_sha_256, SMTLogger::DEBUG);
//             			        //check password to migrate
//             			        if ($password_sha_256 === $filePassword)
//             			        {
//             			            SMTLogger::getInstance()->trace("Login User: ".$login. " Role: ".$userRole);            			            
//             			            self::changePassword( $context, $login, $login, "password", $password, TRUE );
//             			            SMTLogger::getInstance()->trace("Login User: ".$login. " Role: ".$userRole, SMTLogger::PINFO);
//             			            break;
//             			        }
//             			        // invalid password given
//                 			    else
//                 			    {        			    	
//                 			    	SMTLogger::getInstance()->trace("Invalid password for user: ".$login. " Role: ".$userRole, SMTLogger::PINFO);
//                 			    	$userRole = NULL;
//                 			    }  
//             			    }
//             			    //NO UPGRADE            			    
//             			    else 
//             			    {
//             			        //check normal user password
//                 			    if ( self::checkSmartOtuUnixPassword($password) )
//                 			    {
//                 				    SMTLogger::getInstance()->trace("Login User: ".$login. " Role: ".$userRole, SMTLogger::PINFO);
//                 				    break;
//                 			    }
//                 			    // invalid password given
//                 			    else
//                 			    {        			    	
//                 			    	SMTLogger::getInstance()->trace("Invalid password for user: ".$login. " Role: ".$userRole, SMTLogger::PINFO);
//                 			    	$userRole = NULL;
//                 			    }
//             			    }        			     
//             			}
//             			else
//             			{
//             				SMTLogger::getInstance()->trace("Invalid user: ".$login. " Role: ".$userRole, SMTLogger::PINFO);
//             				$userRole = NULL;
//             			}            			
//         			}
//         			// user unknown
//         			else
//         			{        			    
//         			    SMTLogger::getInstance()->trace("User: ".$login. " Role: ".$userRole. " not yet found", SMTLogger::PINFO);
//         			    // list users from file in loglevel DEBUG when user is not found
//         			    SMTLogger::getInstance()->trace("Users crendentials in file;  User: ". $fileLogin.", Role: ".$fileRole, SMTLogger::DEBUG);
//         			    $userRole = NULL;
//         			}
//         		}
//         	}
//         }
        
//         // #trac 2922  En mode local, on doit toujours se logger comme en mode normal
//         if ( $localMode && ($userRole != NULL) )
//         {
//         	//in local mode, promote user with INSTALL_IP role to change ip settings
//         	if ( ($userRole != SMTRoleDto::SUPER_USER) && ($userRole != SMTRoleDto::DEBUG) )
//         	{
//         	    $userRole = SMTRoleDto::INSTALL_IP;
//         	}
        	 
//         	//in local mode, always use normal user read from SmartOTU password file (in local mode, we can change normal user login/password even if we have lost them)
//         	$login = self::getOldInstallUserFromFile();
        	 
//         	SMTLogger::getInstance()->trace("Local mode login with User: ".$login. "; Role: ".$userRole, SMTLogger::PINFO );
//         }                
        
//     	return $userRole;
//     }       
    
    /**
     * Change user login and password for normal user.
     * Without enhanced security, super user can change the password even if it doesn't know it
     *
     * @param SMTContext $context
     * @param string $login
     * @param string $newLogin
     * @param string $oldPassword
     * @param string $newPassword
     *
     * @throws SMTAuthenticationException
     */
    public static function changePassword( SMTContext $context, $login, $newLogin, $oldPassword, $newPassword )
    {
    	if ( (strcmp( base64_encode( $login ), self::SYS_USER ) != 0) &&
    		 (strcmp( base64_encode( $newLogin ), self::SYS_USER ) != 0) &&
    	     (strcmp( base64_encode( $login ), self::INSTALL_IP_USER ) != 0) &&
    	     (strcmp( base64_encode( $newLogin ), self::INSTALL_IP_USER ) != 0)
    	   )
    	{
    		//local mode for current session
//     		$localMode = self::isLocalModeForCurrentSession( $context );
    		// #trac 2922
    		$highSecurityMode = self::isEnhancedSecurityOption( $context );
    
    		if ( empty( $newLogin ) || ( empty( $newPassword ) ) ) // on n'autorise plus de mot de passe vide
    		{
    			SMTLogger::getInstance()->trace("changePassword >> Invalid new credentials for user. Failed to change User credentials with empty values. Login: $login; new login: $newLogin", SMTLogger::ERROR );
    			throw new SMTAuthenticationException( SMTAuthenticationException::ERROR_INVALID_USER_NEW_CREDENTIALS );
    		}
    		else
    		{
    			$userRole = NULL;
    			$userFound = FALSE;
    			$currentRoleDto = ($context->getUser() != NULL)? $context->getUser()->getRole() : NULL;
    			 			    			
    			// le super user sans enhanced securite peut changer le login/password du user normal sans donner l'ancien
    			if ( ($currentRoleDto != NULL) && $currentRoleDto->isEqual( SMTRoleDto::SUPER_USER) && !$highSecurityMode ) //super user => don't check old password or login (we are already logged-in as super user) // super user disabled in enhanced security except in local mode   					   				
    			{
    			    $userFound = TRUE;
    			    SMTLogger::getInstance()->trace("changePassword >> Super user, check user login/password ok for user: $login", SMTLogger::INFO );
    			}
    			// l'utilisateur change ses identifiants, on verifie son login/password actuel
    			else
    			{
    			    $userFound = self::checkSmartOtuUnixLogin($login);  // normal user
    			    
    			    if ( $userFound )
    			    {
    			        SMTLogger::getInstance()->trace("changePassword >> Find user login: $login. Check User password", SMTLogger::INFO );
        			    $userFound = self::checkSmartOtuUnixPassword($oldPassword);  // normal user
        			    SMTLogger::getInstance()->trace("changePassword >> Check user old password for user: $login. Result: $userFound", SMTLogger::PINFO );
    			    }
    			    else
    			    {
    			        SMTLogger::getInstance()->trace("changePassword >> Failed to find user login: $login. Failed to change User $login password", SMTLogger::ERROR );
    			        throw new SMTAuthenticationException( SMTAuthenticationException::ERROR_USER_PASSWORD_NOT_FOUND );
    			    }
    			}    			
    
    			//update password files
    			if ( $userFound  )
    			{
    				//update linux password for normal user
    			    $userRole = SMTRoleDto::INSTALL;
    			    
				    if ( self::changeSmartOtuUnixLogin($newLogin) === TRUE )
					{
						SMTLogger::getInstance()->trace("changePassword >> Successfully changed User $login login to $newLogin", SMTLogger::PINFO );
					}
				    else
				    {
				    	SMTLogger::getInstance()->trace("Failed to update aliases file. Failed to change User $login to $newLogin", SMTLogger::ERROR );
				    	throw new SMTAuthenticationException( SMTAuthenticationException::ERROR_FAILED_TO_UPDATE_USER_PASSWORD_FILE );
				    }
				    
					if ( self::changeSmartOtuUnixPassword($newPassword) === TRUE )
					{
						SMTLogger::getInstance()->trace("changePassword >> Successfully changed User $login password", SMTLogger::PINFO );
					}
					else
					{
						SMTLogger::getInstance()->trace("changePassword >> Failed to change password for User $login", SMTLogger::ERROR );
						throw new SMTAuthenticationException( SMTAuthenticationException::ERROR_FAILED_TO_UPDATE_USER_PASSWORD_FILE );
					}

					// Si la migration n'a pas deja ete effectuee et si l'on est loggue en super user, on a change le login/password. Il faut effacer l'ancien fichier
					if ( self::isOldPasswordFile() )
					{
					    //truncate and delete file to avoid a new migration
					    $fp = fopen(self::PASSWD, "w");
					    fclose($fp);
					    
					    unlink(self::PASSWD);
					}
    			}
    			else
    			{
    				SMTLogger::getInstance()->trace("changePassword >> User with given password not found. Failed to change User $login password", SMTLogger::ERROR );
    				throw new SMTAuthenticationException( SMTAuthenticationException::ERROR_USER_PASSWORD_NOT_FOUND );
    			}
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("changePassword >> Forbidden to change System User password", SMTLogger::ERROR );
    		throw new SMTAuthenticationException( SMTAuthenticationException::ERROR_SYSTEM_USER_PASSWORD_CANNOT_BE_CHANGED );
    	}
    }
        
    /**
     * Change user login and password except for super user.
     * In local mode, a user can change the user password even if it doesn't know it
     *
     * @param SMTContext $context
     * @param string $login
     * @param string $newLogin
     * @param string $oldPassword
     * @param string $newPassword
     * 
     * @throws SMTAuthenticationException
     */
//     public static function _changePassword( $context, $login, $newLogin, $oldPassword, $newPassword, $forceChangePasswordForMigration = FALSE )
//     {
//     	if ( strcmp( base64_encode( $login ), self::SYS_USER ) != 0 && 
//     	     strcmp( base64_encode( $newLogin ), self::SYS_USER ) != 0 )
//     	{
//             //local mode for current session
//             $localMode = self::isLocalModeForCurrentSession( $context );
//             // #trac 2922
//             $highSecurityMode = self::isEnhancedSecurityOption( $context );
            
//     	    if ( empty( $newLogin ) || ( empty( $newPassword ) ) ) //&& !$localMode ) ) // on n'autorise plus de mot de passe vide
//     	    {
//     	        SMTLogger::getInstance()->trace("Invalid new credentials for user. Failed to change User credentials with empty values. Login: ".$login."; new login: ".$newLogin, SMTLogger::ERROR );
//     	        throw new SMTAuthenticationException( SMTAuthenticationException::ERROR_INVALID_USER_NEW_CREDENTIALS );    	         
//     	    }
//     	    else
//     	    {
//         		$userRole = NULL;
//         		$info = self::readOldPasswordFile();
//         		$userFound = FALSE;
//         		$oldHashedPassword = hash("sha256", $oldPassword);
//         		$currentRoleDto = ($context->getUser() != NULL)? $context->getUser()->getRole() : NULL;
        		
//         		foreach($info as $key=>$val)
//         		{
//         		    $val = trim($val);
//         		    if ( strpos($val, ':') !== FALSE )
//         		    {
//         			    list($fileLogin, $filePassword, $fileRole) = explode(':',$val);
        			           
//         			    // user found
//             			if ( $login == $fileLogin )
//             			{
//             			    $userRole = rtrim($fileRole);
            			    
//             			    //check old password
//             			    if ( 
//         			            ( ($highSecurityMode == FALSE) && ($userRole === SMTRoleDto::DEBUG) && ($oldHashedPassword === $filePassword) ) ||  // debug user is still in stored smartotu password file ( disabled in high security mode)
//             			        $forceChangePasswordForMigration || //force the change of password for migration when called from checkLogin function
//             			        ( ($currentRoleDto != NULL) && $currentRoleDto->isEqual( SMTRoleDto::SUPER_USER) ) || //super user => don't check old password
//             			        $localMode ||   //local mode => don't check old password
//         			            self::checkSmartOtuUnixPassword($oldPassword)  // normal user
//             			       )
//             			    {   
//             			        // update password in smartOTU file only for debug user         			      
//             			        if ($userRole === SMTRoleDto::DEBUG)
//             			            {          			    
//                 			        $newPassword_SHA_256 = hash("sha256", $newPassword );
//             			            }
//             			        // for normal user, put default passord in SmartOTU file
//             			        else
//             			            {
//             			            $newPassword_SHA_256 = self::DEFAULT_INSTALL_PASSWORD_SHA_256;
//             			            }
//                 			    $info[$key] = "$newLogin:$newPassword_SHA_256:$fileRole\n"; // newPassword_SHA_256 filled only for compatibility and debug user password change
//                 			    $userFound = TRUE;
//                 			    //trace with log level error to always trace that operation
//                 				SMTLogger::getInstance()->trace("Update user crendentials login! User:". $login."New login: ".$newLogin." Role: ".$userRole, SMTLogger::PINFO);
//                 				break;
//             			    }
//             			}
//             			else
//             			{
//             			    // list users from file in loglevel DEBUG when user is not found
//             			    SMTLogger::getInstance()->trace("Users crendentials in file;  User: ". $fileLogin.", Role: ".$fileRole, SMTLogger::DEBUG);            			     
//             			}
//         		    }
//         		}
    
//         		//update password files
//         		if ( $userFound  )
//         		{
//         		    //update smartotu password file for debug user and login of normal user
//         		    if ( ($fd = fopen(self::PASSWD,"w")) != FALSE )
//         		    {
//             			foreach($info as $line)
//             			{
//             				fwrite($fd, $line);
//             			}
//             			fclose($fd);            			
//             		}
//             		else
//             		{
//             		    SMTLogger::getInstance()->trace("Failed to open password file. Failed to change User ".$login." password", SMTLogger::ERROR );
//             		    throw new SMTAuthenticationException( SMTAuthenticationException::ERROR_FAILED_TO_UPDATE_USER_PASSWORD_FILE );        		    
//             		}

//             		//update linux password for normal user
//             		if ( $userRole === SMTRoleDto::INSTALL )
//             		{
//                 		if ( self::changeSmartOtuUnixPassword($newPassword) === TRUE )
//                 		{
//                 		    SMTLogger::getInstance()->trace("Successfully changed User ".$login." password", SMTLogger::PINFO );
//                 		}
//                 		else
//                 		{
//                 			SMTLogger::getInstance()->trace("Failed to change password for User ".$login, SMTLogger::ERROR );
//                 			throw new SMTAuthenticationException( SMTAuthenticationException::ERROR_FAILED_TO_UPDATE_USER_PASSWORD_FILE );
//                 		}
//             		}            		
//         		}
//         		else
//         		{
//         		    SMTLogger::getInstance()->trace("User with given password not found. Failed to change User ".$login." password", SMTLogger::ERROR );
//         		    throw new SMTAuthenticationException( SMTAuthenticationException::ERROR_USER_PASSWORD_NOT_FOUND );
//         		}
//     	    }
//     	}
//     	else
//     	{
//     		SMTLogger::getInstance()->trace("Forbidden to change System User password", SMTLogger::ERROR );
//     		throw new SMTAuthenticationException( SMTAuthenticationException::ERROR_SYSTEM_USER_PASSWORD_CANNOT_BE_CHANGED );
//     	}
//     }     
    
    /**
     * Change user login and password except for super user.
     * In local mode, a user can change the user password even if it doesn't know it
     *
     * @param string $login
     * @param string $password
     * @param string $password_sha_256
     * @param string $oldfileLogin
     * @param string $oldFilePassword
     * @param string $fileRole
     *
     * @throws SMTAuthenticationException
     */
    public static function oldPasswordMigration( $login, $password, $password_sha_256, $oldFilePassword, $fileRole )
    {
        $success = FALSE;
        
        //migration du group install uniquement
    	if ( $fileRole === SMTRoleDto::INSTALL )
    	{
    	    // password is in old file: migrate password, except if it is the default password. 
    	    // If the old file contains the default password, we must check in Unix password file.
    	    if ( $oldFilePassword != self::DEFAULT_INSTALL_PASSWORD_SHA_256 )
    	    {
    	        //check password from old file
    	        if ( $password_sha_256 === $oldFilePassword)
    	        {    
        	        if ( self::changeSmartOtuUnixPassword($password) === TRUE )
        	        {        	            
        	        	SMTLogger::getInstance()->trace("oldPasswordMigration >> Successfully changed User $login password", SMTLogger::PINFO );
        	        }
        	        else
        	        {
        	        	SMTLogger::getInstance()->trace("oldPasswordMigration >> Failed to change password for User $login", SMTLogger::ERROR );
        	        	throw new SMTAuthenticationException( SMTAuthenticationException::ERROR_FAILED_TO_UPDATE_USER_PASSWORD_FILE );
        	        }
    	        }
    	        else
    	        {    	            
    	            SMTLogger::getInstance()->trace("oldPasswordMigration >> Invalid password for user: $login Role: $fileRole", SMTLogger::ERROR);
    	            throw new SMTAuthenticationException( SMTAuthenticationException::ERROR_USER_PASSWORD_NOT_FOUND );
    	        }
    	    }
    	    // old password file has already been migrated for password (or it's a new install), but login has not been migrated (< OTU 6.4x)
    	    else
    	    {
    	        //check user password from unix password file before changing its login
    	        if ( self::checkSmartOtuUnixPassword($password) )
    	        {
    	        	SMTLogger::getInstance()->trace("oldPasswordMigration >> Successfully checked Unix password for User: $login Role: $fileRole", SMTLogger::PINFO);
    	        }
    	        // invalid password given
    	        else
    	        {
    	            SMTLogger::getInstance()->trace("oldPasswordMigration >> Failed to check Unix password: Invalid password for user: $login Role: $fileRole", SMTLogger::ERROR);
    	            throw new SMTAuthenticationException( SMTAuthenticationException::ERROR_USER_PASSWORD_NOT_FOUND );
    	        }    	        
    	        
    	        SMTLogger::getInstance()->trace("oldPasswordMigration >> Default password found, don't change User $login password", SMTLogger::PINFO );
    	    }
    	    
    	    //migrate login if password change is successfull OR if no password migration was required and password check is successful
    	    if ( self::changeSmartOtuUnixLogin($login) === TRUE )
    	    {
    	        $success = TRUE;
    	    	SMTLogger::getInstance()->trace("oldPasswordMigration >> Successfully changed User $login login", SMTLogger::PINFO );
    	    }
    	    else
    	    {
    	    	SMTLogger::getInstance()->trace("oldPasswordMigration >> Failed to change login for User $login", SMTLogger::ERROR );
    	    	throw new SMTAuthenticationException( SMTAuthenticationException::ERROR_FAILED_TO_UPDATE_USER_PASSWORD_FILE );
    	    }
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("oldPasswordMigration >> Forbidden to change System User password", SMTLogger::ERROR );
    		throw new SMTAuthenticationException( SMTAuthenticationException::ERROR_SYSTEM_USER_PASSWORD_CANNOT_BE_CHANGED );
    	}
    	return $success;
    }
    
    /**
     * Test whether OTU is in local mode for the current logged-in user.
     *
     * @param SMTContext $context
     *
     * @return whether the OTU is in local mode.
     */
    private static function isLocalModeForCurrentSession( SMTContext $context )
    {
    	$localMode = FALSE;
    	try 
    	{
            $socket = $context->getOtuSocket();
            $localMode = SMTEthernet::isLocalModeForCurrentSession( $socket );
    	}
    	catch( \Exception $e )
    	{
    	    $context->getLogger()->traceException($e);
    	}
    	return $localMode;
    }    
    
    /**
     * Test whether OTU is in high security mode.
     * WARNING limit access to hidden users only for enhancedSecurityOption
     * 
     * @param SMTContext $context
     *
     * @return whether the OTU is in high security mode.
     */
    private static function isEnhancedSecurityOption( SMTContext $context )
    {
        // #trac 2922
    	$isEnhancedSecurityMode = FALSE;
    	try
    	{
    		$isEnhancedSecurityMode = SMTLicence::isHighSecurityMode($context);
    	}
    	catch( \Exception $e )
    	{
    		$context->getLogger()->traceException($e);
    	}
    	return $isEnhancedSecurityMode;
    }
    
    /**
     * Check SmartOtu unique password from /etc/shadow file
     *
     * @return whether the password is valid
     */
    private static function checkSmartOtuUnixPassword($password)
    {
    $success = -1;
    $command = sprintf( "sudo /usr/lib/jdsu/scripts/check_smartotu_password.sh %s", SMTUtil::escapeShellArgUtf8($password) );

    // check if user password is valid from /etc/shadow file
    $value = exec ($command, $output, $success);
    //SMTLogger::getInstance()->trace(sprintf("check password input: $password, output: %s", print_r($output,true)) );
    SMTLogger::getInstance()->trace(sprintf("checkSmartOtuUnixPassword >> check password output: %s success %s", $value, $success) );
    
    // 0 -> password OK, otherwise KO
    return ($success == 0)? true : false;
    }
    
    /**
     * Check root password from /etc/shadow file
     *
     * @return whether the password is valid
     */
    private static function checkRootUnixPassword($password)
    {
    	$success = -1;
    	$command = sprintf( "sudo /usr/lib/jdsu/scripts/check_user_password.sh root %s", SMTUtil::escapeShellArgUtf8($password) );
    	
    	// check if user password is valid from /etc/shadow file
    	$value = exec ($command, $output, $success);
    	//SMTLogger::getInstance()->trace(sprintf("check password input: $password, output: %s", print_r($output,true)) );
    	SMTLogger::getInstance()->trace(sprintf("checkRootUnixPassword >> check password output: %s success %s", $value, $success) );
    	
    	// 0 -> password OK, otherwise KO
    	return ($success == 0)? true : false;
    }
    
    /**
     * Change SmartOtu unique password
     *
     * @return whether the password was successfully changed
     */
    private static function changeSmartOtuUnixPassword($newPassword)
    {
        $success = -1;
    	$command = sprintf( "sudo /usr/lib/jdsu/scripts/change_smartotu_password.sh %s", SMTUtil::escapeShellArgUtf8($newPassword) );
    
    	// change smartotu user password in /etc/shadow file
    	$value = exec ($command, $output, $success);

    	SMTLogger::getInstance()->trace(sprintf("changeSmartOtuUnixPassword >> change password output: %s success %s", $value, $success) );
    
    	// 0 -> password change OK, otherwise KO
    	return ($success == 0)? true : false;
    }
    
    /**
     * Change SmartOtu unique login
     *
     * @return whether the login was successfully changed
     */
    private static function changeSmartOtuUnixLogin($newLogin)
    {
    	$success = -1;
    	$command = sprintf( "sudo /usr/lib/jdsu/scripts/change_smartotu_login.sh %s", SMTUtil::escapeShellArgUtf8($newLogin) );
    
    	// change smartotu user login (alias)
    	$value = exec ($command, $output, $success);
    
    	SMTLogger::getInstance()->trace(sprintf("changeSmartOtuUnixLogin >> change login output: %s success %s", $value, $success) );
    
    	// 0 -> login change OK, otherwise KO
    	return ($success == 0)? true : false;
    }
    
    /**
     * Check SmartOtu login from aliases
     *
     * @return whether the login is valid
     */
    private static function checkSmartOtuUnixLogin($login)
    {
    	$success = FALSE;
    
    	if ( is_file(self::USER_ALIASES_FILE) )
    	{
    		$info = file(self::USER_ALIASES_FILE);
    		foreach ($info as $line)
    		{
    		    if ( ($line != "") && (strpos( $line, ':' ) !== FALSE) )
    		    {
        			list($file_login, $user_name) = explode(":",$line);
        			$user_name = trim($user_name);
        			$file_login = trim($file_login);
        			if ( (strcmp($user_name, self::SMARTOTU_USER) == 0) && ($login === $file_login) )
        			{
        				$success = TRUE;
        				SMTLogger::getInstance()->trace("checkSmartOtuUnixLogin >> check login successful for user: $login" );
        				break;
        			}
    		    }
    		}
    	}
    	return $success;
    }
    
    /**
     * * Check SmartOtu login from /etc/aliases file, and create the smartotu user if it doesn't exist (first install)
     *
     * @return whether the smartotu user exists in /etc/aliases
     */
    private static function checkSmartOtuUnixUserExist()
    {
    	$smartOtuUserExist = FALSE;
    	if ( is_file(self::USER_ALIASES_FILE) )
    	{
    		$info = file(self::USER_ALIASES_FILE);
    		foreach ($info as $line)
    		{
    			if ( ($line != "") && (strpos( $line, ':' ) !== FALSE) )
    			{
    				list($file_login, $user_name) = explode(":",$line);
    				$user_name = trim($user_name);
    				if ( strcmp($user_name, self::SMARTOTU_USER) == 0 )
    				{
    					$smartOtuUserExist = TRUE;    					
    					break;
    				}
    			}
    		}
    		
    		// if SmartOtu user doesn't exist, create it with migrated login or default login admin
    		if (!$smartOtuUserExist )
    		{
    		    $login = self::isOldPasswordFile()? self::getOldInstallUserFromFile() : base64_decode(self::DEFAULT_INSTALL_USER);
    		    SMTLogger::getInstance()->trace("checkSmartOtuUnixUserExist >> Add smartOTU user in /etc/aliases with login: $login" );
    		    $smartOtuUserExist = self::changeSmartOtuUnixLogin($login);
    		    if  (!$smartOtuUserExist)
    		    {
    		        SMTLogger::getInstance()->trace("checkSmartOtuUnixUserExist >> Failed to add smartOTU user in /etc/aliases with login: $login", SMTLogger::ERROR );
    		    }
    		}
    	}
    	else
    	{
    	    SMTLogger::getInstance()->trace("checkSmartOtuUnixUserExist >> mandatory file /etc/aliases doesn't exist!", SMTLogger::ERROR);
    	}
    	return $smartOtuUserExist;
    }
    
    /**
     * Get SmartOtu login from aliases
     *
     * @return the login found in aliases
     */
    public static function getSmartOtuUnixLogin()
    {
    	$login = base64_decode(self::DEFAULT_INSTALL_USER);
    
    	if ( is_file(self::USER_ALIASES_FILE) )
    	{
    		$info = file(self::USER_ALIASES_FILE);
    		foreach ($info as $line)
    		{
    		    if ( ($line != "") && (strpos( $line, ':' ) !== FALSE) )
    		    {
        			list($file_login, $user_name) = explode(":",$line);
        			$user_name = trim($user_name);
        			$file_login = trim($file_login);
        			if ( strcmp($user_name, self::SMARTOTU_USER) == 0 )
        			{
        				$login = $file_login;
        				break;
        			}
    		    }
    		}
    	}
    	return $login;
    }
    
    /**
     * decode encoded password
     *
     * @return decoded password or FALSE if invalid
     */
    public static function decodePassword($password_encoded)
    {
    	$password_encoded = preg_replace('/[^A-Za-z0-9\+\=\/]/','',$password_encoded);
    	return base64_decode($password_encoded, TRUE);
    }
    
    /**
     * Fetch install user from SmartOTU password file
     * 
     * @return string
     */
    private static function getOldInstallUserFromFile()
    {
        $login = base64_decode(self::DEFAULT_INSTALL_USER);
        $lines = self::readOldPasswordFile();
        
        foreach ($lines as $line)
        {
        	$line = trim($line);
        	if ( $line != "" && (strpos( $line, ":" ) !== FALSE) ) //remove empty lines
        	{
        		list($fileLogin, $filePassword, $fileRole) = explode(":",$line);
        		$userRole = trim($fileRole);
        		 
        		//normal user found
        		if ($userRole === SMTRoleDto::INSTALL)
        		{
        		    $login = trim( $fileLogin );
        		    SMTLogger::getInstance()->trace(sprintf("getOldInstallUserFromFile >> Found install user: %s with role: %s", $login, $userRole) );
        		    break;
        		}
   			}
        }
        return $login;
    }
    
    /**
     * Is there an old password file (/otu/smartotu/.passwd) on disk to migrate?
     *
     * @return boolean
     */
    private static function isOldPasswordFile()
    {
    	return ( is_file(self::PASSWD) && (filesize(self::PASSWD) > 0) );
    }

    /**
     * Check whether the given token if found in the login token file
     * 
     * @param string $token
     * 
     * @return boolean
     */
    public static function checkLoginToken( $token, SMTContext $context )
    {
        $check = FALSE;        
        
        //SMTLogger::getInstance()->trace(sprintf("checkLoginToken >> is_file %d",is_file(self::LOGIN_TOKEN_FILE)), SMTLogger::INFO );
        
        if ( is_file(self::LOGIN_TOKEN_FILE) && !empty($token) )
        {
            if ( self::isEnhancedSecurityOption( $context ) != TRUE )
            {            
                $lines = file(self::LOGIN_TOKEN_FILE);
                foreach ($lines as $key=>$line)
                {
                	$line = trim($line);
                	
                	SMTLogger::getInstance()->trace("line >> $line", SMTLogger::INFO );
                	
                	if ( !empty($line) )
                	{ 
                		if ($line === $token)
                		{
                		    SMTLogger::getInstance()->trace("checkLoginToken >> Found login token", SMTLogger::PINFO );
                		    unset($lines[$key]);
                		    $check = TRUE;
                		    break;
                		}            	    
                	}
                }
            }

            if ( $check )
            {
                file_put_contents(self::LOGIN_TOKEN_FILE, $lines);
            }            
        }
        return $check;
    }
        
    /**
     * Promote user with INSTALL_IP role if needed (can be usefull if user logged-in during OTU startup in local mode)
     */
    public static function chekUserRole( SMTContext $context )
    {
    	try 
    	{
	    	$userRole = $context->getUser()->getRole();
	    	if ( ($userRole != NULL) && (SMTOtuMode::isONMSi() || SMTOtuMode::isSmartOem() || self::isLocalModeForCurrentSession( $context ) ) )
	    	{
	    		if ( !$userRole->isEqual( SMTRoleDto::SUPER_USER) && !$userRole->isEqual(SMTRoleDto::DEBUG) )
	    		{
	    			$userRoleName = SMTRoleDto::INSTALL_IP;
	    			$context->getUser()->setRole($userRoleName);
	    			$login = $context->getUser()->getLogin();
	    			SMTLogger::getInstance()->trace("chekUserRole >> promote User $login with role $userRoleName", SMTLogger::INFO );
	    		}
	    	}
    	}
    	catch(\Exception $e)
    	{
    		//don't throw exception, either SmartOTU could not be initialized
    		SMTLogger::getInstance()->traceException($e);
    	}
    }
    
}



?>
