<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\monitoring\cache;

use app\util\SMTUtil;

use app\events\operations\SMTOperationEventDto;

use app\sharedmemory\SMTMemoryManager;

use app\sharedmemory\SMTIMemorySupport;

use app\util\SMTRwDto;

/**
 * Operation (measure or test) data to keep during the life of the operation
 * 
 * @author Sylvain Desplat
 *
 */
class SMTOperationDataDto extends SMTRwDto implements SMTIMemorySupport
{              
    /**
     * Identifier of the operation (measure, test)
     * @var string
     */
    protected $operationId;
    
    /**
     * Port identifier where the operation was launched
     * @var number
     */
    protected $portId = 0;    
    
    /**
     * Last access to query or update the operation (measure or test) cache
     *
     * @var float
     */
    private $lastAccess = 0;
    
    /**
     * Operation current status
     * 
     * @var string
     */
    protected $currentOperationStatus = SMTOperationEventDto::INVALID_STATUS;
    
    public function getLastAccess()
    {
    	return $this->lastAccess;
    }        
    
    public function touchAccess()
    {
    	$this->lastAccess = microtime(true);
    }    
    
    function setOperationId( $operationId )
    {
    	$this->operationId = $operationId;
    }        
    
    function getOperationId()
    {
    	return $this->operationId;
    }    
        
    function getPortId()
    {
    	return $this->portId;
    }
    
    function setPortId( $portId )
    {
    	$this->portId = SMTUtil::convertToInt( $portId );
    }        
    
    function getStatus()
    {
    	return $this->currentOperationStatus;
    }
    
    function setStatus( $currentOperationStatus )
    {
    	$this->currentOperationStatus = $currentOperationStatus;
    }    
    
    /**
     * Save Dto in shared memory and updates its last access timestamp
     * WARNING: updates last access time stamp
     *
     * @throws SMTAPCMemoryException
     */
    public function save()
    {
        //always update last access property before saving an operation DTO
        $this->touchAccess();
        SMTMemoryManager::saveDto( $this );
    }
    
    /**
     * Delete Dto from shared memory
     *
     * @throws SMTAPCMemoryException
     */
    public function delete()
    {
    	SMTMemoryManager::deleteDto( $this );
    }  
      
    /**
     * The DTO identifier
     */
    public function getId()
    {
        return $this->getOperationId();
    }
      
    
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }        
    
    /**
     * Returns the dto class name.
     *
     * @return the dto class name.
     */
    public static function getClass()
    {
    	return __CLASS__;
    }    
    
    /**
     * Create the Dto from a Json Dto in an array
     *
     * @param array $json_array json data in an array
     *
     * @return \app\services\monitoring\SMTLinkTestDto
     */
    static function getInstance( array &$json_array )
    {
    	return self::forge( get_class(), $json_array );
    }        
    
    /**
     * Check whether the operation (measure or test) was not updated or requested since a time > 10 times the period between 2 operation messages retrievals.
     *
     * @return boolean Whether the operation (measure or test) were not updated or requested since a time > 10 times the period between 2 operation messages retrievals.
     */
    public function isObsolete()
    {
        if ( !isset( $this->lastAccess ) )
        {
            $this->touchAccess();
        }
    	//After 10 mins, the cached data for measures or tests are obsolete and should be deleted
    	return ( microtime(TRUE) - $this->getLastAccess() ) > SMTIMemorySupport::DEFAULT_TIMEOUT_SEC;
    }
}

?>