<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\media;

use app\serviceshelper\system\SMTOtu;

use app\serviceshelper\SMTServiceHelper;

use app\parser\SMTOtuApi;
use app\services\setup\SMTConfSmsDto;
use app\services\setup\SMTSmsDto;

define("MSG_SEND_SMS_MESSAGE","Test sent by OTU: %s");

/**
 *
 * @author Sylvain Desplat
 *
 */
class SMTSms extends SMTServiceHelper
{    
    const INTERNAl_1 = "INT1";
    const MIN_SMS_INDEX = 1;
    const MAX_SMS_INDEX = 5;
    
    /**
     * Sms media.
     *
     * @var string
     */
    const SMS_MEDIA = "SMS";    
	
	/**
	 * 
	 * @var string "" or GSM or PSTN or ISDN
	 */
	private $modemType = NULL;
	
	/**
	 * Modem is ready if command OTU:MODEM:READY returns YES
	 * 
	 * @var boolean
	 */
	private $modemReady = NULL;

	/**
	 * Fetch an sms with the given index.
	 *
	 * @param string $smsIndex
	 *
	 * @return \app\services\setup\SMTSmsDto
	 */
	function retrieveSms( $smsIndex )
	{
	    $smsDto = NULL;
	    $phoneNumber = $this->sendReceive( sprintf( SMTOtuApi::CMD_get_sms_conf, $smsIndex) );
	    
	    $isSMSConfigValid = self::isSMSConfigValid( $phoneNumber );
		if ( $isSMSConfigValid )
		{
    		$smsDto = new SMTSmsDto();
    		$smsDto->setValid( $isSMSConfigValid );
    		$smsDto->setIndex( $smsIndex );
    		$smsDto->setPhoneNumber($phoneNumber);
		}
		return $smsDto;
	}
		
	/**
	 * 
	 * @return string the modem type GSM, PSTN, ISDN, "" if not set
	 */
	function retrieveModemType()
	{
	    //modem type GSM...
	    if ( $this->modemType == NULL )
	    {
	        $this->modemType = $this->sendReceive( SMTOtuApi::getModemConfig() );
	    }
	    return $this->modemType;
	}
	
    /**
     * @return boolean
     */
    public function isValidModemType()
    {
    	return ( strcmp( $this->modemType, SMTConfSmsDto::MODEM_GSM  ) == 0 || strcmp( $this->modemType, SMTConfSmsDto::MODEM_ISDN ) == 0 || strcmp( $this->modemType, SMTConfSmsDto::MODEM_PSTN ) == 0 );
    }

	
	/**
	 * 
	 * @return boolean
	 */
	function isModemReady()
	{
	    if ( $this->modemReady == NULL )
	    {
	        $modemReady = $this->sendReceive( SMTOtuApi::getModemReady() );
	        $this->modemReady = ( strcmp( $modemReady, SMTOtuApi::RES_YES ) == 0 );
	    }
	    return $this->modemReady;
	}
	/**
	 *
	 * Set Sms media ON or OFF
	 * @param boolean $on Whether sms media must be activated or not
	 */
	function setSmsMedia( $on )
	{
		$this->send( SMTOtuApi::setSmsMedia( $on ) );
		//update media period and sms max retry
		if ( $on )
		{
			$this->send( SMTOtuApi::setMediaRetryPeriod( SMTOtuApi::MEDIA_RETRY_DELAY ) );
			$this->send( SMTOtuApi::setSmsMaxRetry( SMTOtuApi::MEDIA_RETRY_COUNT ) );
		}
	}	
	
	/**
	 * Check if internal modem for sms is available (option modem, modem type, modem available)
	 * 
	 * @throws SMTMediaException
	 */
	function checkSmsAvailability()
	{
	    //test if modem option is set
	    $modemOption = $this->isModemOption();
	    $validModemType = $this->isValidModemType();
	    $modemReady = $this->isModemReady();
	    
	    //test if modem option is set and if modem is available
        if (!$modemOption )
	    {
	    	throw new SMTMediaException( SMTMediaException::ERROR_OPTION_MODEM_NOT_SET );
	    }
	    else if (!$validModemType )
	    {
	    	throw new SMTMediaException( SMTMediaException::ERROR_INVALID_MODEM_TYPE );
	    }
	    else if (!$modemReady )
	    {
	    	throw new SMTMediaException( SMTMediaException::ERROR_MODEM_NOT_AVAILABLE );
	    }
	}
	
	/**
	 *
	 * Retrieve if Sms media is ON
	 *
	 */
	function isSmsMediaOn()
	{
		//SMS status on ALT2
		$mediaConfig = $this->sendReceive( SMTOtuApi::getSmsMedia() );
		$mediaConfig = explode(",", $mediaConfig );
		 
		$notificationStatus = ($mediaConfig[0] == self::SMS_MEDIA);
		return $notificationStatus;
	}
	
	/**
	 *
	 * Validate the sms configuration
	 *
	 * @param string $smsString the sms configuration to check
	 *
	 * returns true if the sms configuration is valid
	 */
	private static function isSMSConfigValid( $smsString )
	{
		$isSmsConfigValid = false;
		if ( $smsString != SMTOtuApi::RES_UNKNOWN && $smsString != SMTOtuApi::RES_NONE && $smsString != "" )
		{
			$isSmsConfigValid = true;
		}
		return $isSmsConfigValid;
	}
	
	/**
	 * Whether the modem option is set.
	 * 
	 * @return boolean
	 */
	function isModemOption()
	{
	    $modemType = $this->retrieveModemType();
		return (($modemType != '') && ($modemType == 'PSTN' || $modemType == 'ISDN' || $modemType == 'GSM'));
	}	
	
	/**
	 * Build Sms test message
	 * @return string
	 */
	function buildTestSmsMessage()
	{
		$testString = sprintf( MSG_SEND_SMS_MESSAGE, "" );
		try
		{
		    $otu = new SMTOtu();
		    $otu->setContext( $this->getContext() );
		    $serialNumber = $otu->getOtuSerialNumber();
			$testString = sprintf( MSG_SEND_SMS_MESSAGE, $serialNumber );
		}
		//catch exception: continue to sent if the OTU name retrieval failed
		catch( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
		}
		return $testString;
	}
	
	/**
	 * Update sms config on OTU
	 *
	 * @param SMTConfSmsDto $configSmsDto
	 */
	function updateSms( SMTConfSmsDto $configSmsDto )
	{
		$smsIndexToRemove = self::MIN_SMS_INDEX;

		foreach ( $configSmsDto->getSmsArray() as $smsDto )
		{
			$smsIndexToRemove++;
			$this->send( sprintf( SMTOtuApi::CMD_set_sms_conf, $smsDto->getIndex(), $smsDto->getPhoneNumber() ) );
		}
	
		for ( $index = $smsIndexToRemove; $index <= self::MAX_SMS_INDEX; $index++ )
		{
			$this->send( sprintf( SMTOtuApi::CMD_set_sms_conf, $index, "" ) );
		}
	
	}
}
?>