<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\services\maintenance;

use app\sharedmemory\SMTMemoryManager;

use app\sharedmemory\SMTIMemorySupport;

use app\util\SMTDto;

/**
 * DTO tracking upload progress status
 * 
 * @author Sylvain Desplat
 */
class SMTUploadStatusDto extends SMTDto implements SMTIMemorySupport
{
    const UNIQUE_UPLOAD_ID="UNIQUE_UPLOAD_ID";
    const NO_UPLOAD = -1;
    const UPLOAD_COMPLETE = 100;
    const NO_RATE = 0;
    
    const MAX_FILE_SIZE_EXCEEDED="MAX_FILE_SIZE_EXCEEDED";
    const PARTIAL_FILE_UPLOAD="PARTIAL_FILE_UPLOAD";
    const NO_FILE_ATTACHED="NO_FILE_ATTACHED";
    const FILE_PERMISSION_DENIED="FILE_PERMISSION_DENIED";
    const UNEXPECTED_ERROR = "UNEXPECTED_ERROR";
    const COPY_FAILED = "COPY_FAILED";
    const UPLOAD_ALREADY_RUNNING = "UPLOAD_ALREADY_RUNNING";
    const UPGRADE_ALREADY_RUNNING = "UPGRADE_ALREADY_RUNNING";
    const NO_ERROR = "NO_ERROR";
    
    /**
     * Last access to query or update the upload status cache
     *
     * @var float
     */
    private $lastAccess = 0;
    
    /**
     * The progress of the upload. Ranges from 0 to 100. < 0 if invalid
     * 
     * @var integer
     */
	protected $progress = self::NO_UPLOAD;

	/**
	 * The rate (Mo/s) of the upload. ( >= 0 )
	 *
	 * @var float
	 */
	protected $uploadRateMBs = self::NO_RATE;
	
	/**
	 * The upload error status among the values:
	 * MAX_FILE_SIZE_EXCEEDED, PARTIAL_FILE_UPLOAD, NO_FILE_ATTACHED, FILE_PERMISSION_DENIED, UNEXPECTED_ERROR, COPY_FAILED, UPLOAD_ALREADY_RUNNING, NO_ERROR 
	 * 
	 * @var string
	 */	
	protected $uploadErrorStatus = self::NO_ERROR;
	
	/**
	 * The uploaded file name
	 * 
	 * @var string
	 */
	protected $fileName;
		
	/**
	 * The progress of the upload. Ranges from 0 to 100. < 0 if invalid
	 *
	 * @param integer
	 */
	public function setProgress( $progress )
	{
		$this->progress = $progress;
	}
	
	/**
	 * The name of the file being uploaded
	 *
	 * @param string
	 */
	public function setFileName( $fileName )
	{
		$this->fileName = $fileName;
	}

	/**
	 * The rate (MB/s or Mo/s) of the upload. ( >= 0 )
	 *
	 * @param float $rateMBs
	 */
	public function setRateMBs( $rateMBs )
	{
		$this->uploadRateMBs = $rateMBs;
	}
	
	/**
	 * The rate (MB/s or Mo/s) of the upload. ( >= 0 )
	 *
	 * @return float $rateMBs
	 */
	public function getRateMBs()
	{
		return $this->uploadRateMBs;
	}	
	
	/**
	 * The progress of the upload. Ranges from 0 to 100. < 0 if invalid
	 *
	 * @return integer 
	 */
	public function getProgress()
	{
		return $this->progress;
	}
	
	/**
	 * The name of the file being uploaded
	 *
	 * @return string
	 */
	public function getFileName()
	{
		return $this->fileName;
	}
	
	/**
	 * The upload error status among the values:
	 * MAX_FILE_SIZE_EXCEEDED, PARTIAL_FILE_UPLOAD, NO_FILE_ATTACHED, FILE_PERMISSION_DENIED, UNEXPECTED_ERROR, COPY_FAILED, UPLOAD_ALREADY_RUNNING, UPGRADE_ALREADY_RUNNING, NO_ERROR
	 * 
	 * @param string
	 */
	public function setUploadErrorStatus( $uploadErrorStatus )
	{
		$this->uploadErrorStatus = $uploadErrorStatus;
	}
	
	/**
	 * The upload error status among the values:
	 * MAX_FILE_SIZE_EXCEEDED, PARTIAL_FILE_UPLOAD, NO_FILE_ATTACHED, FILE_PERMISSION_DENIED, UNEXPECTED_ERROR, COPY_FAILED, UPLOAD_ALREADY_RUNNING, UPGRADE_ALREADY_RUNNING, NO_ERROR
	 * 
	 * @return string
	 */
	public function getUploadErrorStatus()
	{
		return $this->uploadErrorStatus;
	}	
	
	/**
	 * Whether the upload is in error
	 * 
	 * @return boolean
	 */
	public function hasErrorStatus()
	{
	    return $this->uploadErrorStatus != self::NO_ERROR;
	}
	
	/**
	 * Whether an upload is in progress
	 *
	 * @return boolean
	 */
	public function isUploadInProgress()
	{
		return $this->progress > self::NO_UPLOAD && $this->progress < self::UPLOAD_COMPLETE;
	}
	
	/**
	 * Serialize as Json data (array) the user object.<br>
	 *
	 * @return $objectProperties the object properties serialized as a Json array.
	 */
	function getJsonData()
	{
		$objectProperties = get_object_vars($this);
		$this->serializeObjectAsJsonData( $objectProperties );
		return $objectProperties;
	}

	/**
	 * Returns Dto class name.
	 *
	 * @return string the dtop class name
	 */
	function getDtoClassName()
	{
		return get_class();
	}
	
	/**
	 * Returns the dto class name.
	 *
	 * @return the dto class name.
	 */
	public static function getClass()
	{
		return __CLASS__;
	}
	
	/**
	 * Save Dto in shared memory and updates its last access timestamp
	 * WARNING: updates last access time stamp
	 *
	 * @throws SMTAPCMemoryException
	 */
	public function save()
	{
		$this->touchAccess();
		SMTMemoryManager::saveDto( $this );
	}
	
	/**
	 * Delete Dto from shared memory
	 *
	 * @throws SMTAPCMemoryException
	 */
	public function delete()
	{
		SMTMemoryManager::deleteDto( $this );
	}
	
	/**
	 * The DTO identifier
	 */
	public function getId()
	{
		return self::UNIQUE_UPLOAD_ID;
	}
	
	/**
	 * No obsolescence
	 *
	 * @return FALSE
	 */
	public function isObsolete()
	{
		return FALSE;
	}
	
	/**
	 * Returns the timestamp of the last APC memory access (used to handle obsolescence of dtos in cache)
	 * @see isObsolete()
	 * @return float Timestamp of the last APC memory access (used to handle obsolescence of dtos in cache)
	 */
	public function getLastAccess()
	{
		return $this->lastAccess;
	}
	
	/**
	 * Update Timestamp of the last APC memory access (used to handle obsolescence of dtos in cache)
	 * @see isObsolete()
	 *
	 */
	public function touchAccess()
	{
		$this->lastAccess = microtime(true);
	}	
}


?>