<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\database;

use app\serviceshelper\system\SMTSystemConfigurationDto;

use app\parser\SMTSocketException;

use app\http\SMTContext;

use app\services\alarm\SMTAlarmEventDto;

use app\services\alarm\SMTAlarmDto;

use app\util\SMTResultSetDto;

use app\settings\SMTOtuStatusCacheDto;

use app\database\patch\SMTSql01p0001;
use app\database\patch\SMTSql01p0002;
use app\database\patch\SMTSql01p0003;
use app\database\patch\SMTSql01p0004;
use app\database\patch\SMTSql01p0005;
use app\database\patch\SMTSql01p0006;

use app\util\SMTLogger;

/**
 * Database object: wraps a SQLite3 database object on smartOTU datafile.
 * 
 * @author Sylvain Desplat
 */
class SMTSmartOtuDB extends \SQLite3
{
    /**
     * Database file was created and its model is not yet created.
     * @var boolean
     */
    var $isNewDatabase = FALSE;
    var $currentDatabaseVersion;
    var $currentDatabasePatch;
    /** 
     * @var integer
     */
    var $targetDatabaseVersion;
    /**
     * @var integer
     */    
    var $targetDatabasePatch;
    
    /**
     * @var boolean
     */
    var $transactionRunning = FALSE;
    
    const SQL_ENABLE_FOREIGN_KEYS = "PRAGMA foreign_keys = %s";
    
    const LOCK_CHECK_DATABASE = "/../../tmp/checkDatabase.lock";
        
    /**
     * SQLite database busy error code
     * @var integer
     */
    const SQLITE_BUSY = 5;
    /**
     * SQLite table locked error code
     * @var integer
     */    
    const SQLITE_LOCKED = 6;
    
    /**
     * Default sleep time when database of table is locked: 0.25s
     * @var integer
     */
    const SLEEP_LOCKED_uS = 250000;
    
    /**
     * Whether database connection is closed
     * @var boolean
     */
    private $isClosed = FALSE;
    
    /**
     * @var string
     */
    private static $checkDatabaseLockFile = NULL;    

    const LOCK_ALARM_SEQUENCE_DATABASE = "/../../tmp/alarmSequenceDatabase.lock";
    
    /**
     * @var string
     */
    private static $alarmSequenceDatabaseLockFile = NULL;
    
    const LOCK_ALARM_EVENT_SEQUENCE_DATABASE = "/../../tmp/alarmEventSequenceDatabase.lock";
    
    /**
     * @var string
     */
    private static $alarmEventSequenceDatabaseLockFile = NULL;
    
    /**
     * @return SMTSmartOtuDB Returns a new SMTSmartOtuDB instance
     */
    public static function getInstance( SMTContext $context )
    {
    	return new SMTSmartOtuDB( $context );
    }
    
    function __construct( SMTContext $context )
    {
        //Database check must be performed in a critical section
        if ( $this->acquireCheckDatabaseLock() )
        {
            try 
            {
                //Whether a new empty database file was created.
                $this->isNewDatabase = $this->checkDatabaseFile( $context );                                
                
                //open database connection
                $databaseFileName = SMTDatabaseUtil::getDatabaseFileName(); 
                $this->open( $databaseFileName );
                
                SMTLogger::getInstance()->trace("Database opened using: ".$databaseFileName, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
                
                //Set parameter to wait a few seconds for database locks to clear: 
                //sleep until the database is not locked or the timeout is reached
                $this->busyTimeout(5000);
                
                //check database version (only one time after reboot) and create or update the database model if needed
                $this->checkDatabaseModel();
                
                $this->releaseCheckDatabaseLock();
                
                //MANDATORY: force foreign_key support
                $this->exec( self::getEnableForeignKey( TRUE ) );
            } 
            catch (\Exception $e) 
            {
                $this->releaseCheckDatabaseLock();
                
                SMTLogger::getInstance()->traceException($e);
                throw new SMTDatabaseException( SMTDatabaseException::OPEN_DATABASE_FAILURE );
            }
        }
        else
        {
            SMTLogger::getInstance()->trace( "Couldn't acquire lock to check database!", SMTLogger::ERROR);
            throw new SMTDatabaseException( SMTDatabaseException::OPEN_DATABASE_FAILURE );
        }
    }
    
    /**
     * Try to release the database connection when exiting
     */
    function __destruct()
    {
    	$this->close();
    }
    
    /**
     * Closes the database connection if it is not already closed
     * 
     * @link http://www.php.net/manual/en/sqlite3.close.php
     * @return bool true on success, false on failure.
     */
    public function close () 
    {
        if ( !$this->isClosed )
        {
            $this->isClosed = parent::close();
        }
        return $this->isClosed ;
    }
    
    /**
     * Returns whether database connection is closed
     * 
     * @return boolean
     */
    public function isClosed()
    {
        return $this->isClosed;
    }
    
    /**
     * Check whether the database exists or whether the database must be droped and recreated:
     *  - if the database file doesn't exist, create it.
     *  - if the database file must be droped and recreated, delete the database file and recreate it.
     * 
     * @return boolean Returns true if a new empty database file was created.
     * @throws SMTSocketException
     */
    private function checkDatabaseFile( SMTContext $context )
    {
        //if the database file doesn't exist, create it
        $isNewDatabase = SMTDatabaseUtil::createDatabaseFileIfNeeded();
        if ( $isNewDatabase )
        {
        	SMTLogger::getInstance()->trace(sprintf("New database: %b",$isNewDatabase), SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
        }
        
        //No longer used: database is already cleaned up by OTU application when changing mode and when adding a new test, we delete the old data referred by that test 
//         //cleanup database after reboot if no monitoring test is found to handle the case of an OTU with an old SmartOTU database with old data
//         //when switching to SmartOTU mode
//         else
//         {
//         	if ( !SMTOtuStatusCacheDto::getDatabaseClearedStatusChecked() )
//         	{
//         		//check only one time after reboot whether the database must be cleared.
//         		SMTOtuStatusCacheDto::setDatabaseClearedStatusChecked( TRUE );
        		
//         		//if the database file must be droped and recreated, delete the database file and recreate it
//         		$isNewDatabase = SMTSmartOtu::cleanUpSmartOtuIfNeeded( $isNewDatabase );
//         	}
//         }
        return $isNewDatabase;
    }
    
    /**
     * Check databaseModel:
     * - check database version (only one time after reboot) and create or update the database model if needed
     * 
     * @throws Exception If database model couldn't be created.
     */
    private function checkDatabaseModel()
    {
        //if database model has not been checked after reboot or if database model is not created yet, perform the check:
        if ( !SMTOtuStatusCacheDto::getDatabaseVersionChecked() || $this->isNewDatabase  )
        {
        	SMTLogger::getInstance()->trace("Start check database", SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        
        	//Only one database check of the version after reboot
        	SMTOtuStatusCacheDto::setDatabaseVersionChecked( TRUE );
        	try
        	{
        		//Database is new, create the model
        		if ( $this->isNewDatabase || !$this->doesDatabasePatchTableExist() )
        		{
        			SMTLogger::getInstance()->trace("********* Start database model creation ************");
        			$this->createModel();
        			SMTLogger::getInstance()->trace("********* End database model creation ************");
        		}
        		//database model already exists, tests that all the patches have been applied
        		else
        		{            		
            		SMTLogger::getInstance()->trace("********* Start database apply patches ************");
          			$this->checkDatabaseVersion();
            		SMTLogger::getInstance()->trace("********* End database apply patches ************");
        		}
        	}
        	catch( \Exception $e)
        	{
        		//force database to be checked again
        		SMTOtuStatusCacheDto::setDatabaseVersionChecked( FALSE );
        		throw $e;
        	}
        }
    }
    
    /**
     * Executes a result-less query against a given database
     * @link http://www.php.net/manual/en/sqlite3.exec.php
     * @param query string <p>
     * The SQL query to execute (typically an INSERT, UPDATE, or DELETE
     * query).
     * </p>
     * @return bool true if the query succeeded, false on failure.
     */
    public function execWithTrace ($query, $level = SMTLogger::INFO ) 
    {
        SMTLogger::getInstance()->trace( sprintf("Execute query: %s.", $query ), $level );
        return $this->exec($query);
    }
    
    /**
     * Executes an SQL query, logs the query and returns a resultset.
     * 
     * @param query string <p>
     * The SQL query to execute.
     * </p>
     * @return SQLite3Result an SQLite3Result object if the query returns results. Otherwise,
     * returns true if the query succeeded, false on failure.
     */
    public function queryWithTrace ($query, $level = SMTLogger::INFO )
    {
    	SMTLogger::getInstance()->trace( sprintf("Execute query: %s.", $query ), $level );
    	return $this->query($query);
    }
    
    /**
     * Executes a query, logs the query and returns a single result
     * 
     * @param query string <p>
     * The SQL query to execute (typically an INSERT, UPDATE, or DELETE
     * query).
     * </p>
     * @return mixed the value of the first column of results or an array of the entire
	 * first row (if entire_row is true).
	 * 
	 * <p>
	 * If the query is valid but no results are returned, then &null; will be
	 * returned if entire_row is false, otherwise an
	 * empty array is returned.
	 * </p>
	 * 
	 * Invalid or failing queries will return false.
     */
    public function querySingleWithTrace ($query, $level = SMTLogger::INFO )
    {
    	SMTLogger::getInstance()->trace( sprintf("Execute query: %s.", $query ), $level );
    	return $this->querySingle($query);
    }
    
    /**
     * Execute the given operations in a transaction.
     * If a transaction is already running, join the transaction.
     * 
     * @param SMTIExecuteInTransaction $runner
     *
     * @throws SMTDatabaseException  in case of failure
     */
    public function runInTransaction( SMTIExecuteInTransaction $runner )
    {
        //join the current running transaction 
        if ( $this->transactionRunning )
        {
            $runner->run( $this );
        }
        else
        {
            try
            {
        		//begin transaction
        		$this->beginTransaction( $runner );
            
            	try
            	{
            	    $runner->run( $this );
            		
        			//end transaction
        			$this->commitTransaction( $runner );
            	}
            	catch(\Exception $e)
            	{
        			//rollback transaction
        			$success = $this->rollbackTransaction( $runner );
    
            		SMTLogger::getInstance()->traceException($e);
            		throw $e;
            	}
            }
            catch( \Exception $e )
            {
            	SMTLogger::getInstance()->trace( sprintf("Save dto %s of id %s failed", $runner->getResultSetDto() != NULL? $runner->getResultSetDto()->getDtoClassName() : get_class( $runner ), $runner->getResultSetDto() != NULL? $runner->getResultSetDto()->getId() : -1), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            	throw $e;
            }
        }        
    }
    
    /**
     * Open a transaction if none is already started
     * 
     * @param SMTIExecuteInTransaction $runner
     * @param int $retryCount optional: 0 by default. Retry counter to handle retry if database is locked
     * 
     * @throws SMTDatabaseException  in case of failure
     */
    public function beginTransaction( SMTIExecuteInTransaction $runner, $retryCount = 0 )
    {
        if ( !$this->transactionRunning )
        {
            SMTLogger::getInstance()->trace( sprintf("Begin transaction for dto %s of id %s", $runner->getResultSetDto() != NULL? $runner->getResultSetDto()->getDtoClassName() : get_class( $runner ), $runner->getResultSetDto() != NULL? $runner->getResultSetDto()->getId() : -1), SMTLogger::DEBUG );
            $success = $this->exec("BEGIN");
            
            if ( $success === FALSE )
            {
                if ( $retryCount < 3 && ( $this->lastErrorCode() == self::SQLITE_BUSY || $this->lastErrorCode() == self::SQLITE_LOCKED ) )
                {
                    //retry after sleeping 
                    usleep( self::SLEEP_LOCKED_uS );
                    $success = $this->beginTransaction($runner, $retryCount++);
                }
                if ( $success === FALSE )
                {
                    $className = get_class( $runner );
                    $id =-1;
                    if ( $runner->getResultSetDto() != NULL )
                    {
                    	$className = $runner->getResultSetDto()->getDtoClassName();
                    	$id = $runner->getResultSetDto()->getId();
                    }
                    
               	    SMTLogger::getInstance()->trace( sprintf("Failed to open transaction for dto %s of id %s ", $className, $id), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
                	throw new SMTDatabaseException(SMTDatabaseException::OPEN_TRANSACTION_FAILURE);
                }
            }
            else
            {
                $this->transactionRunning = TRUE;
            }
        }
        return $success;
    }
    
    /**
     * Commit a transaction if a transaction is opened
     *
     * @param SMTIExecuteInTransaction $runner
     * @param int $retryCount optional: 0 by default. Retry counter to handle retry if database is locked
     *
     * @throws SMTDatabaseException in case of failure
     */
    public function commitTransaction( SMTIExecuteInTransaction $runner, $retryCount = 0 )
    {
        if ( $this->transactionRunning )
        {            
            SMTLogger::getInstance()->trace( sprintf("Commit transaction for dto %s of id %s", $runner->getResultSetDto() != NULL? $runner->getResultSetDto()->getDtoClassName() : get_class( $runner ), $runner->getResultSetDto() != NULL? $runner->getResultSetDto()->getId() : -1), SMTLogger::DEBUG );
        	$success = $this->exec("COMMIT");
        	if ( $success === FALSE )
        	{
                if ( $retryCount < 3 && ( $this->lastErrorCode() == self::SQLITE_BUSY || $this->lastErrorCode() == self::SQLITE_LOCKED ) )
                {
                    //retry
                    usleep( self::SLEEP_LOCKED_uS );
                    $success = $this->commitTransaction($runner, $retryCount++);
                }
                if ( $success === FALSE )
                {
            	    $className = get_class( $runner );
            	    $id =-1;
            	    if ( $runner->getResultSetDto() != NULL )
            	    {
            	    	$className = $runner->getResultSetDto()->getDtoClassName();
            	    	$id = $runner->getResultSetDto()->getId();
            	    }
            		SMTLogger::getInstance()->trace( sprintf("Failed to commit and close transaction for dto %s of id %s. SQL Error: %s ", $className, $id, $this->lastErrorMsg() ), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            		throw new SMTDatabaseException(SMTDatabaseException::COMMIT_TRANSACTION_FAILURE);
                }
        	}
        	else
        	{
        	    $this->transactionRunning = FALSE;
        	}
        }
        return $success;
    }    
    
    /**
     * Rollback a transaction if a transaction is opened
     *
     * @param SMTIExecuteInTransaction $runner
     * @param int $retryCount optional: 0 by default. Retry counter to handle retry if database is locked
     *
     * @throws SMTDatabaseException in case of failure
     */
    public function rollbackTransaction( SMTIExecuteInTransaction $runner, $retryCount = 0 )
    {
        if ( $this->transactionRunning )
        {        
            SMTLogger::getInstance()->trace( sprintf("Rollback transaction for dto %s of id %s", $className, $id), SMTLogger::DEBUG );
        	$success = $this->exec("ROLLBACK");
        
        	if ( $success === FALSE )
        	{
        	    if ( $retryCount < 3 && ( $this->lastErrorCode() == self::SQLITE_BUSY || $this->lastErrorCode() == self::SQLITE_LOCKED ) )
        	    {
        	    	//retry
        	    	usleep( self::SLEEP_LOCKED_uS );
        	    	$success = $this->rollbackTransaction($runner, $retryCount++);
        	    }
        	    if ( $success === FALSE )
        	    {        	        
            	    $className = get_class( $runner );
            	    $id =-1;
            	    if ( $runner->getResultSetDto() != NULL )
            	    {
            	        $className = $runner->getResultSetDto()->getDtoClassName();
            	        $id = $runner->getResultSetDto()->getId();
            	    }
            		SMTLogger::getInstance()->trace( sprintf("Failed to rollback transaction for dto %s of id %s. SQL Error: %s ", $className, $id, $this->lastErrorMsg()), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            		throw new SMTDatabaseException(SMTDatabaseException::ROLLBACK_TRANSACTION_FAILURE);
        	    }
        	}
        	else
        	{
        	    $this->transactionRunning = FALSE;
        	}
        }
        return $success;
    }        
    
    /**
     * Test if main database table exists
     * 
     * @return boolean
     */
    function doesDatabasePatchTableExist()
    {        
        $result = $this->querySingleWithTrace('select patch_number, db_version from database_patch');
        $exists = ( $result != NULL && $result !== FALSE );
        SMTLogger::getInstance()->trace(sprintf("Database patch exists: %b", $exists), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        return $exists;
    }
    
    /**
     * Check the database and patch version regarding the patches embedded in php.
     * If needed, apply the required patches.
     *  
     */
    function checkDatabaseVersion()
    {
        SMTLogger::getInstance()->trace("Check database version: check that all patches have been applied", SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        
        $this->currentDatabaseVersion = $this->querySingleWithTrace('select max(db_version) from database_patch');
        
        if ( $this->currentDatabaseVersion !== FALSE && is_numeric( $this->currentDatabaseVersion ) )
        { 
            $this->currentDatabasePatch = $this->querySingleWithTrace("select max(patch_number) from database_patch where db_version = $this->currentDatabaseVersion");        
            
            if ( $this->currentDatabasePatch !== FALSE && is_numeric( $this->currentDatabasePatch ) )
            {
                SMTLogger::getInstance()->trace( sprintf( "Current database version %s, current database patch version: %s.", $this->currentDatabaseVersion, $this->currentDatabasePatch ) , SMTLogger::INFO );
                
                $this->retrieveLastPatchVersion();
                
                if ( $this->currentDatabaseVersion == $this->targetDatabaseVersion )
                {
                    if ( $this->targetDatabasePatch > $this->currentDatabasePatch )
                    {
                        //apply patches
                        for ( $patchNumber = $this->currentDatabasePatch +1;  $patchNumber <= $this->targetDatabasePatch; $patchNumber++ )
                        {
                            $className = "app\database\patch\SMTSql%02sp%04s";
                            $className = sprintf( $className, $this->targetDatabaseVersion, $patchNumber );
                            $this->applyPatch( $className );
                        }                
                    }
                    else if ( $this->targetDatabasePatch < $this->currentDatabasePatch)
                    {
                        SMTLogger::getInstance()->trace("WARNING: Current database patch:".$this->currentDatabaseVersion ." is higher than. target database patch version: ".$this->targetDatabasePatch, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
                    }
                    else
                    {
                        //nothing to do.
                    }
                }
                //upgrade needed
                else 
                {
                    SMTLogger::getInstance()->trace("DATABASE UPGRADE NEEDED. current database version: ".$this->currentDatabaseVersion." Target database version: ".$this->targetDatabaseVersion, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
                }
            }
            else
            {
                SMTLogger::getInstance()->trace("Couldn't retrieve current database patch number", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
                throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, "Couldn't retrieve current database patch number in database_patch");
            }
        }
        else
        {
            SMTLogger::getInstance()->trace("Couldn't retrieve current database version", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, "Couldn't retrieve current database version in database_patch");
        }
        
    }
    
    /**
     * Generate new dto database record identifier (moke sequence generation not available in SqLite)
     * 
     * @param SMTResultSetDto $dto
     * 
     * @return the new dto database identifier
     */
    public function generateId( SMTResultSetDto $dto )
    {
        $newId = NULL;
        if ( $dto instanceof SMTAlarmDto )
        {
            self::acquireAlarmSequenceDatabaseLock();
            $newId = $dto->generateId( $this );
            SMTLogger::getInstance()->trace( sprintf("New record id %s for dto %s. ", $newId, $dto->getDtoClassName() ) ,SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
            self::releaseAlarmSequenceDatabaseLock();
        }
        else if ( $dto instanceof SMTAlarmEventDto )
        {
            self::acquireAlarmEventSequenceDatabaseLock();
            $newId = $dto->generateId( $this );
            SMTLogger::getInstance()->trace( sprintf("New record id %s for dto %s. ", $newId, $dto->getDtoClassName() ) ,SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
            self::releaseAlarmEventSequenceDatabaseLock();
        }
        else if ( $dto instanceof SMTSystemConfigurationDto)
        {
            $newId = $dto->generateId( $this );
            SMTLogger::getInstance()->trace( sprintf("New record id %s for dto %s. ", $newId, $dto->getDtoClassName() ) ,SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );            
        }
        else
        {
            SMTLogger::getInstance()->trace( sprintf("Invalid resultset dto: %s. DTO not mapped.", $dto->getDtoClassName() ), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            throw new SMTDatabaseException( SMTDatabaseException::DTO_NOT_MAPPED, $dto->getDtoClassName() );
        }
        return $newId;
    }    
    
    /**
     * Parse patch files and retrieve the last patch target version
     */
    private function retrieveLastPatchVersion()
    {
        //patch files in ascending order
        $patchFiles = scandir( dirname(__FILE__)."/patch/");
        $lastPatchFile = $patchFiles[ count( $patchFiles ) - 1 ];
        
        SMTLogger::getInstance()->trace( "Last patch file: ".$lastPatchFile , SMTLogger::INFO );
        
        //retrieve patch version
        $className = basename($lastPatchFile, '.php');
        preg_match_all('!\d+!', $className, $matches);

        $this->targetDatabaseVersion = intval( $matches[0][0] );
        $this->targetDatabasePatch = intval( $matches[0][1] );

        SMTLogger::getInstance()->trace( sprintf( "Target database version %s, target database patch version: %s.", $this->targetDatabaseVersion, $this->targetDatabasePatch ) , SMTLogger::INFO );
    }
    
    /**
     * Create database model on an empty database.
     * 
     */
    private function createModel()
    {
        SMTLogger::getInstance()->trace( "Create new Database model", SMTLogger::INFO );
        SMTSql01p0001::getInstance()->execute( $this );
        SMTSql01p0002::getInstance()->execute( $this );
        SMTSql01p0003::getInstance()->execute( $this );
        SMTSql01p0004::getInstance()->execute( $this );
        SMTSql01p0005::getInstance()->execute( $this );
        SMTSql01p0006::getInstance()->execute( $this );
    }
    
    /**
     * Apply the required database patch
     *
     */
    private function applyPatch( $className )
    {        
        SMTLogger::getInstance()->trace( sprintf("BEFORE apply patch %s. Current database patch: %s. Current database version: %s. ", $className, $this->currentDatabasePatch, $this->currentDatabaseVersion ), SMTLogger::INFO );

        $instance = new $className();
        $instance->execute( $this );
        
        //update current database patch version
        $this->currentDatabasePatch = $this->querySingleWithTrace("select max(patch_number) from database_patch where db_version = $this->currentDatabaseVersion");
        
        SMTLogger::getInstance()->trace( sprintf("AFTER apply patch %s. Current database patch: %s. Current database version: %s. ", $className, $this->currentDatabasePatch, $this->currentDatabaseVersion ), SMTLogger::INFO );
    }

    /**
     * Script to enable or disable foreign key constraints
     *
     * @param bool $enable Whether foreign key constraints must be enable or disabled
     *
     * @return string Sql script to enable or disable foreign key constraints
     */
    public static function getEnableForeignKey( $enable )
    {
    	return sprintf(self::SQL_ENABLE_FOREIGN_KEYS, ($enable)? "ON" : "OFF");
    }    
    
    /**
     * Lock file handle
     * 
     * @return string
     */
    private static function getCheckDatabaseLockFile()
    {
    	if ( self::$checkDatabaseLockFile == NULL )
    	{
    		self::$checkDatabaseLockFile = fopen( __DIR__.self::LOCK_CHECK_DATABASE, "w+");
    	}
    	return self::$checkDatabaseLockFile;
    }
    
    /**
     * Lock file handling: acquire a lock
     * 
     * @return boolean
     */
    private static function acquireCheckDatabaseLock()
    {
    	$retries = 0;
    	$max_retries = 55;
    	 
    	SMTLogger::getInstance()->trace( "acquireCheckDatabaseLock",SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    
    	// keep trying to get a lock as long as possible ( max 56s) ( n(n+1)(2n+1)/6 )
    	//we use x² suite to sleep longer and longer because process should be very short except after reboot
    	do
    	{
    		if ($retries > 0)
    		{
    			usleep( 1000 * ( $retries*$retries ) );
    		}
    		$retries += 1;
    	} while ( (!flock(self::getCheckDatabaseLockFile(), LOCK_EX, $eWouldBlock) || $eWouldBlock ) && $retries <= $max_retries);
    
    	return ($retries < $max_retries);
    }
    
    /**
     * release the lock 
     * 
     * @return boolean
     */
    private static function releaseCheckDatabaseLock()
    {
    	SMTLogger::getInstance()->trace( "releaseCheckDatabaseLock",SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    	return isset(self::$checkDatabaseLockFile) && self::$checkDatabaseLockFile != NULL? flock( self::$checkDatabaseLockFile, LOCK_UN ) : TRUE;
    }

    /**
     * Lock file handle
     *
     * @return string
     */
    private static function getAlarmSequenceDatabaseLockFile()
    {
    	if ( self::$alarmSequenceDatabaseLockFile == NULL )
    	{
    		self::$alarmSequenceDatabaseLockFile = fopen( __DIR__.self::LOCK_ALARM_SEQUENCE_DATABASE, "w+");
    	}
    	return self::$alarmSequenceDatabaseLockFile;
    }
    
    /**
     * Lock file handling: acquire a lock
     *
     * @return boolean
     */
    private static function acquireAlarmSequenceDatabaseLock()
    {
    	$retries = 0;
    	$max_retries = 25;
    
    	SMTLogger::getInstance()->trace( "acquireAlarmSequenceDatabaseLock",SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    
    	// keep trying to get a lock as long as possible ( max 5s)
    	do
    	{
    		if ($retries > 0)
    		{
    			usleep( 1000 * ( $retries*$retries ) );
    		}
    		$retries += 1;
    	} while ( (!flock(self::getAlarmSequenceDatabaseLockFile(), LOCK_EX, $eWouldBlock) || $eWouldBlock ) && $retries <= $max_retries);
    
    	return ($retries < $max_retries);
    }
    
    /**
     * release the lock
     *
     * @return boolean
     */
    private static function releaseAlarmSequenceDatabaseLock()
    {
    	SMTLogger::getInstance()->trace( "releaseAlarmSequenceLock",SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    	return isset(self::$alarmSequenceDatabaseLockFile) && self::$alarmSequenceDatabaseLockFile != NULL? flock( self::$alarmSequenceDatabaseLockFile, LOCK_UN ) : TRUE;
    }    
    
    /**
     * Lock file handle
     *
     * @return string
     */
    private static function getAlarmEventSequenceDatabaseLockFile()
    {
	    if ( self::$alarmEventSequenceDatabaseLockFile == NULL )
	    {
	    	$handle = fopen( __DIR__.self::LOCK_ALARM_EVENT_SEQUENCE_DATABASE, "w+");
	    
	    	self::$alarmEventSequenceDatabaseLockFile = ( $handle != FALSE)? $handle : NULL;
	    }    	    
    	return self::$alarmEventSequenceDatabaseLockFile;
    }
    
    /**
     * Lock file handling: acquire a lock
     *
     * @return boolean
     */
    private static function acquireAlarmEventSequenceDatabaseLock()
    {
    	$retries = 0;
    	$max_retries = 25;
    
    	SMTLogger::getInstance()->trace( "acquireAlarmEventSequenceDatabaseLock",SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    
    	// keep trying to get a lock as long as possible ( max 5s)
    	do
    	{
    		if ($retries > 0)
    		{
    			usleep( 1000 * ( $retries*$retries ) );
    		}
    		$retries += 1;
    	} while ( (!flock(self::getAlarmEventSequenceDatabaseLockFile(), LOCK_EX, $eWouldBlock) || $eWouldBlock ) && $retries <= $max_retries);
    
    	return ($retries < $max_retries);
    }
    
    /**
     * release the lock
     *
     * @return boolean
     */
    private static function releaseAlarmEventSequenceDatabaseLock()
    {
    	SMTLogger::getInstance()->trace( "releaseAlarmEventSequenceLock",SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    	return isset(self::$alarmEventSequenceDatabaseLockFile) && self::$alarmEventSequenceDatabaseLockFile != NULL? flock( self::$alarmEventSequenceDatabaseLockFile, LOCK_UN ) : TRUE;
    }    
}

?>